/* ***************************************************************************************
* FILE:          Animator.h
* SW-COMPONENT:  HMI-BASE
* DESCRIPTION:  Animator.h is part of HMI-Base ScreenBrokerPlugins
* COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(ScreenBrokerPlugins_Shared_Animation_Animator_h)
#define ScreenBrokerPlugins_Shared_Animation_Animator_h

#include <iostream>
#include <map>
#include <list>
#include <ScreenBroker/ScreenBroker.h>
#include <Shared/Animation/AnimatorTimer.h>
namespace ScreenBroker {
/** Forward declaration of ComplexAnimation Class */
class ComplexAnimation;

/** Forward declaration of AnimatorTimer Class */
class AnimatorTimer;

/** AnimationDataMap is shorthand to use Map of Animation Object File Descriptor ID */
typedef std::map<ComplexAnimation*, Int32> AnimationDataMap;

/** EvenDelayMap is shorthand to use Map of File Descriptor ID Animation Object */
typedef std::map<Int32, std::list<ComplexAnimation*> > EventDelayMap;

/**
 *Animator is a worker class to execute Animation timer controls
 *requested by complex animation.Animator class holds indvidual timer
 *for each animation interval which makes it simple for identification.
 *Derived from IAnimatorTimer to use Timer Interface for Updates
 */
class Animator
{
   public:

      virtual ~Animator();

      /**
       *<Method to get the unique inctance of Animator Class>
       *@param[out] <Animator*> <Animator class instance object>
       */
      static Animator* getInstance()
      {
         if (0 == _instanceSelf)
         {
            _instanceSelf = new Animator();
         }
         return _instanceSelf;
      }

      /**
       *<Method to register an animation timer for respective animation in AnimationDatamap>
       *@param[in] <ComplexAnimation*> <ComplexAnimation class instance object to update>
       */
      bool startAnimation(ComplexAnimation* lCompexAnimation);

      /**
       *<Method to register an animation timer for respective animation in AnimationDatamap>
       *@param[in] <ComplexAnimation*> <ComplexAnimation class instance object to update>
       */
      void InitiateGroupAnimations(std::list<ComplexAnimation*> lCompexAnimationlist);
      /**
       *<Method to register gorup with single animation timer for respective animation in AnimationDatamap>
       *@param[in] <ComplexAnimation*> <ComplexAnimation class instance object to update>
       */
      void startBatchAnimation(std::list<ComplexAnimation*> lCompexAnimationlist);

      /**
       *<Method to deregister an animation timer from AnimationDatamap>
       *@param[in] <ComplexAnimation*> <ComplexAnimation class instance object to update>
       */
      void stopAnimation(ComplexAnimation* lCompexAnimation);

      /**
       *<Method to Stop all animation timer in AnimationDatamap>
       */
      void abortAllAnimations();

      /**
       *<Below Method Name should not be modified without update>
       *<Call Back Method to get Expired Timer Updated from Timer class>
       *@param[in] <timerFD> <TimerFD id of the expired timer>
       *@param[in] <missed> <No of missed count from last update>
       */
      void expiredTimerUpdate(Int32 timerFD, UInt32 missed);

   private:
      /**Static self instance object reference holder*/
      static Animator* _instanceSelf;

      /**Method to Create and set Callback for Timer*/
      void initializeTimer();

      /**Attribute to sotre AnimationDataMap*/
      AnimationDataMap _AnimationDataMap;

      /**Attribute to sotre Delayed EvenDelayMap*/
      EventDelayMap _EventDelayMap;
      /**
       *<Constructor made private for unique instance>
       */
      Animator();
};


} //namespace ScreenBroker
#endif /* ScreenBrokerPlugins_Shared_Animation_Animator_h */
