/* ***************************************************************************************
* FILE:          Animation.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Animation.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_Shared_Animation_Animation_h)
#define ScreenBrokerPlugins_Shared_Animation_Animation_h

#include <ScreenBroker/Types.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <Shared/Animation/AnimationBase.h>
#include <Shared/Animation/AnimationProperties.h>
#include <Shared/Animation/ComplexAnimation.h>
#include <map>

namespace ScreenBroker {

///
template <typename TAnimatedProperty>
class AnimationSurfaceProperties : public AnimationSurfacePropertiesBase
{
      typedef AnimationSurfacePropertiesBase Base;
   public:
      TAnimatedProperty originalPropertyValue;
      TAnimatedProperty finalPropertyValue;
      TAnimatedProperty delta;

      AnimationSurfaceProperties() :
         Base(),
         originalPropertyValue(),
         finalPropertyValue(),
         delta()
      {}

      AnimationSurfaceProperties(
         UInt32 surfaceId,
         UInt32 layerId,
         AnimationDirection::Enum animationDirection,
         AnimationUserData animationUserData,
         const TAnimatedProperty& originalPropertyValue,
         const TAnimatedProperty& finalPropertyValue) :
         Base(surfaceId, layerId, animationDirection, animationUserData),
         originalPropertyValue(originalPropertyValue),
         finalPropertyValue(finalPropertyValue),
         delta()
      {}
};


///
template <typename TSurfaceProperties>
class Animation : public ComplexAnimation
{
      typedef ComplexAnimation Base;

   public:

      typedef std::map<UInt32, TSurfaceProperties> SurfacePropertiesMap;

      Animation() : Base(),
         mSurfacePropertiesMap()
      { }

      void AddSurfaceProperties(const TSurfaceProperties& surfaceProperties);
      virtual UInt32 GetSurfaceId();

      virtual void Reset();

   protected:
      virtual bool InitializeSurfaceStates(bool commit);
      virtual bool UpdateSurfaceStates(UInt32 count, bool commit);
      virtual bool SetFinalSurfaceStates(bool commit);
      virtual void NotifyAnimationsStart();
      virtual void NotifyAnimationsFinished(bool triggerNext);

      virtual bool InitializeSurfaceState(TSurfaceProperties& surfaceProperties, bool setCommit = true) = 0;
      virtual bool UpdateSurfaceState(const TSurfaceProperties&, UInt32 count, bool commit = true) = 0;
      virtual bool SetFinalSurfaceState(const TSurfaceProperties& surfaceProperties, bool commit = true) = 0;

   private:
      SurfacePropertiesMap mSurfacePropertiesMap;
      CriticalSection mCs;
};


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
void Animation<TSurfaceProperties>::Reset()
{
   CriticalSectionLocker lLock(&mCs);
   Base::Reset();
   mSurfacePropertiesMap.clear();
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
bool Animation<TSurfaceProperties>::InitializeSurfaceStates(bool commit)
{
   CriticalSectionLocker lLock(&mCs);
   bool lRc = true;
   for (typename SurfacePropertiesMap::iterator lIt = mSurfacePropertiesMap.begin(); mSurfacePropertiesMap.end() != lIt; ++lIt)
   {
      TSurfaceProperties& surfaceProperties = lIt->second;
      if (!InitializeSurfaceState(surfaceProperties, commit))
      {
         lRc = false;
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
bool Animation<TSurfaceProperties>::UpdateSurfaceStates(UInt32 count, bool commit)
{
   CriticalSectionLocker lLock(&mCs);
   bool lRc = true;
   for (typename SurfacePropertiesMap::const_iterator lIt = mSurfacePropertiesMap.begin(); mSurfacePropertiesMap.end() != lIt; ++lIt)
   {
      const TSurfaceProperties& surfaceProperties = lIt->second;
      if (!UpdateSurfaceState(surfaceProperties, count, commit))
      {
         lRc = false;
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
bool Animation<TSurfaceProperties>::SetFinalSurfaceStates(bool commit)
{
   CriticalSectionLocker lLock(&mCs);
   bool lRc = true;
   for (typename SurfacePropertiesMap::const_iterator lIt = mSurfacePropertiesMap.begin(); mSurfacePropertiesMap.end() != lIt; ++lIt)
   {
      const TSurfaceProperties& surfaceProperties = lIt->second;
      if (!SetFinalSurfaceState(surfaceProperties, commit))
      {
         lRc = false;
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
void Animation<TSurfaceProperties>::NotifyAnimationsStart()
{
   BeforeAnimationCallback lCallback = GetBeforeAnimationCallback();
   if (lCallback == 0)
   {
      return;
   }

   CriticalSectionLocker lLock(&mCs);
   for (typename SurfacePropertiesMap::const_iterator lIt = mSurfacePropertiesMap.begin(); mSurfacePropertiesMap.end() != lIt; ++lIt)
   {
      const TSurfaceProperties& surfaceProperties = lIt->second;
      lCallback(surfaceProperties);
   }
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
void Animation<TSurfaceProperties>::NotifyAnimationsFinished(bool triggerNext)
{
   AfterAnimationCallback lCallback = GetAfterAnimationCallback();
   if (lCallback == 0)
   {
      return;
   }

   CriticalSectionLocker lLock(&mCs);
   for (typename SurfacePropertiesMap::const_iterator lIt = mSurfacePropertiesMap.begin(); mSurfacePropertiesMap.end() != lIt; ++lIt)
   {
      const TSurfaceProperties& surfaceProperties = lIt->second;
      lCallback(surfaceProperties, triggerNext);
   }
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
void Animation<TSurfaceProperties>::AddSurfaceProperties(const TSurfaceProperties& surfaceProperties)
{
   CriticalSectionLocker lLock(&mCs);
   if (surfaceProperties.layerId != 0)
   {
      mSurfacePropertiesMap[surfaceProperties.layerId] = surfaceProperties;
   }
   else
   {
      mSurfacePropertiesMap[surfaceProperties.surfaceId] = surfaceProperties;
   }
}


// ------------------------------------------------------------------------
template <typename TSurfaceProperties>
UInt32 Animation<TSurfaceProperties>::GetSurfaceId()
{
   if (1 == mSurfacePropertiesMap.size())
   {
      typename SurfacePropertiesMap::const_iterator lIt = mSurfacePropertiesMap.begin();
      if (mSurfacePropertiesMap.end() != lIt)
      {
         return lIt->first;
      }
   }
   return 0;
}


}

#endif
