/* ***************************************************************************************
* FILE:          ActiveSurfaces.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ActiveSurfaces.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "ActiveSurfaces.h"
#include "IlmAccessor.h"

#include "PluginActions.h"
#include <Shared/FocusHandler.h>
#include <ScreenBrokerSettings/Settings.h>
#include <algorithm>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/ActiveSurfaces.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::States);

// ------------------------------------------------------------------------
// (Plugin) global handled ActiveSurfaces object has to be instantiated once for all plugins
// This is normally done in ScreenBrokerActivator.cpp.
extern ActiveSurfaces gActiveSurfacesInstance;

// ------------------------------------------------------------------------
ActiveSurfaces& ActiveSurfaces::GetInstance()
{
   return gActiveSurfacesInstance;
}


// ------------------------------------------------------------------------
ActiveSurfaces::ActiveSurfaces()
{
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::Remove(UInt32 surfaceId)
{
   UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(surfaceId);
   ActiveSurfaceList* lActiveSurfaceList = &mActiveSurfaceMap[lDisplayId];

   // Retrieve new focused, if given surfaceId has currently focus and is removed.
   if (!lActiveSurfaceList->empty())
   {
      lActiveSurfaceList->remove(surfaceId);
      ETG_TRACE_USR4((" remove %u", surfaceId));
   }

   FocusHandler::GetInstance().RemoveSurface(surfaceId);

   UInt32 lNewSurfaceId = GetTop(lDisplayId);
   ETG_TRACE_USR1(("Remove: %u (top: %u)", surfaceId, lNewSurfaceId));
   mSurfacePropertiesMap.erase(surfaceId);
   return lNewSurfaceId;
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::PutOnTop(UInt32 surfaceId, UInt32 userData, UInt32 appId)
{
   UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(surfaceId);
   ActiveSurfaceList* lActiveSurfaceList = &mActiveSurfaceMap[lDisplayId];
   UInt32 lPrevSurfaceId = GetTop(lDisplayId);
   if (!ProjectSettings::IsPopup(surfaceId))
   {
      ETG_TRACE_USR1(("PutOnTop: app(%u) %u/0x%08X [prev: app(%u) %u/0x%08X]",
                      appId,
                      surfaceId,
                      userData,
                      GetAppId(lPrevSurfaceId),
                      lPrevSurfaceId,
                      GetUserData(lPrevSurfaceId)));

      // Remove existing surface ID for a latter re-order (via push back)
      if (Exists(surfaceId))
      {
         lActiveSurfaceList->remove(surfaceId);
         ETG_TRACE_USR4((" remove %u", surfaceId));

         FocusHandler::GetInstance().RemoveSurface(surfaceId);
      }
      lActiveSurfaceList->push_back(surfaceId);
      ETG_TRACE_USR4((" push %u", surfaceId));

      // Update userData, which may have changed even for the same surface
      mSurfacePropertiesMap[surfaceId] = Properties(userData, appId);

      FocusHandler::GetInstance().PushBackSurface(surfaceId);
   }
   else
   {
      ETG_TRACE_SYS(("PutOnTop failed: %u is a popup surface", surfaceId));
   }

   return lPrevSurfaceId;
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::GetTop(UInt32 displayId, bool ignorePermanentSurfaces) const
{
   UInt32 lSurfaceId = 0;
   ActiveSurfaceMap::const_iterator it = mActiveSurfaceMap.find(displayId);
   if (it != mActiveSurfaceMap.end())
   {
      ActiveSurfaceList lActiveSurfaceList = it->second;
      if (ignorePermanentSurfaces)
      {
         ActiveSurfaceList::const_reverse_iterator rit = lActiveSurfaceList.rbegin();
         while ((lActiveSurfaceList.rend() != rit) && ProjectSettings::IsPermanent(*rit))
         {
            ++rit;
         }
         if ((lActiveSurfaceList.rend() != rit))
         {
            lSurfaceId = *rit;
         }
      }
      else
      {
         if (!lActiveSurfaceList.empty())
         {
            lSurfaceId = lActiveSurfaceList.back();
         }
      }
   }
   return lSurfaceId;
}


// ------------------------------------------------------------------------
bool ActiveSurfaces::Exists(UInt32 surfaceId) const
{
   bool lExists = false;
   ActiveSurfaceMap::const_iterator itMap = mActiveSurfaceMap.begin();
   while (!lExists &&  mActiveSurfaceMap.end() != itMap)
   {
      ActiveSurfaceList lActiveSurfaceList = itMap->second;
      ActiveSurfaceList::iterator itEnd = lActiveSurfaceList.end();
      lExists = (itEnd != std::find(lActiveSurfaceList.begin(),
                                    itEnd,
                                    surfaceId));
      ++itMap;
   }
   return lExists;
}


// ------------------------------------------------------------------------
const ActiveSurfaces::Properties* ActiveSurfaces::GetProperties(UInt32 surfaceId) const
{
   const Properties* lProperties = 0;
   SurfacePropertiesMap::const_iterator it = mSurfacePropertiesMap.find(surfaceId);
   if (mSurfacePropertiesMap.end() != it)
   {
      lProperties = &((*it).second);
   }
   return lProperties;
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::GetUserData(UInt32 surfaceId) const
{
   const Properties* lProperties = GetProperties(surfaceId);
   return ((0 != lProperties) ? lProperties->UserData() : 0);
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::GetAppId(UInt32 surfaceId) const
{
   const Properties* lProperties = GetProperties(surfaceId);
   return ((0 != lProperties) ? lProperties->AppId() : 0);
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::GetFirst(UInt32 displayId, bool ignorePermanentSurfaces)
{
   UInt32 lSurfaceId = 0;
   ActiveSurfaceMap::iterator it = mActiveSurfaceMap.find(displayId);
   if (it != mActiveSurfaceMap.end())
   {
      ActiveSurfaceList* lActiveSurfaceList = &(it->second);

      // Get first surface
      mIt = lActiveSurfaceList->begin();
      // Ignore permanent surfaces, if option is set
      if (ignorePermanentSurfaces)
      {
         while ((lActiveSurfaceList->end() != mIt) &&
                ProjectSettings::IsPermanent(*mIt))
         {
            ++mIt;
         }
      }
      lSurfaceId = (lActiveSurfaceList->end() != mIt) ? (*mIt) : 0;
   }
   return lSurfaceId;
}


// ------------------------------------------------------------------------
UInt32 ActiveSurfaces::GetNext(UInt32 displayId, bool ignorePermanentSurfaces)
{
   UInt32 lSurfaceId = 0;
   ActiveSurfaceMap::iterator it = mActiveSurfaceMap.find(displayId);
   if (it != mActiveSurfaceMap.end())
   {
      ActiveSurfaceList* lActiveSurfaceList = &(it->second);

      // Get next surface
      if (lActiveSurfaceList->end() != mIt)
      {
         ++mIt;
      }
      // Ignore permanent surfaces, if option is set
      if (ignorePermanentSurfaces)
      {
         while ((lActiveSurfaceList->end() != mIt) &&
                ProjectSettings::IsPermanent(*mIt))
         {
            ++mIt;
         }
      }
      lSurfaceId = (lActiveSurfaceList->end() != mIt) ? (*mIt) : 0;
   }
   return lSurfaceId;
}


// ------------------------------------------------------------------------
void ActiveSurfaces::Dump()
{
   Int i = 0;
   for (ActiveSurfaceMap::const_iterator itMap = mActiveSurfaceMap.begin(); mActiveSurfaceMap.end() != itMap; ++itMap)
   {
      ActiveSurfaceList lActiveSurfaceList = itMap->second;
      ETG_TRACE_USR1(("&&& Dump active surfaces of display %u (count: %d):", itMap->first, lActiveSurfaceList.size()));
      for (ActiveSurfaceList::const_iterator itList = lActiveSurfaceList.begin(); lActiveSurfaceList.end() != itList; ++itList)
      {
         ++i;
         UInt32 lSurfaceId = *itList;
         ETG_TRACE_USR1(("&   %d. surface: %u/0x%08X (appId: %d, %40spermanent)",
                         i,
                         lSurfaceId,
                         mSurfacePropertiesMap[lSurfaceId].UserData(),
                         mSurfacePropertiesMap[lSurfaceId].AppId(),
                         ProjectSettings::IsPermanent(lSurfaceId) ? "" : "non-"));
      }

      ETG_TRACE_USR1(("&   Top overall application (incl. permanent surface): %d", GetAppId(GetTop(itMap->first, false))));
      ETG_TRACE_USR1(("&   Top application (non-permanent surface only): %d", GetAppId(GetTop(itMap->first))));
   }
}


// ------------------------------------------------------------------------
void ActiveSurfaces::Reset()
{
   mActiveSurfaceMap.clear();
   mSurfacePropertiesMap.clear();

   FocusHandler::GetInstance().RemoveSurfaces();
}


}
