/* ***************************************************************************************
* FILE:          ActivePopups.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ActivePopups.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Use of lExistingPopup null pointer in ScreenBroker::ActivePopups::Insert is
// prevented by several if conditions
//lint -efunc(613, ScreenBroker::ActivePopups::Insert)   "Possible use of null pointer"
// Default constructor is not required for this test class
//lint -esym(1712, TestPopupState) "Default constructor not defined for class"
// Using the "do { ... } while(0) construct within macros is a common practice in order
// to make the use of macros safe within conditions.
//lint -emacro({717}, POPUP_LOG)    "do ... while(0);"
// =============================================================================

//#define SCREENBROKER_PRINTF_LOG 1

#include "ActivePopups.h"
#include "ActiveSurfaces.h"
#include "IlmAccessor.h"

#include "PluginActions.h"
#include <ScreenBroker/Service/PopupManager/PopupState.h>
#include <Shared/FocusHandler.h>
#include <ScreenBrokerSettings/Settings.h>
#include <algorithm>
#include <stdio.h>

// Macro for easy popup state logging
#define POPUP_LOG(level, popup, info, ...)                              \
    do {                                                                \
        if (0 != popup) {                                               \
            sprintf(str, info  "app(%u) - "                  \
                "%u/0x%08X - prio(%u), order(%u), "                     \
                "modality(%s), visible(%s)",                            \
                ## __VA_ARGS__,                                         \
                (popup)->GetAppId(),                                    \
                (popup)->GetSurfaceId(),                                \
                (popup)->GetUserData(),                                 \
                (popup)->GetPriority(),                                 \
                GetPopupInsertionOrder(popup),                          \
                (((popup)->GetModality() == Modality::Application) ?    \
                    "app" :                                             \
                    (((popup)->GetModality() == Modality::System) ?     \
                        "sys" : "n/a")),                                \
                ((popup)->IsVisible() ? "yes" : "no"));                   \
        }                                                               \
    } while(0)
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/ActivePopups.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::States);

// ------------------------------------------------------------------------
// (Plugin) global handled ActivePopups object has to be instantiated once for all plugins
// This is normally done in ScreenBrokerActivator.cpp.
extern ActivePopups gActivePopupsInstance;

// ------------------------------------------------------------------------
ActivePopups& ActivePopups::GetInstance()
{
   return gActivePopupsInstance;
}


// ------------------------------------------------------------------------
ActivePopups::ActivePopups()
{
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::Insert(const PopupState& sourcePopupState,
                                 PopupState*& destinationPopupState)
{
   // Search for popup state or create a new instance of locale popup state, if not found
   PopupState* lExistingPopup = Find(sourcePopupState.GetUserData());

   // Remove an existing entry in any priority list in case the modality resp. the appId has changed
   bool lReinsertExisting = (0 != lExistingPopup) &&
                            ((lExistingPopup->GetModality() != sourcePopupState.GetModality()) ||
                             ((lExistingPopup->GetModality() == Modality::Application) && (lExistingPopup->GetAppId() != sourcePopupState.GetAppId())));
   if (lReinsertExisting)
   {
      switch (lExistingPopup->GetModality())
      {
         case Modality::Application:
         {
            mActiveApplicationPopupList.remove(lExistingPopup);
            break;
         }
         case Modality::System:
         {
            mSystemPopupList.remove(lExistingPopup);
            break;
         }
         default:
            // Cleanup an invalid/orphaned entry
            mPopupStateMap.erase(lExistingPopup->GetUserData());
            break;
      }

      FocusHandler::GetInstance().RemoveSurface(lExistingPopup->GetSurfaceId());
   }

   // Create/update popup container
   mPopupStateMap[sourcePopupState.GetUserData()] = sourcePopupState;
   destinationPopupState = &mPopupStateMap[sourcePopupState.GetUserData()];

   // Add new insertion order map entry if not yet exists
   if (0 == lExistingPopup)
   {
      UInt32 lOrderValue = 0;
      for (PopupInsertionOrderMap::const_iterator it = mPopupInsertionOrderMap.begin();
            mPopupInsertionOrderMap.end() != it;
            ++it)
      {
         lOrderValue = std::max(lOrderValue, (*it).second);
      }
      mPopupInsertionOrderMap[destinationPopupState] = ++lOrderValue;
   }

   // Insert entry in the respective priority list, if applicable
   if ((0 == lExistingPopup) || lReinsertExisting)
   {
      switch (destinationPopupState->GetModality())
      {
         case Modality::Application:
         {
            PopupList& lApplicationPopupList = mApplicationPopupMap[destinationPopupState->GetAppId()];
            lApplicationPopupList.insert(lApplicationPopupList.end(), destinationPopupState);
            lApplicationPopupList.sort(ComparePopupList);
            break;
         }
         case Modality::System:
         {
            mSystemPopupList.insert(mSystemPopupList.end(), destinationPopupState);
            mSystemPopupList.sort(ComparePopupList);
            break;
         }
         default:
            ETG_TRACE_ERR(("Insert popup %u/0x%08X failed: Invalid popup modality: %d",
                           destinationPopupState->GetSurfaceId(),
                           destinationPopupState->GetUserData(),
                           destinationPopupState->GetModality()));
            break;
      }

      // update focus list according to popup priorities
      FocusHandler::GetInstance().RemoveSurfaces(true);

      PopupState* lPopupState = GetOverallFirst();

      while (0 != lPopupState)
      {
         if ((destinationPopupState == lPopupState) || (lPopupState->IsVisible()))
         {
            FocusHandler::GetInstance().PushBackSurface(lPopupState->GetSurfaceId());
         }
         lPopupState = GetOverallNext();
      }
   }
   {
      char str[230];
      POPUP_LOG(INFO, destinationPopupState, "Inserted: ");
      ETG_TRACE_USR4(("%40s", str));
   }

   PopupState* lTopPopupState = GetOverallTop(false, destinationPopupState->GetPopupScreenAreaId());
   return lTopPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::Remove(PopupState* popupState)
{
   PopupState lRemovedPopupState;
   PopupState* lTopPopupState = 0;
   // Insert entry in the respective priority list
   if (0 != popupState)
   {
      lRemovedPopupState = *popupState;
      switch (popupState->GetModality())
      {
         case Modality::Application:
         {
            PopupList& lApplicationPopupList = mApplicationPopupMap[popupState->GetAppId()];
            lApplicationPopupList.remove(popupState);
            UpdateApplicationPopupList(popupState->GetLayerId());
            break;
         }
         case Modality::System:
         {
            mSystemPopupList.remove(popupState);
            break;
         }
         default:
            // Shall never happen!!!
            ETG_TRACE_SYS(("Remove popup %u/0x%08X in active popup lists failed: Invalid popup modality: %d",
                           popupState->GetSurfaceId(),
                           popupState->GetUserData(),
                           popupState->GetModality()));
            break;
      }
      // Erase instance of locally held popupState
      mPopupStateMap.erase(popupState->GetUserData());
      // Erase corresponding insertion order map entry
      mPopupInsertionOrderMap.erase(popupState);

      PopupState* lTmpPopupState = &lRemovedPopupState;
      {
         char str[230];
         POPUP_LOG(INFO, lTmpPopupState, "Removed: ");
         ETG_TRACE_USR4(("%40s", str));
      }

      lTopPopupState = GetOverallTop(false, lRemovedPopupState.GetPopupScreenAreaId());
      if (0 != lTopPopupState)
      {
         {
            char str[230];
            POPUP_LOG(DEBUG, lTopPopupState, "  Top: ");
            ETG_TRACE_USR4(("%40s", str));
         }
      }
   }

   return lTopPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetOverallTop(bool applyFilter,
                                        UInt32 screenArea)
{
   PopupState* lTopSysPopupState = GetSysTop(applyFilter, screenArea);
   PopupState* lTopAppPopupState = GetAppTop(applyFilter, screenArea);

   return Top(lTopSysPopupState, lTopAppPopupState);
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetSysTop(bool applyFilter,
                                    UInt32 screenArea) const
{
   PopupList::const_reverse_iterator lIt = mSystemPopupList.rbegin();
   while ((mSystemPopupList.rend() != lIt) &&
          (0 != *lIt) &&
          ((applyFilter && PluginActions::IsPopupFiltered(**lIt)) ||
           ((screenArea != (*lIt)->GetPopupScreenAreaId()) &&
            (0 != screenArea))))
   {
      ++lIt;
   }
   return (mSystemPopupList.rend() != lIt) ? *lIt : 0;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetAppTop(bool applyFilter,
                                    UInt32 screenArea)
{
   UpdateApplicationPopupList(screenArea);

   PopupList::const_reverse_iterator lIt = mActiveApplicationPopupList.rbegin();
   while ((mActiveApplicationPopupList.rend() != lIt) &&
          (0 != *lIt) &&
          ((applyFilter && PluginActions::IsPopupFiltered(**lIt)) ||
           ((screenArea != (*lIt)->GetPopupScreenAreaId()) &&
            (0 != screenArea))))
   {
      ++lIt;
   }

   return (mActiveApplicationPopupList.rend() != lIt) ? *lIt : 0;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetOverallFirst(UInt32 screenArea)
{
   UpdateOverallPopupList(screenArea);
   PopupState* lPopupState = 0;
   mOverallPopupListIt = mOverallPopupList.begin();
   while ((mOverallPopupList.end() != mOverallPopupListIt) && (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mOverallPopupListIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mOverallPopupListIt);
      }
      ++mOverallPopupListIt;
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetOverallNext(UInt32 screenArea)
{
   PopupState* lPopupState = 0;
   while ((mOverallPopupList.end() != mOverallPopupListIt) && (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mOverallPopupListIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mOverallPopupListIt);
      }
      ++mOverallPopupListIt;
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetOverallLast(UInt32 screenArea)
{
   UpdateOverallPopupList(screenArea);
   PopupState* lPopupState = 0;
   mOverallPopupListRIt = mOverallPopupList.rbegin();
   while ((mOverallPopupList.rend() != mOverallPopupListRIt) && (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mOverallPopupListRIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mOverallPopupListRIt);
      }
      ++mOverallPopupListRIt;
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetOverallPrevious(UInt32 screenArea)
{
   PopupState* lPopupState = 0;
   while ((mOverallPopupList.rend() != mOverallPopupListRIt) && (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mOverallPopupListRIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mOverallPopupListRIt);
      }
      ++mOverallPopupListRIt;
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetSysFirst(UInt32 screenArea)
{
   PopupState* lPopupState = 0;
   mSystemPopupListIt = mSystemPopupList.begin();
   while ((mSystemPopupList.end() != mSystemPopupListIt) &&
          (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mSystemPopupListIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mSystemPopupListIt);
      }
      ++mSystemPopupListIt;
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetSysNext(UInt32 screenArea)
{
   PopupState* lPopupState = 0;
   while ((mSystemPopupList.end() != mSystemPopupListIt) &&
          (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mSystemPopupListIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mSystemPopupListIt);
      }
      ++mSystemPopupListIt;
   }
   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetAppFirst(UInt32 screenArea, bool refreshActiveApplicationList)
{
   PopupState* lPopupState = 0;
   if (refreshActiveApplicationList)
   {
      UpdateApplicationPopupList(screenArea);
   }

   mActiveApplicationPopupListIt = mActiveApplicationPopupList.begin();
   while ((mActiveApplicationPopupList.end() != mActiveApplicationPopupListIt) &&
          (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mActiveApplicationPopupListIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mActiveApplicationPopupListIt);
      }
      ++mActiveApplicationPopupListIt;
   }

   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::GetAppNext(UInt32 screenArea, bool refreshActiveApplicationPopupListIt)
{
   PopupState* lPopupState = 0;
   if (refreshActiveApplicationPopupListIt)
   {
      //Due to remove of Popup instance the iterator address requires address update
      mActiveApplicationPopupListIt = mActiveApplicationPopupList.begin();
   }

   while ((mActiveApplicationPopupList.end() != mActiveApplicationPopupListIt) &&
          (0 == lPopupState))
   {
      if ((0 == screenArea) || ((*mActiveApplicationPopupListIt)->GetPopupScreenAreaId() == screenArea))
      {
         lPopupState = (*mActiveApplicationPopupListIt);
      }
      ++mActiveApplicationPopupListIt;
   }

   return lPopupState;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::Find(UInt32 contentId)
{
   PopupStateMap::iterator it = mPopupStateMap.find(contentId);

   // If not found then return 0
   return (mPopupStateMap.end() == it) ? 0 : &((*it).second);
}


// ------------------------------------------------------------------------
UInt32 ActivePopups::GetAppId(UInt32 surfaceId)
{
   for (PopupStateMap::iterator it = mPopupStateMap.begin();
         it != mPopupStateMap.end();
         ++it)
   {
      if (((*it).second).GetSurfaceId() == surfaceId)
      {
         return ((*it).second).GetAppId();
      }
   }
   return 0;
}


// ------------------------------------------------------------------------
void ActivePopups::Dump()
{
   ETG_TRACE_USR1(("### Dump all active popups:"));
   ETG_TRACE_USR1(("#   Active system popups: %d", mSystemPopupList.size()));
   Int i = 0;
   for (PopupList::const_iterator it = mSystemPopupList.begin();
         mSystemPopupList.end() != it;
         ++it)
   {
      ++i;
      {
         char str[230];
         POPUP_LOG(INFO, *it, "#     %d. popup: ", i);
         ETG_TRACE_USR4(("%40s", str));
      }
   }

   for (ApplicationPopupMap::const_iterator it = mApplicationPopupMap.begin();
         mApplicationPopupMap.end() != it;
         ++it)
   {
      UInt32 lAppId = (*it).first;
      const PopupList& lApplicationPopupList = (*it).second;
      if (lApplicationPopupList.size() > 0)
      {
         ETG_TRACE_USR1(("#   Active popups for application %u: %d", lAppId, lApplicationPopupList.size()));

         i = 0;
         for (PopupList::const_iterator itt = lApplicationPopupList.begin();
               lApplicationPopupList.end() != itt; ++itt)
         {
            ++i;
            {
               char str[230];
               POPUP_LOG(INFO, *itt, "#     %d. popup: ", i);
               ETG_TRACE_USR4(("%40s", str));
            }
         }
      }
   }

   PopupState* lTopPopupState = GetOverallTop();
   if (0 != lTopPopupState)
   {
      {
         char str[230];
         POPUP_LOG(INFO, lTopPopupState, "#   Top: ");
         ETG_TRACE_USR4(("%40s", str));
      }
   }
}


// ------------------------------------------------------------------------
void ActivePopups::DumpPriorityList(UInt32 appId)
{
   if (0 != appId)
   {
      ETG_TRACE_USR1(("*** Dump priority popup list considering application %u:", appId));
      Int i = 0;
      // Then traverse through active popups list for setting focus
      PopupState* lPopupState = GetOverallFirst();
      while (0 != lPopupState)
      {
         ++i;
         {
            char str[230];
            POPUP_LOG(INFO, lPopupState, "*     %d. popup: ", i);
            ETG_TRACE_USR4(("%40s", str));
         }
         lPopupState = GetOverallNext();
      }
   }
}


// ------------------------------------------------------------------------
void ActivePopups::UpdateApplicationPopupList(UInt32 screenArea)
{
   mActiveApplicationPopupList.clear();
   PluginActions::ApplicationList lActiveApplications = PluginActions::GetActiveApplications(screenArea);

   for (PluginActions::ApplicationList::const_iterator lIt = lActiveApplications.begin();
         lIt != lActiveApplications.end(); ++lIt)
   {
      ApplicationPopupMap::const_iterator it = mApplicationPopupMap.find(*lIt);
      if (mApplicationPopupMap.end() != it)
      {
         PopupList lMergee((*it).second);
         mActiveApplicationPopupList.merge(lMergee);
      }
   }
}


// ------------------------------------------------------------------------
UInt32 ActivePopups::GetPopupInsertionOrder(PopupState* popupState) const
{
   UInt32 lOrder = 0;
   if (0 != popupState)
   {
      PopupInsertionOrderMap::const_iterator it = mPopupInsertionOrderMap.find(popupState);
      if (mPopupInsertionOrderMap.end() != it)
      {
         lOrder = (*it).second;
      }
   }
   return lOrder;
}


// ------------------------------------------------------------------------
PopupState* ActivePopups::Top(PopupState* systemPopupState,
                              PopupState* appPopupState) const
{
   PopupState* lTopPopupState = 0;
   if (0 == systemPopupState)
   {
      lTopPopupState = appPopupState;
      if (0 != lTopPopupState)
      {
         ETG_TRACE_USR4((">>> Top popup: %u/0x%08X (app(%u): %u/0x%08X, system: n/a)",
                         lTopPopupState->GetSurfaceId(),
                         lTopPopupState->GetUserData(),
                         lTopPopupState->GetAppId(),
                         lTopPopupState->GetSurfaceId(),
                         lTopPopupState->GetUserData()));
      }
      else
      {
         ETG_TRACE_USR4((">>> Top popup: n/a: (app(n/a): n/a, system: n/a)"));
      }
   }
   else if (0 == appPopupState)
   {
      lTopPopupState = systemPopupState;
      ETG_TRACE_USR4((">>> Top popup: %u/0x%08X (app(%u): n/a, system: %u/0x%08X)",
                      lTopPopupState->GetSurfaceId(),
                      lTopPopupState->GetUserData(),
                      lTopPopupState->GetAppId(),
                      lTopPopupState->GetSurfaceId(),
                      lTopPopupState->GetUserData()));
   }
   else
   {
      // With equal priorities take the one first inserted
      if (appPopupState->GetPriority() == systemPopupState->GetPriority())
      {
         lTopPopupState = (GetPopupInsertionOrder(appPopupState) < GetPopupInsertionOrder(systemPopupState)) ?
                          appPopupState :
                          systemPopupState;
      }
      else
      {
         lTopPopupState = (appPopupState->GetPriority() < systemPopupState->GetPriority()) ?
                          appPopupState :
                          systemPopupState;
      }

      ETG_TRACE_USR4((">>> Top popup: %u/0x%08X (app(%u): %u/0x%08X, system: %u/0x%08X)",
                      lTopPopupState->GetSurfaceId(),
                      lTopPopupState->GetUserData(),
                      lTopPopupState->GetAppId(),
                      appPopupState->GetSurfaceId(),
                      appPopupState->GetUserData(),
                      systemPopupState->GetSurfaceId(),
                      systemPopupState->GetUserData()));
   }

   return lTopPopupState;
}


// ------------------------------------------------------------------------
void ActivePopups::UpdateOverallPopupList(UInt32 screenArea)
{
   mOverallPopupList.clear();
   mOverallPopupList = mSystemPopupList;

   UpdateApplicationPopupList(screenArea);
   PopupList lMergee(mActiveApplicationPopupList);
   mOverallPopupList.merge(lMergee, ComparePopupList);
}


// ------------------------------------------------------------------------
bool ActivePopups::ComparePopupList(PopupState* const& first,
                                    PopupState* const& second)
{
   ActivePopups lActivePopups = GetInstance();
   return ((first->GetPriority() == second->GetPriority()) ?
           (lActivePopups.GetPopupInsertionOrder(first) < lActivePopups.GetPopupInsertionOrder(second)) :
           (first->GetPriority() > second->GetPriority()));
}


// ------------------------------------------------------------------------
void ActivePopups::Reset()
{
   ETG_TRACE_USR1(("Reset state of ActivePopups"));
   mOverallPopupList.clear();
   for (ApplicationPopupMap::iterator it = mApplicationPopupMap.begin();
         mApplicationPopupMap.end() != it;
         ++it)
   {
      (*it).second.clear();
   }
   mActiveApplicationPopupList.clear();
   mApplicationPopupMap.clear();
   mSystemPopupList.clear();
   mPopupInsertionOrderMap.clear();
   mPopupStateMap.clear();

   FocusHandler::GetInstance().RemoveSurfaces(true);
}


// ------------------------------------------------------------------------
class TestPopupState : public PopupState
{
   public:
      TestPopupState(UInt32 appId,
                     UInt32 surfaceId,
                     UInt32 userData,
                     UInt32 priority,
                     Modality::Enum modality)
         : PopupState(surfaceId)
      {
         PopupPresentationArg prop(priority, modality,
                                   0, 0, 0, HorizontalAlignment::Center,
                                   VerticalAlignment::Center, Scaling::None, 0);
         RequestArg req(0, appId, userData);
         PopupState::ApplyProperties(0, 0, prop, req);
      }
      virtual ~TestPopupState() {}
};


// ------------------------------------------------------------------------
void ActivePopups::Test(UInt32 appCount,
                        UInt32 popupCount,
                        UInt32 maxPriority)
{
   ETG_TRACE_USR1(("+++++++++++++++++++++++++++++++"));
   ETG_TRACE_USR1(("+++ Start ActivePopups test +++"));
   Reset();

   PopupList lTestDeletionPopupList;

   ETG_TRACE_USR1(("+ Adding randomly %u popups in %u applications (priority 1..%u):",
                   popupCount,
                   appCount,
                   maxPriority));

   UInt32 lAppValueOffset = 1000;
   UInt32 lCount = popupCount;
   // Create random number of popup states & insert them
   while (lCount-- != 0)
   {
      // First simulate switch of application
      UInt32 lAppId = (UInt32(rand()) % appCount + 1) * lAppValueOffset;
      UInt32 lAppSurfaceId = lAppId / 10;
      UInt32 lAppUserData = UInt32(-Int(lAppId * 11111111) / 17);
      ActiveSurfaces::GetInstance().PutOnTop(lAppSurfaceId,
                                             lAppUserData,
                                             lAppId);
      ETG_TRACE_USR1(("+++"));

      // Create random popupState based on application data
      UInt32 lPopupSurfaceId = lAppSurfaceId + (UInt32(rand()) % 9 + 1) * 10;
      TestPopupState lPopupState(lAppId,
                                 lPopupSurfaceId,
                                 UInt32(rand()),
                                 UInt32(rand()) % maxPriority + 1,
                                 (0 == (rand() % 2)) ?
                                 Modality::System :
                                 Modality::Application);
      PopupState* lNewPopupState = 0;
      Insert(lPopupState, lNewPopupState);
      // Create a random popupState deletion list
      if (0 != (rand() % 2))
      {
         lTestDeletionPopupList.push_back(lNewPopupState);
      }
      ETG_TRACE_USR1(("+++"));
   }

   {
      ETG_TRACE_USR1(("+ Dump priority list of system popups (testing iteration functions):"));
      Int i = 0;
      PopupState* lPopupState = GetSysFirst();
      while (0 != lPopupState)
      {
         ++i;
         {
            char str[230];
            POPUP_LOG(INFO, lPopupState, "+     %d. popup: ", i);
            ETG_TRACE_USR4(("%40s", str));
         }
         lPopupState = GetSysNext();
      }
      ETG_TRACE_USR1(("+++"));
   }

   // Dump priority list per application
   for (UInt32 i = 1; i <= appCount; ++i)
   {
      UInt32 lAppId = i * lAppValueOffset;
      ETG_TRACE_USR1(("+ Dump priority list of app(%u) popups:", lAppId));
      Int j = 0;
      PopupState* lPopupState = GetAppFirst(lAppId);
      while (0 != lPopupState)
      {
         ++j;
         {
            char str[230];
            POPUP_LOG(INFO, lPopupState, "+     %d. popup: ", j);
            ETG_TRACE_USR4(("%40s", str));
         }
         lPopupState = GetAppNext(lAppId);
      }
      ETG_TRACE_USR1(("+++"));
   }

   // Dump priority list per application
   for (UInt32 i = 1; i <= appCount; ++i)
   {
      DumpPriorityList(i * lAppValueOffset);
      ETG_TRACE_USR1(("+++"));
   }

   {
      // Dump and delete test deletion list
      ETG_TRACE_USR1(("+ Dump test deletion popup list:"));
      Int j = 0;
      for (PopupList::const_iterator it = lTestDeletionPopupList.begin();
            lTestDeletionPopupList.end() != it;
            ++it)
      {
         {
            char str[230];
            POPUP_LOG(INFO, (*it), "+     %d. popup: ", ++j);
            ETG_TRACE_USR4(("%40s", str));
         }
      }
      for (PopupList::const_iterator it = lTestDeletionPopupList.begin();
            lTestDeletionPopupList.end() != it;
            ++it)
      {
         Remove(*it);
      }
      ETG_TRACE_USR1(("+ Dump after removal of test deletion popup list:"));
      for (UInt32 i = 1; i <= appCount; ++i)
      {
         DumpPriorityList(i * lAppValueOffset);
         ETG_TRACE_USR1(("+++"));
      }
   }

   ETG_TRACE_USR1(("+ Clear all system popups and dump:"));
   mSystemPopupList.clear();
   for (UInt32 i = 1; i <= appCount; ++i)
   {
      DumpPriorityList(i * lAppValueOffset);
      ETG_TRACE_USR1(("+++"));
   }

   ETG_TRACE_USR1(("+ Clear all application popups and dump:"));
   for (ApplicationPopupMap::iterator it = mApplicationPopupMap.begin();
         mApplicationPopupMap.end() != it;
         ++it)
   {
      (*it).second.clear();
   }
   for (UInt32 i = 1; i <= appCount; ++i)
   {
      DumpPriorityList(i * lAppValueOffset);
      ETG_TRACE_USR1(("+++"));
   }

   Reset();
   ETG_TRACE_USR1(("+++ End ActivePopups test +++"));
   ETG_TRACE_USR1(("+++++++++++++++++++++++++++++"));
}


}
