/* ***************************************************************************************
* FILE:          ScreenLayouter.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenLayouter.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
//lint -esym(1511, ScreenBroker::IScreenBrokerPlugin::Create) "static member can't be made virtual"
// This class is intended to be a singleton, for this reason it is ok, that the
// constructor has only private access.
//lint -esym(1704, ScreenBroker::ScreenLayouter::ScreenLayouter)
//"Constructor has private access specification"
// =============================================================================

#if !defined(ScreenBrokerPlugins_ScreenLayouter_ScreenLayouter_h)
#define ScreenBrokerPlugins_ScreenLayouter_ScreenLayouter_h

#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include <Shared/Animation/AnimationProperties.h>
#include <string>
#include <list>
#include <map>
#include <BaseContract/generated/BaseTypes_SurfaceLayerIds.h>

namespace rapidxml {
template <class Ch> class xml_node;
}


namespace ScreenBroker {
/// For direct mapping these values shall map the values available with ilmInputDevice.
namespace InputSource {
enum Enum
{
   None = 0,
   Keyboard = (1 << 0),
   Pointer = (1 << 1),
   Touch = (1 << 2)
};


}

///
namespace Event {
enum Enum
{
   None = 0,
   Key = (1 << 0)
};


}


///
class ScreenLayouter : public IScreenLayouter
{
   public:
      ///

      ///

      ///
      static IScreenBrokerPlugin& Create();

      ///
      static const Char* ScreenLayoutsXmlEnvVar()
      {
         return "PLUGIN_SCREENLAYOUTER_XML";
      }

      ///
      static const Char* ScreenLayoutsXsdEnvVar()
      {
         return "PLUGIN_SCREENLAYOUTER_XSD";
      }

      ///
      static const Char* ScreenLayoutIdEnvVar()
      {
         return "PLUGIN_SCREENLAYOUTER_INITIAL_LAYOUT_ID";
      }

      ///
      virtual ~ScreenLayouter();

      ///
      virtual bool Init();

      ///
      virtual void Reset();

      ///
      virtual void Diagnosis();

      ///
      virtual void Test();

      ///
      virtual ScreenArea* GetScreenArea(UInt32 screenAreaId);

      ///
      virtual ScreenAreaList& GetScreenAreaList();

      ///
      virtual void Action(UInt32 actionId, UInt32 actionData);

      ///
      virtual void RequestCurrentStatus(UInt32 requestId);

      ///
      virtual bool getAnimationMethod(UInt& method);

      /**
       * @brief Provides the (user defined) event information for given
       * surface.
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param surfaceId The surface the information is requested.
       * @return A surface event information (e.g. a bitfield).
       */
      UInt32 GetEvents(UInt32 surfaceId);

      /**
       * @brief Provides the (user defined) input source information
       * (e.g. according to ilmInputDevice) for given surface.
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param surfaceId The surface the information is requested.
       * @return A surface input source information (e.g. a bitfield).
       */
      UInt32 GetInputSources(UInt32 surfaceId);

      /**
       * @brief Provides the (user defined) animation properties information
       * for given surface.
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param surfaceId The surface the information is requested.
       * @param animationDirection The direction the animation properties is requested.
       * @return A surface animation properties information.
       */
      const AnimationProperties* GetAnimationProperties(UInt32 surfaceId,
            AnimationDirection::Enum animationDirection, AnimationType::Enum animationType = AnimationType::None);

      /**
       * @brief Provides the (user defined) animation properties information
       * for given Layer.
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param layerId The surface the information is requested.
       * @param animationDirection The direction the animation properties is requested.
       * @return A surface animation properties information.
       */
      const AnimationProperties* GetLayerAnimationProperties(UInt32 layerId,
            AnimationDirection::Enum animationDirection, AnimationType::Enum animationType = AnimationType::None);

      /**
       * @brief Provides the (user defined) information if commit has to be triggered explicit.
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param surfaceId The surface the information is requested
       * @return Boolean flag if commit has to be triggered explicit.
       */
      bool GetExplicitCommit(UInt32 surfaceId);

      /**
       * @brief Get Id of display where surface is shown.
       *
       * @note This information is not processed in screenbroker core!
       *
       *
       * @param surfaceId The surface the information is requested.
       * @return DisplayId of display where the surface is shown.
       */
      virtual UInt32 GetDisplayIdOfSurface(UInt32 surfaceId);
      /**
      * @brief Get Id of ScreenArea where surface is shown.
      *
      * @note This information is not processed in screenbroker core!
      *
      *
      * @param surfaceId The surface the information is requested.
      * @return ScreenAreaId of ScreenArea where the surface is shown.
      */
      virtual UInt32 GetScreenAreaIdOfSurface(UInt32 surfaceId);
      /**
      * @brief check if observeAndAdd attribute is set for the given surface
      *
      * @note This information is not processed in screenbroker core!
      *
      *
      * @param surfaceId The surface information is requested.
      * @return bool val true if the attribute is set.
      */
      virtual bool IsSurfaceIsObserved(UInt32 surfaceId);
      /**
       * @brief Get Id of display where screenArea is shown.
       *
       * @note This information is not processed in screenbroker core!
       *
       *
       * @param screenArea The screanArea the information is requested.
       * @return DisplayId of display where the surface is shown.
       */
      virtual UInt32 GetDisplayIdOfScreenArea(UInt32 screenArea);

      /**
       * @brief Get list of seats for this surface
       *
       * @note This information is not processed in screenbroker core!
       *
       * @param surfaceId The surface the information is requested.
       * @return List of seats
       */
      virtual SeatList* GetSeatList(UInt32 surfaceId);

      /**
       * @brief GetDisplayIdList
       * @return
       */
      virtual DisplayIdList& GetDisplayIdList();

      AnimationDirection::Enum GetAnimationDirection(UInt32 surfaceId, AnimationType::Enum animationType);

      ///
      UInt32 GetDisplayIdentifier(hmibase::DisplayAliasEnum displayAlias);
      bool GetDisplayIdentifier(UInt32 displayAlias, UInt32& displayId);

   private:
      typedef std::map<AnimationDirection::Enum, AnimationProperties> AnimationPropertiesMap;
      typedef std::list<AnimationProperties> CustomAnimationList;
      typedef std::map<AnimationDirection::Enum, CustomAnimationList> CustomAnimationPropertiesMap;

      Int16 _animationMethod;
      struct SurfaceProperties
      {
         /// Bitfield using Screenbroker::Event values
         UInt32 events;
         /// Bitfield using Screenbroker::InputSource values
         UInt32 inputSources;
         /// Animation properties per animation direction
         AnimationPropertiesMap animationProperties;
         /// Animation properties per animation direction
         CustomAnimationPropertiesMap customAnimationProperties;
         /// Boolean flag to set explicit commit for this surface
         bool explicitCommit;
         /// Set default values
         SurfaceProperties() : events(0), inputSources(0), explicitCommit(false) {}
      };

      struct LayerProperties
      {
         /// Animation properties per animation direction
         AnimationPropertiesMap animationProperties;
         /// Animation properties per animation direction
         CustomAnimationPropertiesMap customAnimationProperties;
      };

      typedef std::map<UInt32, ScreenArea*> ScreenAreaMap;
      typedef std::list<const ScreenArea*> ScreenAreaRefList;
      typedef std::map<UInt32, ScreenAreaRefList> LayerRenderOrder;
      typedef std::map<UInt32, SurfaceProperties> SurfacePropertiesMap;
      typedef std::map<UInt32, LayerProperties> LayerPropertiesMap;

      typedef std::map<UInt32, UInt32> SurfaceLayerMap;
      typedef std::map<UInt32, UInt32> LayerDisplayMap;
      typedef std::map<UInt32, SeatList> SeatMap;
      typedef std::map<UInt32, bool> SurfaceObserveAndAddMap;

      typedef std::map<hmibase::DisplayAliasEnum, UInt32> DisplayIdentifierMap; // key is alias name, value is real display id

      /**
       * @brief Fast access to screen area list, via screen area ID.
       */
      ScreenAreaMap mScreenAreaMap;
      SurfaceObserveAndAddMap mSurfaceObserveAndAddMap;

      /**
       * @brief List of z-ordered screen areas of initial screen layout.
       */
      ScreenAreaList mScreenAreaList;

      ///
      LayerRenderOrder mLayerRenderOrder;

      ///
      LayerPropertiesMap mLayerPropertiesMap;

      ///
      SurfacePropertiesMap mSurfacePropertiesMap;

      ///
      SurfaceLayerMap mSurfaceLayerMap;

      ///
      LayerDisplayMap mLayerDisplayMap;

      ///
      SeatMap mSeatMap;

      ///
      DisplayIdList mDisplayIdList;

      ///
      DisplayIdentifierMap mDisplayIdentifierMap;

      bool mInitialized;

      ///
      ScreenLayouter();

      ///
      std::string GetEnvironmentVariable(const std::string& key) const;

      ///
      template <typename T> static T ConvertTo(std::string from);

      ///
      bool ConfigureDisplay(UInt32 displayId);

      ///
      void ReadDisplays(const rapidxml::xml_node<char>* screenLayoutNode);

      ///
      void ReadScreenArea(const rapidxml::xml_node<char>* screenAreaNode, UInt32 displayId);

      ///
      void ReadScreenAreaRef(const rapidxml::xml_node<char>* ScreenAreaRefNode, UInt32 displayId);

      ///
      void ReadSurfaces(const rapidxml::xml_node<char>* displayNode);

      ///
      void readAnimationConfig(const rapidxml::xml_node<char>* animationNode);

      ///
      void ReadRegistrationData(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId);

      ///
      void ReadEvents(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId);

      ///
      void ReadInputSources(const rapidxml::xml_node<char>* inputSourcesNode, UInt32 surfaceId);

      ///
      void ReadSeats(const rapidxml::xml_node<char>* screenLayoutNode, UInt32 surfaceId);

      ///
      void ReadAnimation(const rapidxml::xml_node<char>* animationNode, UInt32 surfaceId, bool isLayerContent = false);

      ///
      void ReadAnimationProperties(const rapidxml::xml_node<char>* animationNode,
                                   const char* xmlNode,
                                   UInt32 surfaceId, bool isLayerContent = false);

      ///
      void ReadCustomAnimationProperties(const rapidxml::xml_node<char>* animationNode,
                                         const char* xmlNode,
                                         UInt32 surfaceId, bool isLayerContent = false);

      ///
      void ReadCommit(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId);

      ///
      void CreateBuildInLayer(UInt32 displayId);

      ///
      void UpdateDisplayIdentifier(hmibase::DisplayAliasEnum displayAlias, UInt32 displayId);

      ///
      hmibase::DisplayAliasEnum GetDisplayAlias(UInt32 displayId);
};


}

#endif
