/* ***************************************************************************************
* FILE:          PopupManagerActivator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupManagerActivator.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Use of lPopupState null pointer in ScreenBroker::PopupManagerActivator::HidePopup
// is prevented by if conditions
//lint -efunc(613, ScreenBroker::PopupManagerActivator::HidePopup) "Possible use of null pointer"
// This header file is out of scope of sbplugins.
//lint -esym(451, errno.h) "Header file '/usr/include/errno.h' repeatedly included but does not have a standard include guard"
// Required for global registration for plugin
//lint -esym(1502, gPopupManagerActivatorPlugin) "defined object has no nonstatic data members
//lint -esym(1502, sInstance) "defined object has no nonstatic data members
// =============================================================================

#include "PopupManagerActivator.h"
#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include <ScreenBroker/Service/PopupManager/PopupManager.h>
#include <ScreenBroker/Service/ServiceApi.h>
#include <ScreenBroker/Util/Time.h>
#include <Shared/ActiveSurfaces.h>
#include <Shared/ActivePopups.h>
#include <Shared/AnimationHandler.h>
#include <Shared/FocusHandler.h>
#include <Shared/IlmAccessor.h>
#include <ScreenLayouter/ScreenLayouter.h>

#include <Shared/PluginActions.h>
#include <Shared/PopupFilter.h>
#include <Base.h>
#include <bitset>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/PopupManagerActivator.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::PopupManagerActivator);

// ------------------------------------------------------------------------
// Instantiate (plugin) global handled ActivePopups object
ActivePopups gActivePopupsInstance;

// ------------------------------------------------------------------------
// Instantiate (plugin) global handled PopupFilter object
ActivePopups gPopupFilterInstance;

// ------------------------------------------------------------------------
UInt32 cInvalidPosition = static_cast<UInt32>(-1);

// ------------------------------------------------------------------------
IScreenBrokerPlugin& PopupManagerActivator::Create()
{
   static ScreenBroker::PopupManagerActivator sInstance;
   return sInstance;
}


// ------------------------------------------------------------------------
PopupManagerActivator::PopupManagerActivator()
{
}


// ------------------------------------------------------------------------
PopupManagerActivator::~PopupManagerActivator()
{
}


// ------------------------------------------------------------------------
bool PopupManagerActivator::Init()
{
   bool lRc = VersionCheck();
   if (lRc)
   {
      /* not required here: ETG_TRACE_USR1(("Initializing plugin PopupManagerActivator version %40s", SCREENBROKERPLUGINS_VERSION_STRING)); */
   }
   return lRc;
}


// ------------------------------------------------------------------------
void PopupManagerActivator::Reset()
{
   ETG_TRACE_USR1(("*** Reset ***"));
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   // Remove popup filter
   PopupFilter::GetInstance().Reset();

   // Remove active popups
   ActivePopups& lActivePopups = ActivePopups::GetInstance();
   PopupState* lPopupState = lActivePopups.GetOverallTop();
   while (0 != lPopupState)
   {
      ETG_TRACE_USR1(("Resetting popup (ID:%u)", lPopupState->GetSurfaceId()));
      PopupManager::GetInstance().RemovePopup(*lPopupState);
      FocusHandler::EnableInputEvents(lPopupState->GetSurfaceId(), false);
      IlmAccessor::SetSurfaceVisibility(lPopupState->GetSurfaceId(), ILM_FALSE, false, lIlmError);
      lPopupState = lActivePopups.Remove(lPopupState);
   }
   IScreenLayouter* lScreenLayouter = ServiceApi::ScreenLayouter();
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::DisplayIdList lDisplayIdList = lScreenLayouter->GetDisplayIdList();
      for (IScreenLayouter::DisplayIdList::iterator lIt = lDisplayIdList.begin(); lIt != lDisplayIdList.end(); ++lIt)
      {
         // Update input focus to active surface ID
         (void) FocusHandler::GetInstance().UpdateInputFocus(ActiveSurfaces::GetInstance().GetTop(*lIt));
      }
   }
   else
   {
      ETG_TRACE_SYS(("@ Resetting popup: ScreenLayouter instance not accessible!"));
   }
   (void) IlmAccessor::Commit(lIlmError);
}


// ------------------------------------------------------------------------
void PopupManagerActivator::Diagnosis()
{
   DumpPopupList();
   ActivePopups::GetInstance().Dump();
   PopupFilter::GetInstance().Dump();
}


// ------------------------------------------------------------------------
void PopupManagerActivator::Test()
{
   ActivePopups::GetInstance().Test(3, 10, 9);
}


// ------------------------------------------------------------------------
void PopupManagerActivator::ShowPopup(const ServiceRequestArg& serviceRequestArg,
                                      PopupState& popupState,
                                      UInt32 hint)
{
   (void) hint;
   ETG_TRACE_USR1(("ShowPopup called for popup %u/0x%08X",
                   popupState.GetSurfaceId(),
                   popupState.GetUserData()));

   // TODO: remove when we have fix for https://rb-alm-20-p.de.bosch.com/ccm/resource/itemName/com.ibm.team.workitem.WorkItem/261204
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   //workaround fix for Bug 261204
   usleep(50000);
   ETG_TRACE_USR1(("ShowPopup delayed for 50 ms"));
#endif

   // Get instance of active popups container
   ActivePopups& lActivePopups = ActivePopups::GetInstance();

   // Retrieve instance of popup manager
   PopupManager& lPopupManager = PopupManager::GetInstance();

   // Get surface ID and appId of active application
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   UInt32 lActiveSurfaceId = lActiveSurfaces.GetTop(PluginActions::GetDisplayIdOfSurface(popupState.GetSurfaceId()));
   UInt32 lActiveAppId = lActiveSurfaces.GetAppId(lActiveSurfaceId);

   // If popupState is already queued, mark it as update
   bool lIsPopupUpdate = (0 != lActivePopups.Find(popupState.GetUserData()));

   // Add popup state to active popup states
   PopupState* lCurrentTopPopupState = lActivePopups.GetOverallTop(true, popupState.GetPopupScreenAreaId());
   PopupState* lPopupState = 0;
   PopupState* lNewTopPopupState = lActivePopups.Insert(popupState, lPopupState);

   bool lRc = (0 != lPopupState);
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   if (lRc)
   {
      // Set focus priority of popup in focus priority map
      FocusHandler::GetInstance().SetKeyboardPriority(lPopupState->GetSurfaceId(), lPopupState->GetFocusPriority());
      FocusHandler::GetInstance().SetPointerPriority(lPopupState->GetSurfaceId(), lPopupState->GetFocusPriority());

      // Get popup screen area ID the given popup is related to
      UInt32 lPopupScreenAreaId = lPopupState->GetPopupScreenAreaId();

      // -----------
      // Adds given surface to given layer, if not already done & commit it right away
      // Note: This is done only once and not removed during lifecycle
      if (!lIsPopupUpdate)
      {
         lRc = lRc && IlmAccessor::AddSurfaceToLayer(lPopupState->GetLayerId(),
               lPopupState->GetSurfaceId(),
               false,
               lIlmError);
         if (!lRc)
         {
            ServiceApi::NotifyError(serviceRequestArg, 0, static_cast<Int32>(lIlmError));
         }
      }

      // -----------
      // Do surface position and dimension manipulation as requested from popupState
      // Update scaling and alignment as requested from popupState
      if (lRc)
      {
         lRc = IlmAccessor::UpdateAppearance(lPopupState->GetLayerId(),
                                             lPopupState->GetSurfaceId(),
                                             lPopupState->GetScaling(),
                                             lPopupState->GetHorizontalAlignment(),
                                             lPopupState->GetVerticalAlignment(),
                                             false, lIlmError);
      }

      // -----------
      // Add/update popup to/in popup manager to enable time-out handling and perform surface state manipulation and notification
      if (lRc)
      {
         // Enable time-out handling if presentation time > 0 (timer update done by adding the popup to popup manager)
         // Note: Time-out is used to notify popup expiration.
         lPopupState->SetNextUpdatePeriod(lPopupState->GetPresentationTime());

         // Only add/update popupState if timing values are valid
         if (PluginActions::IsPopupTimingValid(*lPopupState))
         {
            // Cancel running animations (if any) with new possible surface change request
            AnimationHandler::AbortAnimations(lPopupState->GetSurfaceId());

            // Reschedule popup state, if already queued in popup management
            if (lIsPopupUpdate)
            {
               lPopupManager.ResetTimer();
               ETG_TRACE_USR1(("Popup %u/0x%08X is already queued - reschedule to %ums",
                               lPopupState->GetSurfaceId(),
                               lPopupState->GetUserData(),
                               lPopupState->GetPresentationTime()));
            }
            else
            {
               // Add new popup
               ETG_TRACE_USR4(("Popup %u/0x%08X add to popup screen area(ID:%u)",
                               lPopupState->GetSurfaceId(),
                               lPopupState->GetUserData(),
                               lPopupScreenAreaId));
               lPopupManager.AddPopup(*lPopupState);
            }

            // Print info about popup states current update period, if set
            if (0 != lPopupState->GetNextUpdatePeriod())
            {
               ETG_TRACE_USR1(("Popup %u/0x%08X new update period is %u",
                               lPopupState->GetSurfaceId(),
                               lPopupState->GetUserData(),
                               lPopupState->GetNextUpdatePeriod()));
            }

            // Update focus and do notification, if requested popup becomes new top popup
            // (and in case it is application modal, the referenced appId must be the currently active)
            if ((lPopupState == lNewTopPopupState) &&
                  PluginActions::IsPopupActive(*lNewTopPopupState))
            {
               // If any current top popup was available (means currently visible) close/queue it
               if (0 != lCurrentTopPopupState)
               {
                  // If the requested popup (the new top popup in this decision path) differs from the current top popup
                  if (lNewTopPopupState != lCurrentTopPopupState)
                  {
                     // If special handling "CloseOnSuperseded" is not applicable to the current top popup, queue it
                     if (!CloseOnSuperseded(lCurrentTopPopupState, lNewTopPopupState))
                     {
                        bool lUpdateSurfaceVisibility = (lCurrentTopPopupState->GetSurfaceId() != lNewTopPopupState->GetSurfaceId());
                        bool lSendNotification = true;
                        (void) PluginActions::NotifyPopupVisibility(lCurrentTopPopupState,
                              SurfaceState::Queued,
                              lUpdateSurfaceVisibility,
                              lSendNotification);
                     }
                  }
               }

               // Map popup & notify respective application considering popup filter
               SurfaceState::Enum lSurfaceState = PluginActions::IsPopupFiltered(*lNewTopPopupState) ?
                                                  SurfaceState::Queued :
                                                  SurfaceState::Mapped;
               bool lUpdateSurfaceVisibility = (SurfaceState::Mapped == lSurfaceState) &&
                                               ((0 == lCurrentTopPopupState) ||
                                                (lNewTopPopupState->GetSurfaceId() != lCurrentTopPopupState->GetSurfaceId()));

               // notify if first popup of a layer comes to Foreground & filter - notifyOnPopupStatus is set
               bool notifyOnPopupStatus = false;
               if ((SurfaceState::Mapped == lSurfaceState) && (0 == lCurrentTopPopupState))
               {
                  notifyOnPopupStatus = PluginActions::IsNotifyOnPopStatusForScreenAreaId(lNewTopPopupState->GetPopupScreenAreaId());
               }

               bool lSendNotification = true;
               lRc = lRc && PluginActions::NotifyPopupVisibility(lNewTopPopupState,
                     lSurfaceState,
                     lUpdateSurfaceVisibility,
                     lSendNotification,
                     notifyOnPopupStatus);

               // Update input focus
               if (lNewTopPopupState != lCurrentTopPopupState)
               {
                  (void) FocusHandler::GetInstance().UpdateInputFocus(lPopupState->GetSurfaceId());
               }
            }
            else
            {
               // If special handling "CloseOnSuperseded" is not applicable to the requested top popup, queue it
               // Note: lPopupState may become 0, in case "close on superseded" is applied (returns true)
               if (!CloseOnSuperseded(lPopupState, lNewTopPopupState))
               {
                  bool lUpdateSurfaceVisibility = false;
                  bool lSendNotification = true;
                  lRc = lRc && PluginActions::NotifyPopupVisibility(lPopupState,
                        SurfaceState::Queued,
                        lUpdateSurfaceVisibility,
                        lSendNotification);
               }
            }

            // Perform visibility transformation (if available)
            AnimationHandler::TriggerAnimations();

            // Do a final all-in-one commit of all transactions towards layer manager
            if (lRc)
            {
               (void) IlmAccessor::Commit(lIlmError);
            }
         }
         else
         {
            // If popup is rescheduled but timing values are not valid remove popupState
            if (lIsPopupUpdate)
            {
               ServiceRequestArg lServiceRequestArg(0, lPopupState->GetAppId(), lPopupState->GetUserData(), "");   // Use dummy service request arg
               HidePopup(lServiceRequestArg, *lPopupState);
            }
         }
      }

      // Dump popup timing list
      DumpPopupList();

      // Dump popup priority list
      lActivePopups.DumpPriorityList(lActiveAppId);
   }
   else
   {
      ETG_TRACE_ERR(("Insert popup %u/0x%08X to active popups container failed",
                     popupState.GetSurfaceId(),
                     popupState.GetUserData()));
   }
}


// ------------------------------------------------------------------------
void PopupManagerActivator::HidePopup(const ServiceRequestArg& serviceRequestArg,
                                      PopupState& popupState)
{
   ETG_TRACE_USR1(("HidePopup called for popup %u/0x%08X",
                   popupState.GetSurfaceId(),
                   serviceRequestArg.UserData()));

   // Get instance of active popups container
   ActivePopups& lActivePopups = ActivePopups::GetInstance();

   // Retrieve instance of popup manager
   PopupManager& lPopupManager = PopupManager::GetInstance();

   // Ignore hide request if popup state queued has not the same userData
   // (aka. viewId) as given in request.
   PopupState* lPopupState = lActivePopups.Find(serviceRequestArg.UserData());

   // Do a thorough check if the found popup state is the correct one
   bool lRc = ((0 != lPopupState) &&
               (lPopupState->GetAppId() == serviceRequestArg.AppId()) &&
               (lPopupState->GetAppId() == popupState.GetAppId()) &&
               (lPopupState->GetSurfaceId() == popupState.GetSurfaceId()));

   ilmErrorTypes lIlmError = ILM_SUCCESS;

   if (lRc)
   {
      // Get popup screen area ID the given popup is related to
      UInt32 lPopupScreenAreaId = lPopupState->GetPopupScreenAreaId();

      // Remove popup state in popup manager prior to ActivePopups. Remove,
      // as the latter will delete the instance referred in PopupManager
      ETG_TRACE_USR4(("Remove popup with surface ID:%u from popup screen area ID:%u",
                      lPopupState->GetSurfaceId(),
                      lPopupScreenAreaId));
      lPopupManager.RemovePopup(*lPopupState);

      // Make a save copy for popup state for the ability to access data
      // after removal of instance stored in active popups container
      PopupState lRemovedPopupState = *lPopupState;

      // Remove popup from active popups
      // (Attention: This also removes the instance referred by lPopupState!!!)
      (void) lActivePopups.Remove(lPopupState);

      // Cancel running animations (if any) with new possible surface change request
      AnimationHandler::AbortAnimations(lPopupState->GetSurfaceId());

      // Get surface ID of active application
      ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
      UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(lRemovedPopupState.GetSurfaceId());
      UInt32 lActiveSurfaceId = lActiveSurfaces.GetTop(lDisplayId);
      UInt32 lActiveAppId = lActiveSurfaces.GetAppId(lActiveSurfaceId);

      bool lVisible = lRemovedPopupState.IsVisible();
      PopupState* lNewTopPopupState = 0;
      bool lUpdateSurfaceVisibility = lVisible;

      if (lVisible)
      {
         // Re-activate next top popup, if available
         lNewTopPopupState = PluginActions::RemoveExpiredTopPopups(lPopupScreenAreaId);
         if (0 != lNewTopPopupState)
         {
            lUpdateSurfaceVisibility = lRemovedPopupState.GetSurfaceId() != lNewTopPopupState->GetSurfaceId();
            // otherwise re-activate currently active surface, if available
         }
         else if (0 != lActiveSurfaceId)
         {
            lUpdateSurfaceVisibility = lRemovedPopupState.GetSurfaceId() != lActiveSurfaceId;
         }
      }

      //notify if last popup of a layer goes to background & filter-notifyOnPopupStatus is set
      bool notifyOnPopupStatus = false;
      if ((lVisible) && (lNewTopPopupState == 0))
      {
         notifyOnPopupStatus = PluginActions::IsNotifyOnPopStatusForScreenAreaId(lRemovedPopupState.GetPopupScreenAreaId());
      }

      bool lSendNotification = true;
      // Unmap/unqueue requested popup surface and notify respective application
      lRc = lRc && PluginActions::NotifyPopupVisibility(&lRemovedPopupState,
            lVisible ?
            SurfaceState::Unmapped :
            SurfaceState::Unqueued,
            lUpdateSurfaceVisibility,
            lSendNotification,
            notifyOnPopupStatus);

      if (lVisible)
      {
         UInt32 lNewSurfaceId = 0;
         if (0 != lNewTopPopupState)
         {
            // Set new focused surface ID
            lNewSurfaceId = lNewTopPopupState->GetSurfaceId();

            // Map & notify re-activated surface
            lRc = lRc && PluginActions::NotifyPopupVisibility(lNewTopPopupState,
                  SurfaceState::Mapped);

            FocusHandler& lFocusHandler = FocusHandler::GetInstance();
            // Set focus priority of popup
            lFocusHandler.SetKeyboardPriority(lNewTopPopupState->GetSurfaceId(),
                                              lNewTopPopupState->GetFocusPriority());
            lFocusHandler.SetPointerPriority(lNewTopPopupState->GetSurfaceId(),
                                             lNewTopPopupState->GetFocusPriority());

            // Add popup surface to focus handler
            lFocusHandler.PushBackSurface(lNewTopPopupState->GetSurfaceId());
            // otherwise re-activate currently active surface, if available
         }
         else if (0 != lActiveSurfaceId)
         {
            // Set new focused surface ID
            lNewSurfaceId = lActiveSurfaceId;
         }
         else if (0 != lActiveSurfaces.GetTop(lDisplayId, false))
         {
            // Set new focused surface ID to a permanent surface of no other surface is available
            lNewSurfaceId = lActiveSurfaces.GetTop(lDisplayId, false);
         }

         // Update input focus
         (void) FocusHandler::GetInstance().UpdateInputFocus(lNewSurfaceId);
      }

      // Perform visibility transformation (if available)
      AnimationHandler::TriggerAnimations();

      // Do a final all-in-one commit of all transactions towards layer manager
      if (lRc)
      {
         (void) IlmAccessor::Commit(lIlmError);
      }
      // Dump popup timer list
      DumpPopupList();

      // Dump popup priority list
      lActivePopups.DumpPriorityList(lActiveAppId);
   }
   else
   {
      ETG_TRACE_SYS(("Ignoring hide popup (%u/0x%08X) for app %u: No popup found matching request { %u, %u, 0x%08X }",
                     popupState.GetSurfaceId(),
                     popupState.GetUserData(),
                     popupState.GetAppId(),
                     serviceRequestArg.RequestId(),
                     serviceRequestArg.AppId(),
                     serviceRequestArg.UserData()));
   }
}


// ------------------------------------------------------------------------
void PopupManagerActivator::CloseOnExternalTouch(const ServiceRequestArg& serviceRequestArg)
{
   ETG_TRACE_USR1(("CloseOnExternalTouch called"));
   (void) serviceRequestArg;

   PopupState* lPopupState = PluginActions::GetTopVisiblePopupForDisplay(serviceRequestArg.UserData());
   if (0 != lPopupState)
   {
      ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowPopupMode::Enum> lRequestId(lPopupState->GetRequestId());
      bool lCloseOnExternalTouch = lRequestId.Test(ScreenBrokerProtocol::ShowPopupMode::CloseOnExternalTouch);

      if (lCloseOnExternalTouch)
      {
         UpdatePopup(*lPopupState);
      }
   }
}


// ------------------------------------------------------------------------
void PopupManagerActivator::UpdatePopup(PopupState& popupState)
{
   ETG_TRACE_USR1(("UpdatePopup called for popup %u/0x%08X",
                   popupState.GetSurfaceId(),
                   popupState.GetUserData()));

   ServiceRequestArg serviceRequestArg(0, popupState.GetAppId(), popupState.GetUserData(), "");   // Use dummy service request arg
   HidePopup(serviceRequestArg, popupState);
}


// ------------------------------------------------------------------------
void PopupManagerActivator::Action(UInt32 actionId, UInt32 actionData)
{
   ETG_TRACE_SYS(("No action %u with data %u available! Request ignored!",
                  actionId,
                  actionData));
}


// ------------------------------------------------------------------------
void PopupManagerActivator::RequestCurrentStatus(UInt32 requestId)
{
   // Get instance of active popups container
   ActivePopups& lActivePopups = ActivePopups::GetInstance();

   switch (requestId)
   {
      case ScreenBrokerProtocol::CurrentStatusRequestId::ActivePopups:
      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::ActivePopups:
      {
         // Traverse through active popups list
         PopupState* lPopupState = lActivePopups.GetOverallFirst();
         if (0 != lPopupState)
         {
            while (0 != lPopupState)
            {
               std::bitset<sizeof(UInt32)> lStatus(0);
               if (lPopupState->IsVisible())
               {
                  lStatus.set(0);
               }
               ServiceApi::NotifyCurrentStatus(requestId,
                                               lPopupState->GetSurfaceId(),
                                               UInt32(lStatus.to_ulong()),
                                               lPopupState->GetUserData());
               ETG_TRACE_USR1(("%40s (ActivePopups): %u/0x%08x: status(%u)",
                               __FUNCTION__,
                               lPopupState->GetSurfaceId(),
                               lPopupState->GetUserData(),
                               UInt32(lStatus.to_ulong())));
               lPopupState = lActivePopups.GetOverallNext();
            }
         }
         else
         {
            ETG_TRACE_SYS(("%40s (ActivePopups): No active popups found! Request ignored!", __FUNCTION__));
         }
         break;
      }
      case ScreenBrokerProtocol::CurrentStatusRequestId::TopActivePopup:
      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActivePopup:
      {
         std::set<UInt32> lScreenAreas;

         PopupState* lPopupState = lActivePopups.GetOverallFirst();
         while (0 != lPopupState)
         {
            lScreenAreas.insert(lPopupState->GetPopupScreenAreaId());
            lPopupState = lActivePopups.GetOverallNext();
         }

         for (std::set<UInt32>::iterator lIt = lScreenAreas.begin(); lIt != lScreenAreas.end(); ++ lIt)
         {
            lPopupState = lActivePopups.GetOverallTop(false, *lIt);
            if (0 != lPopupState)
            {
               std::bitset<sizeof(UInt32)> lStatus(0);
               if (lPopupState->IsVisible())
               {
                  lStatus.set(0);
               }
               ServiceApi::NotifyCurrentStatus(requestId,
                                               lPopupState->GetSurfaceId(),
                                               UInt32(lStatus.to_ulong()),
                                               lPopupState->GetUserData());
               ETG_TRACE_USR1(("%40s (TopActivePopup): %u/0x%08x: status(%u)",
                               __FUNCTION__,
                               lPopupState->GetSurfaceId(),
                               lPopupState->GetUserData(),
                               UInt32(lStatus.to_ulong())));
            }
         }

         if (0 == lPopupState)
         {
            ETG_TRACE_SYS(("%40s (TopActivePopup): No top popup found! Request ignored!", __FUNCTION__));
         }
         break;
      }
      case ScreenBrokerProtocol::CurrentStatusRequestId::TopActiveVisiblePopup:
      case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveVisiblePopup:
      {
         std::set<UInt32> lScreenAreas;

         PopupState* lPopupState = lActivePopups.GetOverallFirst();
         while (0 != lPopupState)
         {
            lScreenAreas.insert(lPopupState->GetPopupScreenAreaId());
            lPopupState = lActivePopups.GetOverallNext();
         }

         for (std::set<UInt32>::iterator lIt = lScreenAreas.begin(); lIt != lScreenAreas.end(); ++ lIt)
         {
            lPopupState = lActivePopups.GetOverallLast(*lIt);
            while ((0 != lPopupState) && (!lPopupState->IsVisible()))
            {
               lPopupState = lActivePopups.GetOverallPrevious(*lIt);
            }

            if (0 != lPopupState)
            {
               std::bitset<sizeof(UInt32)> lStatus(0);
               if (lPopupState->IsVisible())
               {
                  lStatus.set(0);
               }
               ServiceApi::NotifyCurrentStatus(requestId,
                                               lPopupState->GetSurfaceId(),
                                               UInt32(lStatus.to_ulong()),
                                               lPopupState->GetUserData());
               ETG_TRACE_USR1(("%40s (TopActiveVisiblePopup): %u/0x%08x: status(%u)",
                               __FUNCTION__,
                               lPopupState->GetSurfaceId(),
                               lPopupState->GetUserData(),
                               UInt32(lStatus.to_ulong())));
            }
         }

         if (0 == lPopupState)
         {
            ServiceApi::NotifyCurrentStatus(requestId,
                                            0,
                                            0,
                                            0);

            ETG_TRACE_USR1(("%40s (TopActiveVisiblePopup): No visible top popup found!", __FUNCTION__));
         }
         break;
      }
      default:
         ETG_TRACE_SYS(("No status for request %u available! Request ignored!", requestId));
         break;
   }
}


// ------------------------------------------------------------------------
void PopupManagerActivator::SetPopupFilter(const ServiceRequestArg& serviceRequestArg,
      bool disableAll,
      const PopupPresentationArg& popupPresentationArg)
{
   // Get instance of popup filter container and insert transported filter data for given application
   UInt32 lFilterDisplayId = serviceRequestArg.UserData();
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      UInt32 tmp = 0;
      if (lScreenLayouter->GetDisplayIdentifier(lFilterDisplayId, tmp) == false)
      {
         ETG_TRACE_ERR(("SetPopupFilter: given display alias %d can't be mapped to display id, so use it without mapping", lFilterDisplayId));
      }
      else
      {
         lFilterDisplayId = tmp;
      }
   }
   UInt32 lFilterAppId = serviceRequestArg.AppId();
   ETG_TRACE_USR1(("Set popup filter for display %u, app: %u", lFilterDisplayId, lFilterAppId));

   PopupFilter& lPopupFilter = PopupFilter::GetInstance();
   lPopupFilter.Insert(lFilterDisplayId, lFilterAppId, popupPresentationArg, disableAll);
   lPopupFilter.Dump();

   // Cancel running animations (if any) with new possible surface change request
   AnimationHandler::AbortAnimations();

   // Check and notify filtered popups (after the filter was added to the filter container)
   PluginActions::NotifyPopupFilter();
   FocusHandler::GetInstance().UpdateInputFocus(0);

   // Perform visibility transformation (if available)
   AnimationHandler::TriggerAnimations();

   // Do a final all-in-one commit of all transactions towards layer manager
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   (void) IlmAccessor::Commit(lIlmError);
}


// ------------------------------------------------------------------------
void PopupManagerActivator::ClearPopupFilter(const ServiceRequestArg& serviceRequestArg)
{
   // Get instance of popup filter container and delete filter data for given application
   UInt32 lFilterAppId = serviceRequestArg.AppId();
   UInt32 lFilterDisplayId = serviceRequestArg.UserData();
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      UInt32 tmp = 0;
      if (lScreenLayouter->GetDisplayIdentifier(lFilterDisplayId, tmp) == false)
      {
         ETG_TRACE_ERR(("ClearPopupFilter: given display alias %d can't be mapped to display id, so use it without mapping", lFilterDisplayId));
      }
      else
      {
         lFilterDisplayId = tmp;
      }
   }
   ETG_TRACE_USR1(("Clear popup filter for display %u, app: %u", lFilterDisplayId, lFilterAppId));

   PopupFilter& lPopupFilter = PopupFilter::GetInstance();
   lPopupFilter.Delete(lFilterDisplayId, lFilterAppId);
   lPopupFilter.Dump();

   // Cancel running animations (if any) with new possible surface change request
   AnimationHandler::AbortAnimations();

   // Check and notify unfiltered popups first (before removing the filter from the filter container)
   PluginActions::NotifyPopupFilter();
   FocusHandler::GetInstance().UpdateInputFocus(0);

   // Perform visibility transformation (if available)
   AnimationHandler::TriggerAnimations();

   // Do a final all-in-one commit of all transactions towards layer manager
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   (void) IlmAccessor::Commit(lIlmError);
}


// ------------------------------------------------------------------------
void PopupManagerActivator::DumpPopupList(UInt32 popupScreenAreaId)
{
   // Retrieve instance of popup manager
   PopupManager& lPopupManager = PopupManager::GetInstance();

   UInt32 lPopupCount = lPopupManager.GetPopupCount(popupScreenAreaId);
   if (0 != lPopupCount)
   {
      ETG_TRACE_USR1(("@   ScreenArea %u: (popup count: %u)", popupScreenAreaId, lPopupCount));

      // Log current queued popups
      for (UInt i = 0; i < lPopupCount; ++i)
      {
         PopupState* lPopupState = lPopupManager.GetPopupAt(popupScreenAreaId, i);
         if (0 != lPopupState)
         {
            UInt32 lPassed = Time::Passed(lPopupState->GetNextUpdateTimeBase());
            ETG_TRACE_USR1(("@     Popup: %u/0x%08X (%40s)",
                            lPopupState->GetSurfaceId(),
                            lPopupState->GetUserData(),
                            lPopupState->IsVisible() ? "visible" : "not-visible"));
            ETG_TRACE_USR1(("@            time passed: %ums", lPassed));
            if (lPopupState->GetPresentationTime() > 0)
            {
               UInt32 lRemainingPresentationTime = (lPopupState->GetNextUpdatePeriod() - lPassed);
               ETG_TRACE_USR1(("@            time-out in: %ums", lRemainingPresentationTime));
            }
         }
      }
   }
}


// ------------------------------------------------------------------------
void PopupManagerActivator::DumpPopupList()
{
   ETG_TRACE_USR1(("@@@ Dump popup timing queue:"));

   // Retrieve instance of screen layouter
   IScreenLayouter* lScreenLayouter = ServiceApi::ScreenLayouter();

   // Dump popups for all popup areas
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::ScreenAreaList& lScreenAreaList = lScreenLayouter->GetScreenAreaList();
      for (IScreenLayouter::ScreenAreaList::const_iterator it = lScreenAreaList.begin();
            lScreenAreaList.end() != it;
            ++it)
      {
         const ScreenArea& lScreenArea = (*it);
         if (lScreenArea.IsPopupArea())
         {
            DumpPopupList(lScreenArea.GetId());
         }
      }
   }
   else
   {
      ETG_TRACE_SYS(("@ Popup dump failed: ScreenLayouter instance not accessible!"));
   }
}


// ------------------------------------------------------------------------
UInt32 PopupManagerActivator::PriorityInsert(PopupState& popupState)
{
   // Retrieve instance of popup manager
   PopupManager& lPopupManager = PopupManager::GetInstance();
   // Get popup screen area ID the given popup is related to
   UInt32 lPopupScreenAreaId = popupState.GetPopupScreenAreaId();

   // Insert popup list based on priority
   UInt32 lPopupPosition = cInvalidPosition;       // Needed for roll-back

   UInt32 lCount = lPopupManager.GetPopupCount(lPopupScreenAreaId);
   if (0 == lCount)
   {
      lPopupManager.AddPopupAt(popupState, 0);
   }
   else
   {
      UInt32 i = 0;
      for (; i < lCount; i++)
      {
         PopupState* lPopupState = lPopupManager.GetPopupAt(lPopupScreenAreaId, i);
         if ((0 != lPopupState) &&
               (popupState.GetPriority() < lPopupState->GetPriority()))
         {
            // Add new popup
            lPopupPosition = i;
            ETG_TRACE_USR4(("Add popup %u/0x%08X to popup screen area %u on position %d",
                            popupState.GetSurfaceId(),
                            popupState.GetUserData(),
                            lPopupScreenAreaId,
                            i));
            lPopupManager.AddPopupAt(popupState, i);
            break;
         }
      }
      if (i == lCount)
      {
         ETG_TRACE_USR4(("Add popup %u/0x%08X to popup screen area %u on position %d",
                         popupState.GetSurfaceId(),
                         popupState.GetUserData(),
                         lPopupScreenAreaId,
                         i));
         lPopupManager.AddPopupAt(popupState, i);
      }
   }
   return lPopupPosition;
}


// ------------------------------------------------------------------------
bool PopupManagerActivator::CloseOnSuperseded(PopupState*& supersededPopupState,
      const PopupState* topPopupState)
{
   bool lCloseOnSuperseded = false;
   if (0 != supersededPopupState)
   {
      // Access extended user data transported via requestId
      // closeOnSuperseded ... Bit 0
      ScreenBrokerProtocol::RequestId<ScreenBrokerProtocol::ShowPopupMode::Enum> lRequestId(supersededPopupState->GetRequestId());
      lCloseOnSuperseded = lRequestId.Test(ScreenBrokerProtocol::ShowPopupMode::CloseOnSuperseded);
      ETG_TRACE_USR1(("Close popup %u/0x%08X on superseded is %40sset",
                      supersededPopupState->GetSurfaceId(),
                      supersededPopupState->GetUserData(),
                      lCloseOnSuperseded ? "" : "not "));

      if (lCloseOnSuperseded)
      {
         // First remove reference in popup managers (timer) queue
         PopupManager::GetInstance().RemovePopup(*supersededPopupState);

         // Make a save copy for popup state for possibility to access data after removal of instance stored in active popups container
         PopupState lRemovedPopupState = *supersededPopupState;

         // Remove popup from active popups
         // (Attention: This also removes the instance!!!)
         (void) ActivePopups::GetInstance().Remove(supersededPopupState);
         supersededPopupState = 0;

         if (0 != topPopupState)
         {
            ETG_TRACE_USR1(("Close popup %u/0x%08X on superseded - top popup %u/0x%08X",
                            lRemovedPopupState.GetSurfaceId(),
                            lRemovedPopupState.GetUserData(),
                            topPopupState->GetSurfaceId(),
                            topPopupState->GetUserData()));
         }
         else
         {
            ETG_TRACE_USR1(("Close popup %u/0x%08X on superseded - no (visible) top popup",
                            lRemovedPopupState.GetSurfaceId(),
                            lRemovedPopupState.GetUserData()));
         }

         // Notify about unmapped popup
         bool lUpdateSurfaceVisibility = (0 != topPopupState) &&
                                         (lRemovedPopupState.GetSurfaceId() != topPopupState->GetSurfaceId()) &&
                                         lRemovedPopupState.IsVisible();
         bool lSendNotification = true;
         (void) PluginActions::NotifyPopupVisibility(&lRemovedPopupState,
               SurfaceState::Unmapped,
               lUpdateSurfaceVisibility,
               lSendNotification);
      }
   }
   return lCloseOnSuperseded;
}


// ------------------------------------------------------------------------
void PopupManagerActivator::SetInputFocus(UInt32 surfaceId,
      bool keyboardFocus,
      bool pointerFocus,
      Int32 priority,
      UInt32 focusData)
{
   ETG_TRACE_USR1(("SetInputFocus calls {%u, %d, %d, %d, %u}",
                   surfaceId,
                   keyboardFocus,
                   pointerFocus,
                   priority,
                   focusData));

   // Set focus priority independent of surface status
   switch (focusData)
   {
      case ScreenBrokerProtocol::InputFocus::Force:
         if (keyboardFocus)
         {
            FocusHandler::GetInstance().ForceKeyboardFocus(surfaceId, priority);
         }

         if (pointerFocus)
         {
            FocusHandler::GetInstance().ForcePointerFocus(surfaceId, priority);
         }
         break;
      case ScreenBrokerProtocol::InputFocus::Reset:
         if (keyboardFocus)
         {
            FocusHandler::GetInstance().ForceKeyboardFocus(0, 0);
         }

         if (pointerFocus)
         {
            FocusHandler::GetInstance().ForcePointerFocus(0, 0);
         }
         break;
      default:
         if (keyboardFocus)
         {
            FocusHandler::GetInstance().SetKeyboardPriority(surfaceId, priority);
         }

         if (pointerFocus)
         {
            FocusHandler::GetInstance().SetPointerPriority(surfaceId, priority);
         }
   }

   // Get instance of active surfaces container
   ActivePopups& lActivePopups = ActivePopups::GetInstance();

   PopupState* lPopupState = lActivePopups.GetOverallLast();
   while ((0 != lPopupState) && (lPopupState->GetSurfaceId() != surfaceId))
   {
      lPopupState = lActivePopups.GetOverallPrevious();
   }

   if ((0 != lPopupState) && (lPopupState->GetSurfaceId() == surfaceId))
   {
      if (keyboardFocus)
      {
         FocusHandler::GetInstance().SetKeyboardFocus(surfaceId);
      }
      if (pointerFocus)
      {
         FocusHandler::GetInstance().SetPointerFocus(surfaceId);
      }
      if (keyboardFocus || pointerFocus)
      {
         if (FocusHandler::GetInstance().UpdateInputFocus(surfaceId))
         {
            ilmErrorTypes lIlmError = ILM_SUCCESS;
            (void) IlmAccessor::Commit(lIlmError);
         }
      }
   }
}


// ------------------------------------------------------------------------
void PopupManagerActivator::DeregisterSurface(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("DeregisterSurface calls {%u}",
                   surfaceId));

   ActivePopups& lActivePopups = ActivePopups::GetInstance();

   PopupState* lPopupState = lActivePopups.GetOverallLast();
   while (0 != lPopupState)
   {
      if (lPopupState->GetSurfaceId() == surfaceId)
      {
         ServiceRequestArg lServiceRequestArg(0, lPopupState->GetAppId(), lPopupState->GetUserData(), "");   // Use dummy service request arg
         HidePopup(lServiceRequestArg, *lPopupState);
         lPopupState = lActivePopups.GetOverallLast();
      }
      else
      {
         lPopupState = lActivePopups.GetOverallPrevious();
      }
   }
}


}

// ========================================================================
// Global registration for plugin
// ========================================================================
// ------------------------------------------------------------------------
class PopupManagerActivatorPluginProxy
{
   public:
      PopupManagerActivatorPluginProxy()
      {
         // Register the plugin instance creation at the global factory map
         gScreenBrokerPluginFactoryMap[ScreenBroker::IPopupManagerActivator::Name()] = ScreenBroker::PopupManagerActivator::Create;
      }
};


// Create plugin instance of the proxy to perform registration
PopupManagerActivatorPluginProxy gPopupManagerActivatorPlugin;
