/* ***************************************************************************************
* FILE:          CriticalSection.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CriticalSection.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Util_CriticalSection_h)
#define ScreenBroker_Util_CriticalSection_h

// =============================================================================
// The intention of this class it to live only within a block scope. Furthermore
// it activates the resp. functionality with the constructor call and
// deactivates it at destruction time. For this reason this class has no other
// virtual functions than the destructor.
//lint -esym(1788, ScreenBroker::CriticalSectionLocker)   "Variable is referenced only by its constructor or destructor"
//lint -esym(1790, ScreenBroker::CriticalSectionLocker)   "Base class has no non-destructor virtual functions"
//lint -esym(1554, ScreenBroker::CriticalSectionLocker::mCs) "We want to copy the pointer"
//lint -esym(1555, ScreenBroker::CriticalSectionLocker::mCs) "We want to copy the pointer"
// Avoid usage of CriticalSectionLocker without CriticalSection
//lint -esym(1704, ScreenBroker::CriticalSectionLocker::CriticalSectionLocker) "Constructor has private access specification"
//To prevent copying of Critical Section
//lint -esym(1704, ScreenBroker::CriticalSection::CriticalSection) "Constructor has private access specification"
// =============================================================================

#include <ScreenBroker/ScreenBroker.h>
#include <pthread.h>

namespace ScreenBroker {
/// @addtogroup ScreenBrokerUtil
/// @{

/**
 * @brief A critical section is a synchronization mechanism to
 * protect a shared resource from being accessed and manipulated from
 * different thread contexts.
 *
 * When two or more threads need to access a shared resource at the same
 * time, synchronization is a coordination process to ensure that the
 * resource is used by only one thread at a time. A critical section is
 * a synchronization primitive that allows exclusive access to the shared
 * resource to only one thread. If the critical section lock of an object
 * is acquired by a thread, the second thread that wants to acquire the
 * lock is suspended until the lock is released by the first thread.
 *
 * Critical sections are per process for synchronization between threads
 * in one process. Critical section objects can be more efficient, as they
 * don't require a transition to kernel mode if there's no contention.
 *
 * Recursive critical section obtain calls within one thread context will be
 * granted. Means if the current thread already owns the critical section, and
 * an obtain is called the critical section will be entered.
 */
class CriticalSection
{
   public:
      /**
       * @brief Initializes the critical section.
       */
      CriticalSection();

      /**
       * @brief Destroys the critical section.
       */
      ~CriticalSection();

      /**
       * @brief Waits for the ownership of the critical section. The state of
       * the critical section is signaled if no thread owns it.
       * The method returns when the calling thread is granted
       * ownership.
       */
      void Obtain();

      /**
       * @brief Releases ownership of the critical section.
       */
      void Release();

      /**
       * @brief Attempts to enter the critical section without blocking.
       *
       * If the call is successful, the calling thread takes ownership
       * of the critical section.
       *
       * @return     <em>true</em>   If the critical section is successfully entered or
       *                             the current thread already owns the critical section.
       *             <em>false</em>  If another thread already owns the critical section.
       */
      bool Try();

   private:
      pthread_mutex_t mMutex;

      // Prevent copying
      CriticalSection(const CriticalSection&);
      CriticalSection& operator=(const CriticalSection&);
};


// ========================================================================
/**
 * @brief The CriticalSectionLocker class encapsulates the obtaining and
 * releasing of a critical section into an object.
 * Because the obtaining is done in the constructor and the releasing
 * in the destructor, it is assured that there is no obtain without its
 * corresponding release.
 */
class CriticalSectionLocker
{
   public:
      /**
       * @brief Waits for the ownership of the critical section (ctor).
       *
       * The state of the critical section is signaled if no thread owns it.
       * The method returns when the calling thread is granted ownership.
       *
       * @param cs Instance of critical section.
       */
      CriticalSectionLocker(CriticalSection* cs) : mCs(cs)
      {
         mCs->Obtain();
      }

      /**
       * @brief Obtains the critical section referenced by locker (copy ctor).
       *
       * @param locker Critical section locker instance.
       */
      CriticalSectionLocker(const CriticalSectionLocker& locker)
      {
         mCs = locker.mCs;
         mCs->Obtain();
      }

      /**
       * @brief Releases ownership of the critical section (dtor).
       */
      virtual ~CriticalSectionLocker()
      {
         if (0 != mCs)
         {
            mCs->Release();
            mCs = 0;
         }
      }

      /**
       * @brief Releases the currently held critical section and obtains
       * the one referenced by locker (assignment operator).
       *
       * @param locker Critical section locker instance to transfer lock.
       * @return Transferred lock.
       */
      CriticalSectionLocker& operator=(const CriticalSectionLocker& locker)
      {
         if ((this != & locker) && (0 != mCs) && (mCs != locker.mCs))
         {
            mCs->Release();
            mCs = locker.mCs;
            mCs->Obtain();
         }
         return *this;
      }

   private:
      /**
       * @brief CriticalSectionLocker
       */
      CriticalSectionLocker() : mCs(0) { }

      CriticalSection* mCs;
};


/// @}
}


#endif
