/* ***************************************************************************************
* FILE:          ServiceApi.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ServiceApi.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_ServiceApi_h)
#define ScreenBroker_Service_ServiceApi_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/KeyState.h>
#include <ScreenBroker/SurfaceState.h>
#include <ScreenBroker/LayerState.h>
#include <ScreenBroker/Service/ServiceRequestArg.h>
#include <ScreenBroker/Plugin/IKeyHandler.h>
#include <ScreenBroker/Plugin/IPopupManagerActivator.h>
#include <ScreenBroker/Plugin/IScreenBrokerActivator.h>
#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include <ScreenBroker/Plugin/IConfigurator.h>

namespace ScreenBroker {
/// @addtogroup ScreenBrokerService
/// @{

/**
 * @brief The ServiceApi class provides the service interface which is exposed for plugin development.
 */
class ServiceApi
{
   public:
      /**
       * @brief Registers the surface with the given ID to the given layer ID.
       *
       * @param surfaceId The ID of the surface to register.
       * @param layerId The layer ID where the surface shall be registered for.
       * @param userData Additional registration userData, which is provided as hint
       * in subsequent plugins calls (e.g. ScreenBrokerActivator::Activate).
       */
      static void RegisterSurfaceToLayer(UInt32 surfaceId,
                                         UInt32 layerId,
                                         UInt32 userData);

      /**
       * @brief Notifies clients that the state of the indicated surface has changed.
       *
       * @note The notification of the state change is triggered by the respective screen broker plugin
       * (ScreenbrokerActivator or PopupManagerActivator).
       *
       * @param userData User data which may give additional information from plugin to application.
       * @param surfaceId Surface ID which this state change applies to.
       * @param surfaceState New state of surface.
       */
      static void NotifySurfaceStateChange(UInt32 userData,
                                           UInt32 surfaceId,
                                           SurfaceState::Enum surfaceState);

      /**
       * @brief Notifies clients that a runtime error has occurred with given request.
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request returned in the error response.
       * @param systemErrno System error codes (like linux extended error codes defined in linux/errno.h).
       * @param presentationErrno Error codes specific to the visual presentation module screen broker
       * is connected to (e.g. layer manager).
       */
      static void NotifyError(const ServiceRequestArg& serviceRequestArg,
                              Int32 systemErrno,
                              Int32 presentationErrno);

      /**
       * @brief Notifies a key event to a client which is holding the focused surface (normal or popup).
       *
       * @param userData User data which may give additional information from plugin to application.
       * @param surfaceId ID of focused surface to which the key event shall be sent.
       * @param keyCode 32bit code of the key event.
       * @param keyState The state of the key event.
       */
      static void NotifyKeyEvent(UInt32 userData,
                                 UInt32 surfaceId,
                                 UInt32 keyCode,
                                 KeyState::Enum keyState);

      /**
       * @brief Notifies about current status details based on the request.
       *
       * @note The notification is sent to the owner of the surface defined via surfaceId.
       *
       * @param requestId 32bit information to inform about details of the requests nature.
       * @param surfaceId The ID of the surface, which identifies the response path, may also be part of the status response.
       * @param status 32bit of response data carrying the status information.
       * @param userData Additional 32bit of user data for whatever use.
       */
      static void NotifyCurrentStatus(UInt32 requestId,
                                      UInt32 surfaceId,
                                      UInt32 status,
                                      UInt32 userData);

      /**
       * @brief Notifies about Layer State details based on the update.
       *
       * @note The notification is sent to the owner of the first valid attached surface.
       *
       * @param requestId 32bit information to inform about details of the requests nature.
       * @param layerId layer ID which this state change applies to.
       * @param surfaceState New state of surface.
       */
      static void NotifyLayerStateChanged(UInt32 requestId,
                                          UInt32 layerId,
                                          LayerState::Enum state);

      /**
       * @brief Notifies clients that close on external touch is enabled or disabled.
       * @param status Enabled or Disabled.
       */
      static void NotifyCloseOnExternalTouch(bool status);

      /**
       * @brief Add information about build-in layers/surfaces
       *
       * @param layerid
       * @param surfaceId
       * @param screenId
       * @param connectorName
       */
      static void AddBuildInLayerInformation(UInt32 layerId, UInt32 surfaceId, UInt32 screenId, UInt32 width, UInt32 height, std::string connectorName = "");

      /**
       * @brief Provides access to plugin KeyHandler.
       * @return Instance of plugin if available, 0 otherwise.
       */
      static IKeyHandler* KeyHandler();

      /**
       * @brief Provides access to plugin PopupManagerActivator.
       * @return Instance of plugin if available, 0 otherwise.
       */
      static IPopupManagerActivator* PopupManagerActivator();

      /**
       * @brief Provides access to plugin ScreenBrokerActivator.
       * @return Instance of plugin if available, 0 otherwise.
       */
      static IScreenBrokerActivator* ScreenBrokerActivator();

      /**
       * @brief Provides access to plugin ScreenLayouter.
       * @return Instance of plugin if available, 0 otherwise.
       */
      static IScreenLayouter* ScreenLayouter();

      /**
       * @brief Provides access to plugin Configurator.
       * @return Instance of plugin if available, 0 otherwise.
       */
      static IConfigurator* Configurator();

      /**
       * @brief Informs if ILM access was established and can be used.
       * @return True if ILM access can be used, false otherwise.
       */
      static bool HasIlmAccess();
};


/// @}
}


#endif
