/* ***************************************************************************************
* FILE:          Service.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Service.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_Service_h)
#define ScreenBroker_Service_Service_h

// =============================================================================
// This class is intended to be a singleton, for this reason it is ok, that the
// constructor and the copy constructor have only private access.
//lint -esym(1704, ScreenBroker::Service::Service)   "Constructor has private access specification"
// =============================================================================

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <string>

namespace ScreenBroker {
class ServiceApi;
namespace Internal {
/// @addtogroup ScreenBrokerService
/// @{

/// Reset modes for screen broker service
namespace ResetMode {
enum Enum
{
   /// Comprises reset call to ScreenBrokerActivatorPlugin and PopupManagerActivatorPlugin
   ActivatorPlugins,

//            /// Comprises reset call to ScreenLayouterPlugin
//            ScreenLayouterPlugin,

//            /// Comprises reset call to all plugins
//            Plugins,

   /**
    * Does a reset call to all instances of surfaces (incl. popups).
    * @note Triggers additionally ResetMode::ActivatorPlugins (for consistency).
    */
   SurfaceInstances,

   /**
    * Does a reset call to all client instances.
    * @note Triggers additionally ResetMode::SurfaceInstances (for consistency).
    */
   ClientInstances

//            /**
//             * Does a full reset of the internal state.
//             * @note No dbus reset is performed.
//             */
//            Full
};


}

/**
 * @brief The main interface to start the screen broker service.
 */
class Service
{
      friend class ScreenBroker::ServiceApi;
   public:
      /**
       * @brief Get the singleton instance of the screen broker service.
       *
       * @return Screen broker service instance.
       */
      static Service& GetInstance();

      ~Service();

      /**
       * @brief Inititializes the screen broker service infrastructure like setting up transport layer
       * to client (e.g. DBus) and to layer manager respectively loading plugins.
       *
       * @brief ignoreIlm Define if ILM communication shall be established or not (mainly used for testing).
       * @return True if successful, false otherwise.
       */
      bool Init(bool establishIlmAccess = true);

      /**
       * @brief Informs if ILM access was established and can be used.
       *
       * @return True if ILM access can be used, false otherwise.
       */
      bool HasIlmAccess() const
      {
         return mIlmAccess;
      }

      /**
       * @brief Triggers a reset the screen broker service internal states based on the given reset level
       * without restarting the service.
       *
       * @param mode Level of reset.
       */
      void Reset(ResetMode::Enum mode);

      /**
       * @brief Triggers a diagnosis call towards plugins for e.g. dumping current states, etc.
       */
      void Diagnosis();

      /**
       * @brief Starts the screen broker service.
       *
       * @note This interface block the current execution, so it is recommended to start the service either
       * within a the main loop, when having a dedicated executable (aka. daemon) or in a
       * separate thread context, when applied as a plugin to an existing service instance.
       */
      void Run();

      /**
       * @brief Requests the unique service ID (normally provided by the respective transport layer instance).
       *
       * @return Unique service ID.
       */
      const Char* ServiceId() const;

      /**
       * @brief Destroys the screen broker service instance while performing a graceful shutdown by
       * terminating the plugins and clean up the requested resources by layer manager.
       */
      void Destroy();

   private:
      Internal::ServiceAdaptor* mServiceAdaptor;
      bool mIlmAccess;

      /// Secure initialization and shutdown of service
      CriticalSection mCs;

      ///
      Service();

      // Dis-allow copying of singleton
      Service(const Service&);
      Service& operator=(const Service&);

      /**
       * @brief Request the service adaptor interface to access client data and communication interface.
       * @return Screen broker service adaptor interface.
       */
      Internal::ServiceAdaptor* GetServiceAdaptor()
      {
         return mServiceAdaptor;
      }
};


/// @}
}


}

#endif
