/* ***************************************************************************************
* FILE:          Service.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Service.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Connection.h>
#include <ScreenBroker/Service/PopupManager/PopupManager.h>
#include "Service.h"
#include "LayerManagerAccessor.h"
#include "PluginManager.h"

#ifdef VARIANT_S_FTR_ENABLE_USE_CODE_COVERAGE
#include "CodeCoverage.hpp"
#endif


#include "ServiceAdaptor.h"
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/Service.cpp.trc.h"
#endif


namespace ScreenBroker {
namespace Internal {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenBrokerService);

// ------------------------------------------------------------------------
Service& Service::GetInstance()
{
   // Lazy initialization of screen broker service singleton
   static Service lService;
   return lService;
}


// ------------------------------------------------------------------------
Service::Service() : mServiceAdaptor(0), mIlmAccess(false)
{
   ETG_TRACE_USR4(("Service created"));
}


// ------------------------------------------------------------------------
Service::~Service()
{
   ETG_TRACE_USR4(("Service destroyed"));
   mServiceAdaptor = 0;
}


// ------------------------------------------------------------------------
bool Service::Init(bool establishIlmAccess)
{
   //SCREENBROKER_LOG_FN();
   bool lRc = true;
   CriticalSectionLocker lLock(&mCs);

#ifdef VARIANT_S_FTR_ENABLE_USE_CODE_COVERAGE
   hmibase::codecoverage::set_signalhandler();
   hmibase::codecoverage::code_coverage_init("/var/opt/bosch/dynamic/hmi/gcov/screenbroker/", "5");
#endif

   // ----------
   // Initialize layer manager accessor
   if (establishIlmAccess)
   {
      lRc = Internal::LayerManagerAccessor::Init();
      mIlmAccess = lRc;
   }

   // ----------
   // Initialize popup manager
   lRc = lRc && PopupManager::GetInstance().Init();

   // ----------
   // Initialize DBus connection and service
   if (lRc)
   {
      lRc = (0 == mServiceAdaptor);
      if (lRc)
      {
         ETG_TRACE_USR4(("Service initializing version %40s",
                         SCREENBROKER_VERSION_STRING));

         mServiceAdaptor = Internal::ServiceAdaptor::GetInstance();
         lRc = mServiceAdaptor != 0 && mServiceAdaptor->Init();
      }
      else
      {
         ETG_TRACE_SYS(("Service already initialized!"));
      }
   }

   // ----------
   // Loading screen broker plugins
   lRc = lRc && Internal::PluginManager::GetInstance().Load();

   return lRc;
}


// ------------------------------------------------------------------------
void Service::Reset(ResetMode::Enum mode)
{
   //SCREENBROKER_LOG_FN();
   if (0 != mServiceAdaptor)
   {
      ETG_TRACE_USR1(("Trigger safe reset with mode %d!", mode));
      mServiceAdaptor->Reset(mode);
   }
   else
   {
      ETG_TRACE_SYS(("Service not running!"));
   }
}


// ------------------------------------------------------------------------
void Service::Diagnosis()
{
   //SCREENBROKER_LOG_FN();
   if (0 != mServiceAdaptor)
   {
      ETG_TRACE_USR1(("Trigger safe diagnosis call towards plugins!"));
      mServiceAdaptor->Diagnosis();
   }
   else
   {
      ETG_TRACE_SYS(("Service not running!"));
   }
}


// ------------------------------------------------------------------------
const Char* Service::ServiceId() const
{
   return (mServiceAdaptor != 0) ? mServiceAdaptor->GetServiceId() : "<uninitialized>";
}


// ------------------------------------------------------------------------
void Service::Run()
{
   //SCREENBROKER_LOG_FN();
   if (0 != mServiceAdaptor)
   {
      ETG_TRACE_USR1(("Service started"));
      mServiceAdaptor->Run();
      ETG_TRACE_USR1(("Service ended"));
   }
   else
   {
      ETG_TRACE_FATAL(("Service not initialized! Terminate.."));
   }
}


// ------------------------------------------------------------------------
void Service::Destroy()
{
   //SCREENBROKER_LOG_FN();
   CriticalSectionLocker lLock(&mCs);

   // Stop DBus service and terminate connection
   if (0 != mServiceAdaptor)
   {
      ETG_TRACE_USR4(("Service terminating"));
      mServiceAdaptor->Terminate();
      mServiceAdaptor = 0;
   }
   else
   {
      ETG_TRACE_SYS(("Service not running!"));
   }

   // Unload screen broker plugins
   // (unload plugins before destroying layer manager accessor,
   // as plugins may still access layer manager service)
   Internal::PluginManager::GetInstance().Unload();

   // Terminate the popup manager
   PopupManager::GetInstance().Destroy();

   // Terminate the layer manager accessor
   Internal::LayerManagerAccessor::Destroy();
}


} //namespae
}
