/* ***************************************************************************************
* FILE:          ScreenArea.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenArea.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_ScreenArea_h)
#define ScreenBroker_Service_ScreenArea_h

#include <ScreenBroker/ScreenBroker.h>
#include <string>
#include <map>
#include <list>

namespace ScreenBroker {
class PopupState;
/// @addtogroup ScreenBrokerService
/// @{

/**
 * @brief The ScreenArea class holds property data of a screen area and also exposes some popup relevant
 * manipulation interfaces.
 *
 * Screen areas itself are created by the plugin ScreenLayouter and propagated to the ScreenBroker.
 */
class ScreenArea
{
   public:
      ///
      ScreenArea();
      /// Update Secondary display configurations
      void ReInit(const ScreenArea& newObj);
      /**
       * @brief Initializes the screen area with its properties
       *
       * @param id Screen area ID.
       * @param width Screen area width.
       * @param height Screen area height.
       * @param xPos X position of screen area on display.
       * @param yPos Y position of screen area on display
       * @param isPopupArea True if screen area is used for popups, false otherwise.
       * @param displayId Display ID which the screen area is linked to.
       * @param isManaged True if screen area is managed, false otherwise.
       * @param isVisible True if screen area is visible, false otherwise.
       * @return True if initialization was successful, false otherwise.
       */
      bool Init(UInt32 id,
                UInt32 width,
                UInt32 height,
                UInt32 xPos,
                UInt32 yPos,
                bool isPopupArea,
                UInt32 displayId,
                UInt32 displayAlias,
                bool isManaged,
                bool IsVisible,
                bool isNotifyOnPopupStatus);

      /**
       * @brief Resets the screen area object.
       */
      void Reset();

      /**
       * @brief In the destructor of the screen area all layers which are linked to this screen area are
       * automatically requested to be deleted at ILM.
       */
      virtual ~ScreenArea();

      /**
       * @brief Provides the ID of the screen area.
       *
       * @note A screen area ID 0 marks the screen area as invalid.
       *
       * @return ID of screen area.
       */
      UInt32 GetId() const
      {
         return mId;
      }

      /**
       * @brief Provides the position of the lower left corner of the screen area in horizontal
       * direction on display.
       *
       * @return X position of screen area on display.
       */
      UInt32 GetXPos() const
      {
         return mXPos;
      }

      /**
       * @brief Provides the position of the lower left corner of the screen area in vertical
       * direction on display.
       *
       * @return Y position of screen area on display.
       */
      UInt32 GetYPos() const
      {
         return mYPos;
      }

      /**
       * @brief Provides the width of the screen area on screen.
       *
       * @return Width of screen area.
       */
      UInt32 GetWidth() const
      {
         return mWidth;
      }

      /**
       * @brief Provides the height of the screen area on screen.
       *
       * @return Height of screen area.
       */
      UInt32 GetHeight() const
      {
         return mHeight;
      }

      /**
       * @brief Provides the main display ID which the screen area corresponds to.
       *
       * @return Display ID which the screen area is linked to.
       */
      UInt32 GetDisplayId() const
      {
         return mDisplayId;
      }

      /**
       * @brief Provides the main display alias which the screen area corresponds to.
       *
       * @return Display Alias which the screen area is linked to.
       */
      UInt32 GetDisplayAlias() const
      {
         return mDisplayAlias;
      }

      /**
       * @brief Provides the ILM Layer ID used by the screen area for the client application
       * addressed by the given client ID.
       *
       * @return ILM layer ID which corresponds to this screen area object.
       */
      UInt32 GetLayerId() const
      {
         return mLayerId;
      }

      /**
       * @brief Checks if screen area dedicated to popups.
       *
       * @return True if screen area is a popup area, false otherwise.
       */
      bool IsPopupArea() const
      {
         return mIsPopupArea;
      }

      /**
       * @brief Checks if screen area is managed (means it referenced layer
       * will be created and destroyed in the respective methods).
       *
       * @return True if screen area is managed, false otherwise.
       */
      bool IsManaged() const
      {
         return mIsManaged;
      }

      /**
       * @brief Checks if screen area is visible.
       * @return True if screen area is visible, false otherwise.
       */
      bool IsVisible() const
      {
         return mIsVisible;
      }

      /**
       * @brief Checks if notify popup FG/BG status for screenArea.
       * @return True if notify is set to true, false otherwise.
       */
      bool IsNotifyOnPopupStatus() const
      {
         return mIsNotifyOnPopupStatus;
      }

      /**
       * Creates an IVI layer using ILM as required.
       *
       * @return True if initialization was successful, false otherwise.
       */
      virtual bool InitializeLayering();

   protected:
      ///
      void SetId(UInt32 id)
      {
         mId = id;
      }

      ///
      void SetXPos(UInt32 xPos)
      {
         mXPos = xPos;
      }

      ///
      void SetYPos(UInt32 yPos)
      {
         mYPos = yPos;
      }

      ///
      void SetWidth(UInt32 width)
      {
         mWidth = width;
      }

      ///
      void SetHeight(UInt32 height)
      {
         mHeight = height;
      }

      ///
      void SetDisplayId(UInt32 displayId)
      {
         mDisplayId = displayId;
      }

      ///
      void SetDisplayAlias(UInt32 displayAlias)
      {
         mDisplayAlias = displayAlias;
      }

      ///
      void SetPopupArea(bool isPopupArea)
      {
         mIsPopupArea = isPopupArea;
      }

      ///
      void SetManaged(bool isManaged)
      {
         mIsManaged = isManaged;
      }

      ///
      void SetVisible(bool isVisible)
      {
         mIsVisible = isVisible;
      }

      ///
      void SetNotifyOnPopupStatus(bool isNotifyOnPopupStatus)
      {
         mIsNotifyOnPopupStatus = isNotifyOnPopupStatus;
      }

   private:
      UInt32 mId;
      UInt32 mXPos;
      UInt32 mYPos;
      UInt32 mWidth;
      UInt32 mHeight;
      UInt32 mDisplayId;
      UInt32 mDisplayAlias;
      UInt32 mLayerId;
      bool mIsPopupArea;
      bool mIsManaged;
      bool mIsVisible;
      bool mIsNotifyOnPopupStatus;
};


/// @}
}


#endif
