/* ***************************************************************************************
* FILE:          PopupState.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupState.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_PopupManager_PopupState_h)
#define ScreenBroker_Service_PopupManager_PopupState_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/RequestArg.h>
#include <set>

namespace ScreenBroker {
/// @addtogroup PopupManager
/// @{

/**
 * @brief The PopupState class encapsulates the internal state and configuration of pop-up overlays.
 */
class PopupState
{
      friend class Internal::PopupArea;
      friend class Internal::ClientState;
      static const UInt32 InvalidId = static_cast<UInt32>(-1);
   public:
      /**
       * @brief PopupState constructor initialized with the surface, where this popup is presented.
       *
       * @param surfaceId The surface ID of the popup
       */
      PopupState(UInt32 surfaceId = InvalidId);

      ///
      virtual ~PopupState();

      /// The surface ID of the popup.
      UInt32 GetSurfaceId() const
      {
         return mSurfaceId;
      }

      /// The ID of screen area this popup is related to.
      UInt32 GetPopupScreenAreaId() const
      {
         return mPopupScreenAreaId;
      }

      /// The layer where the surface of the popup is located.
      UInt32 GetLayerId() const
      {
         return mLayerId;
      }

      /// The presentation period of the popup.
      UInt32 GetPresentationTime() const
      {
         return mPopupPresentationArg.PresentationTime();
      }

      /// The minimum presentation period of the popup.
      UInt32 GetMinimumPresentationTime() const
      {
         return mPopupPresentationArg.MinimumPresentationTime();
      }

      /// The period how long this popup is valid to be presented.
      UInt32 GetValidityPeriod() const
      {
         return mPopupPresentationArg.ValidityPeriod();
      }

      /// The priority of the popup.
      UInt32 GetPriority() const
      {
         return mPopupPresentationArg.Priority();
      }

      /// The modality of the popup's surface.
      Modality::Enum GetModality() const
      {
         return mPopupPresentationArg.Modality();
      }

      /// The optional horizontal alignment relative to the layer.
      HorizontalAlignment::Enum GetHorizontalAlignment() const
      {
         return mPopupPresentationArg.HorizontalAlignment();
      }

      /// The optional vertical alignment relative to the layer.
      VerticalAlignment::Enum GetVerticalAlignment() const
      {
         return mPopupPresentationArg.VerticalAlignment();
      }

      /// The scaling attribute of the pop-up relative to the layer.
      Scaling::Enum GetScaling() const
      {
         return mPopupPresentationArg.Scaling();
      }

      /// The focus priority of the pop-up surface
      Int32 GetFocusPriority() const
      {
         return mPopupPresentationArg.FocusPriority();
      }

      /// The user data of the popup coming from the ShowPopup() request.
      UInt32 GetUserData() const
      {
         return mUserData;
      }

      /// The application ID of the popup coming from the ShowPopup() request.
      UInt32 GetAppId() const
      {
         return mAppId;
      }

      /// The request ID of the popup coming from the ShowPopup() request.
      UInt32 GetRequestId() const
      {
         return mRequestId;
      }

      /**
       * @brief Informs about the visibilty property of popup state.
       *
       * @return True, if popup is visible, false otherwise.
       */
      bool IsVisible() const
      {
         return mVisible;
      }

      /**
       * @brief Changes the visibility property of popup state.
       *
       * @param visible True, if popup is set visible, false otherwise.
       */
      void SetVisible(bool visible)
      {
         mVisible = visible;
      }

      /**
       * @brief Get the next timer update period of this popup state or 0 to deactivate time-based updates.
       *
       * @return The period of time before a next update for this popup is triggered.
       */
      UInt32 GetNextUpdatePeriod() const
      {
         return mNextUpdatePeriod;
      }

      /**
       * @brief Set the next timer update period of this popup state or 0 to deactivate time-based updates.
       *
       * @note By setting this next update period, the next update time based is updated to
       * the current timestamp internally.
       *
       * @param nextUpdatePeriod The timer period before a next update for this popup is triggered.
       */
      void SetNextUpdatePeriod(UInt32 nextUpdatePeriod);

      /**
       * @brief Retrieve the timestamp from which on the next popup update is calculated.
       *
       * @return Time base for the next update period.
       */
      UInt64 GetNextUpdateTimeBase() const
      {
         return mNextUpdateTimeBase;
      }

   protected:
      /**
       * Applies the provided popup properties coming from show popup request.
       *
       * @param popupScreenAreaId ID of popup screen area, this popup is related to.
       * @param layerId ID of layer, this popup is related to.
       * @param popupPresentationArg Popup presentation arguments to apply to popup state.
       * @param requestArg Information of the request arguments (user data, request ID, app ID) to apply to popup state.
       */
      virtual void ApplyProperties(UInt32 popupScreenAreaId,
                                   UInt32 layerId,
                                   const PopupPresentationArg& popupPresentationArg,
                                   const RequestArg& requestArg);

   private:
      /// The ID of the pop-up surface.
      UInt32 mSurfaceId;

      /// The ID of popup screen area, this popup is related to.
      UInt32 mPopupScreenAreaId;

      /// The ID of layer, this popup resides.
      UInt32 mLayerId;

      /// The properties of the respective popup.
      PopupPresentationArg mPopupPresentationArg;

      /// User data given with the show popup request.
      UInt32 mUserData;

      /// Application ID given with the show popup request.
      UInt32 mAppId;

      /// Request ID given with the show popup request (can be used as additional user data).
      UInt32 mRequestId;

      /// Defines whether popup is visible or not.
      bool mVisible;

      /**
       * Timer period before a next update for this popup is triggered.
       *
       * @note: The separation of a "future time" into base time (mNextUpdateTimeBase)
       * and time period (mNextUpdatePeriod) avoids time overflow issues.
       */
      UInt32 mNextUpdatePeriod;

      /**
       * Timestamp from which on the next popup update is calculated.
       *
       * @note: The separation of a "future time" into base time (mNextUpdateTimeBase)
       * and time period (mNextUpdatePeriod) avoids time overflow issues.
       */
      UInt64 mNextUpdateTimeBase;
};


/// @}
}


#endif
