/* ***************************************************************************************
* FILE:          PopupManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupManager.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// This class is intended to be a singleton, for this reason it is ok, that the
// constructor and the copy constructor have only private access.
//lint -esym(1704, ScreenBroker::PopupManager::PopupManager)   "Constructor has private access specification"
// =============================================================================


#if !defined(ScreenBroker_Service_PopupManager_PopupManager_h)
#define ScreenBroker_Service_PopupManager_PopupManager_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/Service/ScreenArea.h>
#include <ScreenBroker/Service/PopupManager/PopupArea.h>
#include <ScreenBroker/Service/PopupManager/PopupState.h>
#include <ScreenBroker/Service/Service.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <map>

namespace ScreenBroker {
class ServiceRequestArg;

/// @addtogroup PopupManager
/// @{

/**
 * @brief The PopupManager class is internal a central component for handling
 * popup requests and additionally provides a public interface for managing
 * the popup queues per popup screen area on the plugin side (PopupManagerActivator).
 */
class PopupManager
{
      friend class Internal::ServiceAdaptor;
      friend class Internal::Service;
   public:
      /**
       * @brief GetInstance provides the singleton instance of popup manager.
       *
       * @return The singleton instance of popup manager.
       */
      static PopupManager& GetInstance();

      ///
      ~PopupManager();

      /**
       * @brief Resets the internal status of popup manager (timer and popup list).
       */
      void Reset();

      /**
       * @brief Informs about the number of popup states currently held in given popup area.
       *
       * @param popupScreenAreaId Popup screen area ID.
       *
       * @return Number of active popup states for this popup area.
       */
      UInt32 GetPopupCount(UInt32 popupScreenAreaId);

      /**
       * @brief Enqueues the given popup state at the given position to popup state's popup area.
       *
       * If the position is less than 0 it is enqueued at the beginning,
       * if the position exceeds the length of the queue it is enqueued at the end of the list.
       *
       * @note The popup screen area ID to which this popup shall be added is part of the popup
       * state object.
       *
       * @param popupState Popup state to add.
       * @param position Position where the popup has to be added.
       */
      void AddPopupAt(PopupState& popupState,
                      UInt32 position);

      /**
       * @brief Enqueues the given popup state at the end to popup state's popup area.
       *
       * @note The popup screen area ID to which this popup shall be added is part of the popup
       * state object.
       *
       * @param popupState Popup state to add.
       */
      void AddPopup(PopupState& popupState);

      /**
       * @brief Provides the popup state at the given position from popup list of the given popup area.
       *
       * @param popupScreenAreaId Popup screen area ID.
       * @param position Requested position.
       * @return The requested popup state object at given position.
       */
      PopupState* GetPopupAt(UInt32 popupScreenAreaId,
                             UInt32 position);

      /**
       * @brief Searches for the popup state in the popup list of popup state's popup area.
       *
       * @param popupState Popup state to search.
       * @param position Position of the popup state found in the popup list of of popup state's popup area
       *                 (if position pointer is 0, then this out parameter is ignored).
       * @return The requested popup state found in the popup list of of popup state's popup area or
       *         0 if not found (in this case position returned is not valid).
       */
      PopupState* FindPopup(PopupState& popupState, UInt32* position);

      /**
       * @brief Removes the popup state at the given position from popup list of the given popup area.
       *
       * @param popupScreenAreaId Popup screen area ID.
       * @param position Position of the popup to remove.
       */
      void RemovePopupAt(UInt32 popupScreenAreaId,
                         UInt32 position);

      /**
       * @brief Removes the given popup state from popup state's popup area.
       *
       * @note The popup screen area ID to which this popup shall be removed is part of the popup
       * state object.
       *
       * @param popupState Popup state to remove.
       */
      void RemovePopup(PopupState& popupState);

      /**
       * @brief Provides the first popup state with a timestamp less than the given timestamp
       * or null if no further popup is available.
       *
       * @param timestamp current timestamp.
       * @return First expired popup state.
       */
      PopupState* GetFirstExpiredPopupState(UInt32 timestamp);

      /**
       * @brief Provides the next popup state with a timestamp less than the given timestamp or
       * null if no further popup is available.
       *
       * @param timestamp Current timestamp.
       * @return Next expired popup state.
       */
      PopupState* GetNextExpiredPopupState(UInt32 timestamp);

      /**
       * @brief Calculates the timestamp for the next time event and sets the system timer accordingly.
       */
      void ResetTimer();

   private:
      typedef std::map<UInt32, Internal::PopupArea> PopupAreaMap;

      ///
      static Int mTimerFd;

      /// Map of popup screen areas
      PopupAreaMap mPopupAreaMap;

      ///
      pthread_t mTimerThread;

      ///
      bool mTimerActive;

      ///
      PopupManager();

      // Dis-allow copying of singleton
      PopupManager(const PopupManager&);
      PopupManager& operator=(const PopupManager&);

      /**
       * @brief Inititializes the popup manager.
       *
       * @return True if initialization was successful, false otherwise.
       */
      bool Init();

      /**
       * @brief Destroys the internal popup manager states.
       */
      void Destroy();

      /**
       * @brief Forwards the request to the PopupManagerActivator plugin, which then may set various
       * popup properties (e.g. visibility, keyboard focus) resp. arbitrate the popups towards ILM.
       *
       * @param clientState The client the popup surface is registered to.
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID of the popup to show.
       * @param popupPresetationArg Various popup presentation properties.
       */
      void ShowPopup(Internal::ClientState* clientState,
                     const ServiceRequestArg& serviceRequestArg,
                     UInt32 surfaceId,
                     const PopupPresentationArg& popupPresentationArg) const;

      /**
       * @brief Forwards the request to the PopupManagerActivator plugin, which then may set various
       * popup properties (e.g. visibility, keyboard focus) resp. arbitrate the popups towards ILM.
       *
       * @param clientState The client the popup surface is registered to.
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID of the popup to hide.
       */
      void HidePopup(Internal::ClientState* clientState,
                     const ServiceRequestArg& serviceRequestArg,
                     UInt32 surfaceId) const;

      /**
       * @brief Provides popup area by given screen area ID.
       *
       * @note If popup area with screen area ID is not found in list, then a new entry with given screen area ID is created.
       *
       * @param screenAreaId Screen area ID to search the corresponding popup area for.
       * @return Popup area corresponding to given screen area ID.
       */
      Internal::PopupArea& GetPopupArea(UInt32 popupScreenAreaId);

      /**
       * Triggers the time-based update of pop-up states via activator plug-in.
       *
       * @param timestamp Timestamp of last timer expiration.
       */
      void UpdatePopupStates(UInt64 timestamp);

      /**
       * @brief Start the popup timer with given expiration time
       * @param expirationPeriod Time period (in ms), when the popup timer shall expire next.
       */
      void StartTimer(Int32 expirationPeriod) const;

      /**
       * @brief Stops the popup timer (if running)
       */
      void StopTimer() const;

      ///
      bool CreateTimer();

      ///
      void TerminateTimer();

      ///
      static void* TimerFn(void* arg);
};


/// @}
}


#endif
