/* ***************************************************************************************
* FILE:          PopupManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupManager.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "PopupManager.h"
#include <ScreenBroker/Service/ClientState.h>
#include <ScreenBroker/Service/PluginManager.h>

#include <ScreenBroker/Service/ServiceRequestArg.h>
#include <ScreenBroker/Service/Synchronize.h>
#include <ScreenBroker/Util/Time.h>
#include <fcntl.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/timerfd.h>
#include <algorithm>
#include <limits>
#include <unistd.h>
#include <errno.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/PopupManager.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::PopupManager);

// Initialization of static variables
Int PopupManager::mTimerFd = 0;

// ------------------------------------------------------------------------
PopupManager& PopupManager::GetInstance()
{
   // Lazy initialization of popup manager singleton
   static PopupManager lPopupManager;
   return lPopupManager;
}


// ------------------------------------------------------------------------
PopupManager::PopupManager() : mTimerThread(0), mTimerActive(false)
{
}


// ------------------------------------------------------------------------
PopupManager::~PopupManager()
{
}


// ------------------------------------------------------------------------
void PopupManager::Reset()
{
   //SCREENBROKER_LOG_FN();
   StopTimer();
   for (PopupAreaMap::iterator it = mPopupAreaMap.begin();
         mPopupAreaMap.end().operator != (it);
         ++it)
   {
      Internal::PopupArea& lPopupArea = (*it).second;
      lPopupArea.Reset();
   }
   mPopupAreaMap.clear();
}


// ------------------------------------------------------------------------
bool PopupManager::Init()
{
   //SCREENBROKER_LOG_FN();
   return CreateTimer();
}


// ------------------------------------------------------------------------
void PopupManager::Destroy()
{
   //SCREENBROKER_LOG_FN();
   StopTimer();
   close(mTimerFd);
   TerminateTimer();
}


// ------------------------------------------------------------------------
void PopupManager::ShowPopup(Internal::ClientState* clientState,
                             const ServiceRequestArg& serviceRequestArg,
                             UInt32 surfaceId,
                             const PopupPresentationArg& popupPresentationArg) const
{
   //SCREENBROKER_LOG_FN();
   // Check if client state is valid and show popup
   if (0 != clientState)
   {
      if (clientState->GetClientId() == serviceRequestArg.ClientId())
      {
         clientState->ShowPopup(serviceRequestArg,
                                surfaceId,
                                popupPresentationArg);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Show popup with surface ID:%u failed, surface is registered to client %40s!",
                        serviceRequestArg.ClientId().c_str(),
                        surfaceId,
                        clientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': Show popup with surface ID:%u failed, surface is not registered!",
                     serviceRequestArg.ClientId().c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void PopupManager::HidePopup(Internal::ClientState* clientState,
                             const ServiceRequestArg& serviceRequestArg,
                             UInt32 surfaceId) const
{
   //SCREENBROKER_LOG_FN();
   // Check if client state is valid and show popup
   if (0 != clientState)
   {
      if (clientState->GetClientId() == serviceRequestArg.ClientId())
      {
         clientState->HidePopup(serviceRequestArg,
                                surfaceId);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Hide popup with surface ID:%u failed, surface is registered to client %40s!",
                        serviceRequestArg.ClientId().c_str(),
                        surfaceId,
                        clientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': Hide popup with surface ID:%u failed, surface is not registered!",
                     serviceRequestArg.ClientId().c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
UInt32 PopupManager::GetPopupCount(UInt32 popupScreenAreaId)
{
   return GetPopupArea(popupScreenAreaId).GetPopupCount();
}


// ------------------------------------------------------------------------
void PopupManager::AddPopupAt(PopupState& popupState,
                              UInt32 position)
{
   GetPopupArea(popupState.GetPopupScreenAreaId()).AddPopupAt(popupState, position);
   ResetTimer();
}


// ------------------------------------------------------------------------
void PopupManager::AddPopup(PopupState& popupState)
{
   GetPopupArea(popupState.GetPopupScreenAreaId()).AddPopup(popupState);
   ResetTimer();
}


// ------------------------------------------------------------------------
PopupState* PopupManager::GetPopupAt(UInt32 popupScreenAreaId,
                                     UInt32 position)
{
   return GetPopupArea(popupScreenAreaId).GetPopupAt(position);
}


// ------------------------------------------------------------------------
PopupState* PopupManager::FindPopup(PopupState& popupState,
                                    UInt32* position)
{
   return GetPopupArea(popupState.GetPopupScreenAreaId()).FindPopup(popupState, position);
}


// ------------------------------------------------------------------------
void PopupManager::RemovePopupAt(UInt32 popupScreenAreaId,
                                 UInt32 position)
{
   StopTimer();
   GetPopupArea(popupScreenAreaId).RemovePopupAt(position);
   ResetTimer();
}


// ------------------------------------------------------------------------
void PopupManager::RemovePopup(PopupState& popupState)
{
   StopTimer();
   GetPopupArea(popupState.GetPopupScreenAreaId()).RemovePopup(popupState);
   ResetTimer();
}


// ------------------------------------------------------------------------
void PopupManager::ResetTimer()
{
   //SCREENBROKER_LOG_FN();
   Int32 lNextTimerExpiration = std::numeric_limits<Int32>::max();
   UInt32 lTotalPopupCount = 0;

   // Stop timer before recalculation of next popup timer expiration
   StopTimer();

   UInt64 lCurrentTime = Time::Now();
   if (0ull != lCurrentTime)
   {
      for (PopupAreaMap::iterator it = mPopupAreaMap.begin();
            mPopupAreaMap.end().operator != (it);
            ++it)
      {
         Internal::PopupArea& lPopupArea = (*it).second;
         UInt32 lPopupCount = lPopupArea.GetPopupCount();

         for (UInt i = 0; i < lPopupCount; ++i)
         {
            PopupState* lPopupState = lPopupArea.GetPopupAt(i);

            // If time-based update value is set, calculate next timer expiration
            if ((0 != lPopupState) && (0 != lPopupState->GetNextUpdatePeriod()))
            {
               Int32 lPassedTime = static_cast<Int32>(lCurrentTime - lPopupState->GetNextUpdateTimeBase());
               Int32 lExpirationPeriod = static_cast<Int32>(lPopupState->GetNextUpdatePeriod() - static_cast<UInt32>(lPassedTime));
               lNextTimerExpiration = std::min(lExpirationPeriod, lNextTimerExpiration);
            }

            // Sum up total popup count
            ++lTotalPopupCount;
         }
      }
      ETG_TRACE_USR1(("Re-calculated popup timer expiration: %dms", lNextTimerExpiration));
   }

   ETG_TRACE_USR1(("Total number of popups queued: %u", lTotalPopupCount));

   // Re-start timer with new expiration value, only if this value is positive (means expiration in future)
   // and at least one popup is queued.
   if ((lNextTimerExpiration > 0) && (lTotalPopupCount > 0))
   {
      StartTimer(lNextTimerExpiration);
   }
}


// ------------------------------------------------------------------------
void PopupManager::UpdatePopupStates(UInt64 timestamp)
{
   //SCREENBROKER_LOG_FN();
   IPopupManagerActivator* lPopupManagerActivator =
      Internal::PluginManager::GetInstance().Plugin<IPopupManagerActivator>();

   if (0 != lPopupManagerActivator)
   {
      // Synchronize time-out context with the screen broker service
      SCREENBROKER_SYNCHRONIZE();
      // Iterate over popup areas, identifying expired popups and call plugin interface
      for (PopupAreaMap::iterator it = mPopupAreaMap.begin();
            mPopupAreaMap.end().operator != (it);
            ++it)
      {
         Internal::PopupArea& lPopupArea = (*it).second;
         PopupState* lPopupState = lPopupArea.GetFirstExpiredPopupState(timestamp);
         while (0 != lPopupState)
         {
            // Notify popup manager activator plugin
            lPopupManagerActivator->UpdatePopup(*lPopupState);
            // Get next expired popup state
            lPopupState = lPopupArea.GetNextExpiredPopupState();
         }
      }
   }
}


// ------------------------------------------------------------------------
Internal::PopupArea& PopupManager::GetPopupArea(UInt32 popupScreenAreaId)
{
   PopupAreaMap::iterator it = mPopupAreaMap.find(popupScreenAreaId);

   // If not found then create a new entry
   if (mPopupAreaMap.end().operator == (it))
   {
      ETG_TRACE_USR1(("Creating popup area with screen area %u", popupScreenAreaId));
      mPopupAreaMap[popupScreenAreaId] = Internal::PopupArea(popupScreenAreaId);
      return mPopupAreaMap[popupScreenAreaId];
   }

   return (*it).second;
}


// ------------------------------------------------------------------------
void PopupManager::StartTimer(Int32 expirationPeriod) const
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR4(("Starting popup timer with expiration in %ums", expirationPeriod));

   // Calculate timer expiration value
   Int32 lSec = expirationPeriod / 1000;
   Int32 lNs = (expirationPeriod - (lSec * 1000)) * 1000000;
   struct itimerspec lItVal;
   lItVal.it_interval.tv_sec = 0;
   lItVal.it_interval.tv_nsec = 0;
   lItVal.it_value.tv_sec = lSec;
   lItVal.it_value.tv_nsec = lNs;

   if (0 == timerfd_settime(mTimerFd, 0, &lItVal, NULL))
   {
      ETG_TRACE_USR4(("Popup timer started"));
   }
   else
   {
      ETG_TRACE_ERR(("Starting popup timer failed! errno=%d", errno));
   }
}


// ------------------------------------------------------------------------
void PopupManager::StopTimer() const
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR4(("Stopping popup timer"));

   // Calculate reset timer expiration value
   struct itimerspec lItVal;
   lItVal.it_interval.tv_sec = 0;
   lItVal.it_interval.tv_nsec = 0;
   lItVal.it_value.tv_sec = 0;
   lItVal.it_value.tv_nsec = 0;

   // Set expiration time to zero
   if (0 == timerfd_settime(mTimerFd, 0, &lItVal, NULL))
   {
      ETG_TRACE_USR4(("Popup timer stopped"));
   }
   else
   {
      ETG_TRACE_ERR(("Stopping popup timer failed! errno=%d", errno));
   }
}


// ------------------------------------------------------------------------
bool PopupManager::CreateTimer()
{
   //SCREENBROKER_LOG_FN();
   mTimerFd = timerfd_create(Time::ClockType(), TFD_CLOEXEC);
   bool lRc = (-1 != mTimerFd);
   if (lRc)
   {
      ETG_TRACE_USR4(("Popup timer created"));
   }
   else
   {
      ETG_TRACE_ERR(("Creating popup timer failed! errno=%d", errno));
   }

   // Starting timer thread waiting for expiration
   mTimerActive = true;
   bool mThreadCreated = pthread_create(&mTimerThread, NULL, TimerFn, &mTimerActive);
   if (mThreadCreated != 0)
   {
      ETG_TRACE_ERR(("Timer thread not created !!"))
   }
   return lRc;
}


// ------------------------------------------------------------------------
void PopupManager::TerminateTimer()
{
   //SCREENBROKER_LOG_FN();
   mTimerActive = false;
   pthread_cancel(mTimerThread);
}


// ------------------------------------------------------------------------
void* PopupManager::TimerFn(void* arg)
{
   //SCREENBROKER_LOG_FN();
   if (0 != arg)
   {
      bool* lTimerActive = static_cast<bool*>(arg);
      ETG_TRACE_USR1(("Popup timer thread started"));
      while (*lTimerActive)
      {
         ETG_TRACE_USR4(("Popup timer cycle started"));
         uint64_t lMissed;
         // Wait for the next timer event
         if (-1 != read(mTimerFd, &lMissed, sizeof(lMissed)))
         {
            ETG_TRACE_USR1(("Popup timer expired (expiration count: %u)", lMissed));
            UInt64 lCurrentTime = Time::Now();
            if (0ull != lCurrentTime)
            {
               PopupManager::GetInstance().UpdatePopupStates(lCurrentTime);
            }
         }
         else
         {
            ETG_TRACE_ERR(("Reading popup timer failed! errno=%d", errno));
         }
      }
      ETG_TRACE_USR1(("Popup timer thread ended"));
   }
   else
   {
      ETG_TRACE_SYS(("Popup timer thread failed to start!"));
   }
   return NULL;
}


}
