/* ***************************************************************************************
* FILE:          PluginManager.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PluginManager.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_PluginManager_h)
#define ScreenBroker_Service_PluginManager_h

// =============================================================================
// This class is intended to be a singleton, for this reason it is ok, that the
// constructor and the copy constructor have only private access.
//lint -esym(1704, ScreenBroker::Internal::PluginManager::PluginManager)   "Constructor has private access specification"
// =============================================================================

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Plugin/IKeyHandler.h>
#include <ScreenBroker/Plugin/IPopupManagerActivator.h>
#include <ScreenBroker/Plugin/IScreenBrokerActivator.h>
#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include <string>
#include <vector>
#include <map>

namespace ScreenBroker {
namespace Internal {
/// @addtogroup ScreenBrokerService
/// @{

/**
 * @brief The PluginManager class manages loading and instantiating the defined set of plugins.
 */
class PluginManager
{
   public:
      ///
      static PluginManager& GetInstance();

      /**
       * @brief ScreenBrokerPluginLibrary is a screen broker container plugin library,
       * which all plugins code.
       *
       * @note This shared library has priority in the plugins loading strategy.
       * If not found then all plugins are tried to be loaded via their own shared library.
       *
       * @return Name of container plugin library to load
       */
      static const Char* ScreenBrokerPluginLibrary()
      {
         return "libscreenbrokerplugins.so";
      }

      /// @return Name of container plugin library to load
      static const Char* KeyHandlerLibrary()
      {
         return "libkeyhandlerplugin.so";
      }

      /// @return Name of PopupManagerActivator plugin library to load
      static const Char* PopupManagerActivatorLibrary()
      {
         return "libpopupmanageractivatorplugin.so";
      }

      /// @return Name of ScreenBrokerActivator plugin library to load
      static const Char* ScreenBrokerActivatorLibrary()
      {
         return "libscreenbrokeractivatorplugin.so";
      }

      /// @return Name of Configurator plugin library to load
      static const Char* ConfiguratorLibrary()
      {
         return "libscreenbrokerconfiguratorplugin_so.so";
      }

      /// @return Name of ScreenLayouter plugin library to load
      static const Char* ScreenLayouterLibrary()
      {
         return "libscreenlayouterplugin.so";
      }
      ///
      ~PluginManager();

      /**
       * @brief Load the defined set of plugin libraries.
       *
       * @return True if plugins are successfully loaded, false otherwise.
       */
      bool Load();

      /**
       * @brief Unload the defined set of plugin libraries.
       */
      void Unload();

      /**
       * @brief Retrieve the plugin instance with the given ID.
       * @param pluginId ID of plugin to retrieve.
       * @return Requested plugin instance if exists, 0 otherwise.
       */
      IScreenBrokerPlugin* GetPlugin(PluginId::Enum pluginId);

      /**
       * @brief Provides access to a plugin with specialized type.
       * @return Instance of plugin if available, 0 otherwise.
       */
      template<typename T> T* Plugin()
      {
         return static_cast<T*>(GetPlugin(T::Id()));
      }

   private:
      struct SharedLibraryContext
      {
         void* handle;
         const Char* name;
      };
      typedef std::vector<SharedLibraryContext> SharedLibraryList;

      typedef std::map<PluginId::Enum, IScreenBrokerPlugin*> PluginMap;

      /// List to hold handles for shared libraries
      SharedLibraryList mSharedLibraryList;

      /// Holds references to all plugin objects
      PluginMap mPlugins;

      ///
      PluginManager();

      // Dis-allow copying of singleton
      PluginManager(const PluginManager&);
      PluginManager& operator=(const PluginManager&);

      ///
      template<typename T> T* LoadPlugin();

      ///
      bool LoadSharedLibrary(const Char* pluginName);
};


/// @}
}


}

#endif
