/* ***************************************************************************************
* FILE:          LayerManagerAccessor.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LayerManagerAccessor.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_LayerManagerAccessor_h)
#define ScreenBroker_Service_LayerManagerAccessor_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include <map>
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include <ilm/ilm_types.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <list>
#include "hmibase/util/Condition.h"
#endif
namespace ScreenBroker {
namespace Internal {
/// @addtogroup ScreenBrokerService
/// @{

/**
 * @brief The LayerManagerAccessor class provides an internal centralized access to layer manager functionality.
 *
 * @note This class is for internal use only. Plugins may access the layer manager through its client interface.
 */
class LayerManagerAccessor
{
   public:
      ///
      static bool Init();

      ///
      static void Destroy();

      /**
       * Creates a layer with the given parameters.
       * @param layerId id of layer
       * @param width width of layer
       * @param height height of layer
       * @param xPos y position of layer
       * @param yPos y position of layer
       * @param visible True, if layer should be visible
       */
      static bool CreateLayer(UInt32 layerId,
                              UInt32 width,
                              UInt32 height,
                              UInt32 xPos,
                              UInt32 yPos,
                              bool visible);

      /**
       * Removes a layer.
       *
       * @param layerId
       */
      static bool RemoveLayer(UInt32 layerId);

      /**
       * Adds the surface with the given ID to the layer with the given ID.
       *
       * @param layerId
       * @param surfaceId
       */
      static bool AddSurfaceToLayer(UInt32 layerId,
                                    UInt32 surfaceId);

      /**
       * Removes the surface with the given ID from the layer with the given ID.
       *
       * @param layerId
       * @param surfaceId
       */
      static bool RemoveSurfaceFromLayer(UInt32 layerId,
                                         UInt32 surfaceId);

      ///
      static void Dump();
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
      static ::hmibase::util::Condition mCondition;
      static void RegisterWithIlm();
      static void ConfigureILMSurface(t_ilm_uint id, struct ilmSurfaceProperties* sp);
      static void RegistercallbackFunction(ilmObjectType object, t_ilm_uint id, t_ilm_bool created, void* user_data);
      static void SurfaceCallbackFunction(t_ilm_uint id, struct ilmSurfaceProperties* sp, t_ilm_notification_mask m);
      static bool isSurfaceConfigured(UInt32 surfaceId);
      static void popFromCofiguredSurfaceList(UInt32 surfaceId);
      static void setValidatingSurfaceId(UInt32 surfaceId);
      static UInt32 getValidatingSurfaceId();
      static void clearValidatingSurfaceId();
      static bool isValidSurface(UInt32 surfaceId);
#endif
   private:
      static bool mInitialized;
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
      static std::list<UInt32> configuredSurfaceList;
      static UInt32 validatingSurfaceID;
      static CriticalSection mCsl;
      static void pushToCofiguredSurfaceList(UInt32 surfaceId);
      static void clearCofiguredSurfaceList();
#endif
};


/// @}
}


}

#endif
