/* ***************************************************************************************
* FILE:          ClientState.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ClientState.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_ClientState_h)
#define ScreenBroker_Service_ClientState_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Service/PopupManager/PopupState.h>
#include <ScreenBroker/Service/ScreenArea.h>
#include <map>
#include <vector>
#include <set>
#include <string>

// Forward declarations
namespace ScreenBroker {
struct ServiceRequestArg;
struct PopupPresentationArg;
}


namespace ScreenBroker {
namespace Internal {
/// @addtogroup ScreenBrokerService
/// @{

/**
 * @brief The ClientState class holds the internal state of each connected client application.
 */
class ClientState
{
   public:
      ///
      ClientState();

      ///
      ClientState(const std::string& clientId);

      /**
       * @brief In the destructor of the client state all surfaces which are handled with this client are
       * automatically disconnected from its layers at ILM.
       */
      virtual ~ClientState();

      /**
       * @brief Resets the internal status of client state (instance list of surfaces and popups).
       */
      void Reset();

      /**
       * @brief GetClientId
       *
       * @return ID of the client
       */
      const std::string& GetClientId() const
      {
         return mClientId;
      }

      /**
       * @brief Request screen area to a given surface ID
       *
       * @param surfaceId
       * @return Corresponding screen area of the given surface ID, if found,
       *         0, otherwise.
       */
      const ScreenArea* GetScreenArea(UInt32 surfaceId);

      /**
       * @brief Registers the surface with the given ID onto the given screen area for the
       * client application.
       *
       * @param requestArg Request data, holding client defined request ID, application ID and user data.
       * @param screenAreaId The reference to the screen area instance the surface is
       * registered for.
       * @param surfaceId The ID of the surface to register.
       */
      void RegisterSurface(const ServiceRequestArg& serviceRequestArg,
                           const ScreenArea& screenArea,
                           UInt32 surfaceId);

      /**
       * @brief Deregisters the surface with the given ID of the client application.
       *
       * @param surfaceId The ID of the surface to register.
       * @param sendToIlm if deregister should be send to ILM
       */
      void DeregisterSurface(UInt32 surfaceId,
                             bool sendToIlm);

      /**
       * @brief Requests activation of that client application via activator plugin.
       *
       * @param requestArg Request data, holding client defined request ID, application ID and user data.
       * @param surfaceId The ID of the surface to activate.
       * @param entryCustomAnimationType execute Hide animation
       * @param exitCustomAnimationType execute Show animation
       */
      void Activate(const ServiceRequestArg& serviceRequestArg,
                    UInt32 surfaceId,
                    UInt32 entryCustomAnimationType,
                    UInt32 exitCustomAnimationType);

      /**
       * @brief Requests activation of that client application via activator plugin.
       *
       * @param requestArg Request data, holding client defined request ID, application ID and user data.
       * @param surfaceId The ID of the surface to activate.
       * @param animationType The enum value of animation type.
       */
      void TransitionAnimation(const ServiceRequestArg& serviceRequestArg,
                               UInt32 surfaceId,
                               UInt32 animationType,
                               UInt32 hint);

      /**
       * @brief Requests activation of that client application via activator plugin.
       *
       * @param requestArg Request data, holding client defined request ID, application ID and user data.
       * @param Layer The ID of the layer to animate.
       * @param list of surfaceId  ID of the surface to ne attached.
       * @param animationType The enum value of animation type.
       */
      void AttachandAnimateLayer(const ServiceRequestArg& serviceRequestArg,
                                 UInt32 layerId,
                                 std::vector<UInt32> surfaceIdList,
                                 UInt32 animationType,
                                 UInt32 hint);

      /**
       * @brief Forwards the request to the PopupManagerActivator plugin, which then may set various
       * popup properties (e.g. visibility, keyboard focus) resp. arbitrate the popups towards ILM.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID of the popup to show.
       * @param popupPresetationArg Various popup presentation properties.
       */
      void ShowPopup(const ServiceRequestArg& serviceRequestArg,
                     UInt32 surfaceId,
                     const PopupPresentationArg& popupPresentationArg);

      /**
       * @brief Forwards the request to the PopupManagerActivator plugin, which then may set various
       * popup properties (e.g. visibility, keyboard focus) resp. arbitrate the popups towards ILM.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID of the popup to hide.
       */
      void HidePopup(const ServiceRequestArg& serviceRequestArg,
                     UInt32 surfaceId);

   private:
      class SurfaceContext
      {
         public:
            SurfaceContext() : mScreenArea(0), mHint(0) {}
            SurfaceContext(const ScreenBroker::ScreenArea* screenArea, UInt32 hint)
               : mScreenArea(screenArea),
                 mHint(hint) {}
            const ScreenBroker::ScreenArea* ScreenArea() const
            {
               return mScreenArea;
            }
            UInt32 Hint() const
            {
               return mHint;
            }
         private:
            const ScreenBroker::ScreenArea* mScreenArea;
            UInt32 mHint;
      };

      typedef std::map<UInt32, SurfaceContext> SurfaceContextMap;
      typedef std::map<UInt32, PopupState> PopupStateMap;

      /// Map holding the registration context data of each surface
      SurfaceContextMap mSurfaceContextMap;

      /// Map holding the instances of popup states.
      PopupStateMap mPopupStateMap;

      ///
      std::string mClientId;

      /**
       * @brief Provides popup state by given surface ID.
       *
       * @note If popup state with surface ID is not found in list, then a new entry with given surface ID is created.
       *
       * @param surfaceId Surface ID to search the corresponding popup state for.
       * @return Popup state corresponding to given surface ID.
       */
      PopupState& GetPopupState(UInt32 surfaceId);
};


/// @}
}


}

#endif
