/* ***************************************************************************************
* FILE:          IScreenBrokerPlugin.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IScreenBrokerPlugin.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Plugin_IScreenBrokerPlugin_h)
#define ScreenBroker_Plugin_IScreenBrokerPlugin_h

#include <ScreenBroker/ScreenBroker.h>
#include <map>
#include <vector>
#include <string>

namespace ScreenBroker {
/// @addtogroup ScreenBrokerPlugin
/// @{

/// Identification enumeration for all supported plugins
namespace PluginId {
enum Enum
{
   /// First plugin ID
   First = 0,

   ///
   ScreenBrokerActivator = First,
   ///
   PopupManagerActivator,
   ///
   ScreenLayouter,
   ///
   Configurator,
   ///
   KeyHandler,

   /// Last plugin ID
   Last = KeyHandler,

   /// Number of supported plugins
   Count = Last + 1
};


}

/**
 * @brief The IScreenBrokerPlugin class defines the base interace every derived screen broker plugin must comply.
 */
class IScreenBrokerPlugin
{
   public:
      /// Plugin creation function type
      typedef ScreenBroker::IScreenBrokerPlugin& Create();

      ///
      virtual ~IScreenBrokerPlugin() {}

      /**
       * @brief Returns the identification of a plugin instance.
       *
       * @return The identification of the plugin instance.
       */
      virtual PluginId::Enum GetId() const = 0;

      /**
       * @brief Returns the name of a plugin instance.
       *
       * @return Name of the plugin instance.
       */
      virtual const Char* GetName() const = 0;

      /**
       * @brief Inititializes the plugin to be ready for accessing its data & interfaces.
       *
       * @return True, if initialization was successful.
       */
      virtual bool Init() = 0;

      /**
       * @brief Resets the plugin owned data.
       */
      virtual void Reset() = 0;

      /**
       * @brief Generic interface to trigger an action.
       *
       * @param actionId 32bit information to determine action.
       * @param actionData 32bit data of the action
       */
      virtual void Action(UInt32 actionId, UInt32 actionData) = 0;

      /**
       * @brief Call the plugin's diagnosis interface, e.g. for dumping the current state.
       *
       * @param requestId 32bit information to transport details of the requests nature.
       */
      virtual void RequestCurrentStatus(UInt32 requestId) = 0;

      /**
       * @brief Call the plugin's diagnosis interface, e.g. for dumping the current state.
       */
      virtual void Diagnosis() {}

      /**
       * @brief Call the plugin's test interface, e.g. for testing plugin logic.
       */
      virtual void Test() {}
};


/// @}
}


// helper Macros for internal use only
#define concat_A(A) gScreenBrokerPluginFactoryMap[ScreenBroker:: I##A
#define concat_B(B) B ::Name()]=
#define concat_C(pluginName, C) C pluginName
#define concat_D(D) D ::Create;


//

#define SCREENBROKER_PLUGIN_STATIC_HEADER(pluginName)             \
    static PluginId::Enum Id() { return PluginId::pluginName; }   \
    static const Char * Name() { return #pluginName; }

#define DECLARE_SCREENBROKER_PLUGIN(pluginName)    										\
		extern "C" {																	\
		   class pluginName##PluginProxy											    \
		   {																			\
		      public:																	\
			  pluginName##PluginProxy()											        \
		         {																		\
		    	  concat_D(concat_C(pluginName, concat_B(concat_A(pluginName)))) \
		    	  /*gScreenBrokerPluginFactoryMap[ScreenBroker##::##I##pluginName##::##Name()] = pluginName##::##Create;*/ \
		    	  /*concat("gScreenBrokerPluginFactoryMap[ScreenBroker::I", concat(pluginName, concat("::Name()] = ", concat(pluginName,"::Create;"))))*/  \
		            /* Register the plugin instance creation at the global factory map*/	\
		            /*gScreenBrokerPluginFactoryMap[ScreenBroker::I##pluginName##::Name()] = pluginName##::Create;*/	\
		         }																		\
		   };																			\
		   	   	   	   	   	   	   	   	   	   	   	   	   	   	   	   	   	   	   	   	\
		   /* Create one instance of the proxy to perform registration */				\
		   pluginName##PluginProxy g##pluginName##Plugin;							\
		}																				\

/// @addtogroup ScreenBrokerPlugin
/// @{

/// Plugin factory map type
typedef std::map<std::string, ScreenBroker::IScreenBrokerPlugin::Create*, std::less<std::string> > ScreenBrokerPluginFactoryMap;

/// Declaration of global shared plugin factory map
extern ScreenBrokerPluginFactoryMap gScreenBrokerPluginFactoryMap;

/// @}

#endif
