/* ***************************************************************************************
* FILE:          IPopupManagerActivator.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IPopupManagerActivator.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Plugin_IPopupManagerActivator_h)
#define ScreenBroker_Plugin_IPopupManagerActivator_h

#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>

namespace ScreenBroker {
class ServiceRequestArg;
class PopupPresentationArg;
class PopupState;

/// @addtogroup ScreenBrokerPlugin
/// @{

/**
 * @brief The IPopupManagerActivator class defines the interace of for a PopupManagerActivator plugin.
 *
 * The PopupManagerActivator plugin is triggered by the screen broker when client requests
 * to manipulate popup presentation. This plugin has to take care to apply the project specific
 * popup arbitration logic at layer manager.
 */
class IPopupManagerActivator : public IScreenBrokerPlugin
{
   public:
      SCREENBROKER_PLUGIN_STATIC_HEADER(PopupManagerActivator)

      /**
       * @brief Returns the identification of a plugin instance.
       *
       * @return The identification of the plugin instance.
       */
      virtual PluginId::Enum GetId() const
      {
         return Id();
      }

      /**
       * @brief Returns the name of a plugin instance.
       *
       * @return Name of the plugin instance.
       */
      virtual const Char* GetName() const
      {
         return Name();
      }

      /**
       * Requests presentation of the pop-up overlay addressed by surface ID.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @note The layer ID, where this popup resides, is included in popup state.
       *
       * @note The userData and appId from serviceRequestArg are always merged to the popupState object, means they are equal.
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param popupState The object encapsulating the state of the pop-up.
       * @param hint Static surface hinting information, which was defined by user data provided
       * with surface registration by client application.
       */
      virtual void ShowPopup(const ServiceRequestArg& serviceRequestArg,
                             PopupState& popupState,
                             UInt32 hint) = 0;

      /**
       * Requests concealing of the pop-up overlay addressed by surface ID.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @note The layer ID, where this popup resides, is included in popup state.
       *
       * @note The userData and appId from serviceRequestArg are NOT merged to the popupState object, means the popupState
       * object holds the last state updated via ShowPopup and the serviceRequestArg hold the new userData and appId coming from
       * the request itself.
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param popupState The object encapsulating the state of the pop-up.
       */
      virtual void HidePopup(const ServiceRequestArg& serviceRequestArg,
                             PopupState& popupState) = 0;

      /**
       * @brief CloseOnExternalTouch Requests closing of popup based on external touch
       *
       * @param ServiceRequestArg Request information comprises request ID, application ID and user data, which is provided
       */
      virtual void CloseOnExternalTouch(const ServiceRequestArg& serviceRequestArg) = 0;

      /**
       * Filters the presentation of popups matching the given popup presentation arguments.
       * Filter arguments are passed to activator plug-ins.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param disableAll If true, presentation of all popups shall be disabled. In this case the parameter popupPresentationArg is "don't care".
       * @param popupPresentationArg The presentation arguments serving as the filter arguments. Only valid, if parameter disableAll is false.
       */
      virtual void SetPopupFilter(const ServiceRequestArg& serviceRequestArg,
                                  bool disableAll,
                                  const PopupPresentationArg& popupPresentationArg) = 0;

      /**
       * Clears a previous set popup filter.
       *
       * @note When notifying a run-time error back to the client the serviceRequestArg it is recommended to return
       * the incoming parameter serviceRequestArg unmodified to the interface ServerApi::NotifyError().
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       */
      virtual void ClearPopupFilter(const ServiceRequestArg& serviceRequestArg) = 0;

      /**
       * Requests updating of the pop-up overlay addressed by surface ID.
       *
       * @note The layer ID, where this popup resides, is included in popup state.
       *
       * @param popupState The object encapsulating the state of the pop-up.
       */
      virtual void UpdatePopup(PopupState& popupState) = 0;

      /**
       * @brief SetInputFocus request for given surface triggerd by client application.
       *
       * @param surfaceId Surface which should get focus.
       * @param keyboardFocus If keyboard focus should be set.
       * @param pointerFocus If pointer focus should be set.
       * @param priority Priority to use when setting the focus.
       * @param actionData Used to force or unforce the input focus.
       */
      virtual void SetInputFocus(UInt32 surfaceId,
                                 bool keyboardFocus,
                                 bool pointerFocus,
                                 Int32 priority,
                                 UInt32 actionData) = 0;

      /**
       * @brief Deregister request for given surface triggered by client application.
       *
       * @param surfaceId The surface ID of the surface which is deregistered.
       */
      virtual void DeregisterSurface(UInt32 surfaceId) = 0;
};


/// @}
}


#endif
