/* ***************************************************************************************
* FILE:          ClientApi.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ClientApi.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(ScreenBroker_Client_ClientApi_h)
#define ScreenBroker_Client_ClientApi_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/DimensionArg.h>
#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/RequestArg.h>
#include <ScreenBroker/SurfaceState.h>
#include <ScreenBroker/LayerState.h>
#include <ScreenBroker/KeyState.h>
#include <ScreenBroker/BuildInLayerInfo.h>

#include <vector>

namespace ScreenBroker {
/// @addtogroup ScreenBrokerClient
/// @{

/**
 * @brief The ClientApi class is the main client interface for setting up screen broker request and receiving screen broker
 * notifications respecivaly responses.
 *
 * This class serves as base class for any application specific client implementation. This class has all the screen broker
 * callback interfaces (for notifications and responses) defined as virtual.
 * In case any client application wants to listen to any of these service callbacks it has to derive from this class, overwrite
 * them and provide the new application specific ClientApi to the ScreenBroker::Client instance (either with the Client::Init()
 * method the Client::SetClientApi() method).
 */
class ClientApi
{
   public:
      ///
      virtual ~ClientApi() {}

      /**
       * @brief Requests the version from screen broker.
       *
       * @return Screen broker version.
       */
      static const Char* Version();

      static bool GetBuildInLayerInfo(std::vector<BuildInLayerInfo>& info);

      /**
       * @brief Requests the ID of the display the addressed screen area is configured to.
       *
       * Requested values are provided by response OnGetDisplayId().
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request.
       * @param screenAreaId Screen area ID for which the display ID is requested.
       */
      static void GetDisplayId(const RequestArg& requestArg,
                               UInt32 screenAreaId);

      /**
       * @brief Requests the dimensions a full area surface should have to fit to the screen area identified
       * by screenAreaId.
       *
       * Requested values are provided by response OnGetPreferredDimensions().
       *
       * @note This request for preferred surface dimensions is forwarded to the dedicated screen broker plugin
       * (ScreenBrokerActivator).
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param screenAreaId Screen area ID for which the preferred dimensions of surfaces is requested.
       */
      static void GetPreferredDimensions(const RequestArg& requestArg,
                                         UInt32 screenAreaId);

      /**
       * @brief Registers the ILM surface addressed by surfaceId to the screen area addressed by screenAreaId for
       * the client application by the screen broker service.
       *
       * @note Surface specific hinting information is provided in requestArg's userData property, which is forwarded
       * to the respective screen broker plugin, when activating the application's surface.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request.
       * @param screenAreaId Screen area to register the given surface to.
       * @param surfaceId Surface to register.
       */
      static void RegisterSurface(const RequestArg& requestArg,
                                  UInt32 screenAreaId,
                                  UInt32 surfaceId);

      /**
       * @brief Deregisters the surface addressed by surfaceId for the client application by the screen broker service.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request.
       * @param surfaceId Surface to deregister.
       * @param sendToIlm if deregister should be send to ILM
       */
      static void DeregisterSurface(const RequestArg& requestArg,
                                    UInt32 surfaceId,
                                    bool sendToIlm = false);

      /**
       * @brief Requests the surface activation of the requesting client application.
       *
       * Additionally it forwards the request to the dedicated plugin of the screen broker service
       *(ScreenBrokerActivator), which then may set various surface properties (e.g. keyboard focus)
       * resp. arbitrate the surfaces towards ILM.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID to activate for the application.
       */
      static void ActivateApplication(const RequestArg& requestArg,
                                      UInt32 surfaceId, UInt32 entryCustomAnimationType = 0, UInt32 exitCustomAnimationType = 0);

      /**
       * @brief Requests the surface activation of the requesting client application.
       *
       * Additionally it forwards the request to the dedicated plugin of the screen broker service
       *(ScreenBrokerActivator), which then may set various surface properties (e.g. keyboard focus)
       * resp. arbitrate the surfaces towards ILM.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID to activate for the application.
       */
      static void StartAnimation(const RequestArg& requestArg,
                                 UInt32 surfaceId,
                                 UInt32 animationType,
                                 UInt32 hint = 0);

      /**
       * @brief Requests the Layer animation of the requesting client application.
       *
       * Additionally it forwards the request to the dedicated plugin of the screen broker service
       *(ScreenBrokerActivator), which then may set various layer properties
       * resp. arbitrate the layers towards ILM.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param layer ID to animate the layer
       */
      static void StartLayerAnimation(const RequestArg& requestArg,
                                      UInt32 layerId,
                                      std::vector<UInt32> surfaceIdList,
                                      UInt32 animationType,
                                      UInt32 hint = 0);
      /**
       * @brief Requests the presentation of the surface pop-up overlay addressed by surfaceId.
       *
       * The surface must be registered on a screen area dedicated for pop-ups in order to be presented
       * as pop-up. The popup presentation arguments are passed to dedicated screen broker service plugin
       * (PopupManagerActivator).
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID of the popup to show.
       * @param popupPresetationArg Various popup presentation properties.
       */
      static void ShowPopup(const RequestArg& requestArg,
                            UInt32 surfaceId,
                            const PopupPresentationArg& popupPresentationArg);

      /**
       * @brief Requests the concealing of the surface popup overlay addressed by surfaceId.
       *
       * @note The request for hiding a popup is (like the ShowPopup() request) forwarded to the dedicated screen broker
       * plugin (PopupManagerActivator).
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       * @param surfaceId Surface ID of the popup to hide.
       */
      static void HidePopup(const RequestArg& requestArg,
                            UInt32 surfaceId);

      /**
       * @brief Requests closing of popup based on external touch
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request and may be used by the plugin.
       */
      static void CloseOnExternalTouch(const RequestArg& requestArg);

      /**
       * Filters the presentation of popups matching the given popup presentation arguments.
       * Filter arguments are passed to activator plug-ins.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * @param disableAll If true, presentation of all popups shall be disabled. In this case the parameter popupPresentationArg is "don't care".
       * @param popupPresentationArg The presentation arguments serving as the filter arguments. Only valid, if parameter disableAll is false.
       */
      static void SetPopupFilter(const RequestArg& requestArg,
                                 bool disableAll,
                                 const PopupPresentationArg& popupPresentationArg = PopupPresentationArg());

      /**
       * Clears a previous set popup filter.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       */
      static void ClearPopupFilter(const RequestArg& requestArg);

      /**
       * @brief Gives a feedback if respective key was consumed by a client or not.
       *
       * @param userData User data which may give additional information from application to plugin.
       * @param keyCode 32bit code of the key event.
       * @param keyState The state of the key event.
       * @param consumed True, if the key event was already consumed,
       *                 false otherwise.
       */
      static void KeyFeedback(UInt32 userData,
                              UInt32 keyCode,
                              KeyState::Enum keyState,
                              bool consumed);

      /**
       * @brief Generic interface to trigger an action.
       *
       * @param actionId 32bit information to determine action.
       * @param actionData 32bit data of the action
       */
      static void Action(UInt32 actionId, UInt32 actionData);

      /**
       * @brief Interface to set input focus to a certain surface.
       *
       * @param surfaceId id of surface which should get input focus
       * @param keyboardFocus If keyboard focus should be set.
       * @param pointerFocus If pointer focus should be set.
       * @param priority Focus priority (0 default priority, high value -> low priority)
       * @param focusData 32bit data of focus
       */
      static void SetInputFocus(UInt32 surfaceId,
                                bool keyboardFocus,
                                bool pointerFocus,
                                Int32 priority,
                                UInt32 focusData);

      /**
       * @brief Generic interface for requesting the current status based on the request ID,
       * which defines the details of the status requests nature.
       *
       * @note: To which plugin the status request is routed to is defined via the plugin ID.
       *
       * @param requestId 32bit information to transport details of the requests nature.
       */
      static void RequestCurrentStatus(UInt32 requestId);

      /**
       * @brief The response to request GetDisplayId() providing the requested display ID.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request returned in the response.
       * @param displayId Requested display ID to which the screenArea of the request corresponds to.
       */
      virtual void OnGetDisplayId(const RequestArg& requestArg,
                                  UInt32 displayId);

      /**
       * @brief The response to request GetPreferredDimensions() providing the surface dimensions (width/height)
       * for requeste screen area.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request returned in the response.
       * @param dimensionArg Surface dimensions of requested screen area.
       */
      virtual void OnGetPreferredDimensions(const RequestArg& requestArg,
                                            const DimensionArg& dimensionArg);

      /**
       * @brief Notifies clients that the state of the indicated surface has changed.
       *
       * @note The notification of the state change is triggered by the respective screen broker plugin
       * (ScreenbrokerActivator or PopupManagerActivator).
       *
       * @param userData User data which may give additional information from plugin to application.
       * @param surfaceId Surface ID which this state change applies to.
       * @param surfaceState New state of surface.
       */
      virtual void OnSurfaceStateChanged(UInt32 userData,
                                         UInt32 surfaceId,
                                         UInt32 displayId,
                                         UInt32 displayAlias,
                                         SurfaceState::Enum surfaceState);

      /**
       * @brief Notifies clients that a runtime error has occurred with given request.
       *
       * @param requestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request returned in the error response.
       * @param systemErrno System error codes (like linux extended error codes defined in linux/errno.h).
       * @param presentationErrno Error codes specific to the visual presentation module screen broker
       * is connected to (e.g. layer manager).
       */
      virtual void OnError(const RequestArg& requestArg,
                           Int32 systemErrno,
                           Int32 presentationErrno);

      /**
       * @brief Notifies clients that a key event has occurred.
       *
       * @param userData User data which may give additional information from plugin to application.
       * @param surfaceId ID of focused surface on which the key was triggered.
       * @param keyCode 32bit code of the key event.
       * @param keyState The state of the key event.
       */
      virtual void OnKeyEvent(UInt32 userData,
                              UInt32 surfaceId,
                              UInt32 keyCode,
                              KeyState::Enum keyState);

      /**
       * @brief Notifies about current status details based on the request.
       *
       * @note The notification is sent to the owner of the surface defined via surfaceId.
       *
       * @param requestId 32bit information to inform about details of the requests nature.
       * @param surfaceId The ID of the surface, which identifies the response path, may also be part of the status response.
       * @param status 32bit of response data carrying the status information.
       * @param userData Additional 32bit of user data for whatever use.
       */
      virtual void OnCurrentStatus(UInt32 requestId,
                                   UInt32 surfaceId,
                                   UInt32 status,
                                   UInt32 userData);

      /**
       * @brief Notifies about Layer State details based on the update.
       *
       * @note The notification is sent to the owner of the first valid attached surface.
       *
       * @param requestId 32bit information to inform about details of the requests nature.
       * @param layerId layer ID which this state change applies to.
       * @param layerState of layer.
       */
      virtual void OnLayerStateChanged(UInt32 requestId,
                                       UInt32 layerId,
                                       LayerState::Enum state);

      /**
       * @brief Notifies clients that close on external touch is enabled or disabled.
       *
       * @param status Enabled or Disabled
       */
      virtual void OnCloseOnExternalTouch(bool status);

      virtual void OnBuildInLayerInfo(const std::vector<BuildInLayerInfo>& info);

      virtual void OnServiceStatusChanged(bool available);
};


/// @}
}


#endif
