/* ***************************************************************************************
* FILE:          ClientApi.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ClientApi.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Using the "do { ... } while(0) construct within macros is a common practice in order
// to make the use of macros safe within conditions.
//lint -emacro({717}, CHECK_SERVICE_PROXY)    "do ... while(0);"
// =============================================================================

#include "Client.h"
#include "ClientApi.h"
#include "ServiceProxy.h"
#include <ScreenBroker/RequestArg.h>

#include <cassert>

#define CHECK_SERVICE_PROXY(ok, r)                                                      \
    do {                                                                                \
        if (ok) {                                                                       \
            if (!Client::GetInstance().IsRunning()) {                                   \
                /* not required here: ETG_TRACE_SYS_THR(("Client is not running, discarding request %40s!",  \
                                      #r)); */                                               \
                ok = false;                                                             \
            }                                                                           \
        }                                                                               \
        if (ok) {                                                                       \
            Internal::ServiceProxy * lProxy = Client::GetInstance().GetServiceProxy();  \
            assert(0 != lProxy);                                                        \
            if (0 == lProxy) {                                                          \
                /* not required here: ETG_TRACE_ERR_THR(("ServiceProxy handle is invalid!")); */               \
                ok = false;                                                             \
            }                                                                           \
        }                                                                               \
    } while(0)
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERCLIENT
#include "trcGenProj/Header/ClientApi.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenBrokerClient);

// ------------------------------------------------------------------------
const Char* ClientApi::Version()
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, Version);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " Version"));
   }
   return (ok ? Client::GetInstance().GetServiceProxy()->GetVersion() : 0);
}


bool ClientApi::GetBuildInLayerInfo(std::vector<BuildInLayerInfo>& info)
{
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, Version);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " GetBuildInLayerInfo"));
   }
   else
   {
      Client::GetInstance().GetServiceProxy()->GetBuildInLayerInfo(info);
   }

   return (info.size() > 0);
}


// ------------------------------------------------------------------------
void ClientApi::GetDisplayId(const RequestArg& requestArg,
                             UInt32 screenAreaId)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, GetDisplayId);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " GetDisplayId"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->GetDisplayId(requestArg,
            screenAreaId);
   }
}


// ------------------------------------------------------------------------
void ClientApi::GetPreferredDimensions(const RequestArg& requestArg,
                                       UInt32 screenAreaId)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, GetPreferredDimensions);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " GetPreferredDimensions"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->GetPreferredDimensions(requestArg,
            screenAreaId);
   }
}


// ------------------------------------------------------------------------
void ClientApi::RegisterSurface(const RequestArg& requestArg,
                                UInt32 screenAreaId,
                                UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, RegisterSurface);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " RegisterSurface"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->RegisterSurface(requestArg,
            screenAreaId,
            surfaceId);
   }
}


// ------------------------------------------------------------------------
void ClientApi::DeregisterSurface(const RequestArg& requestArg,
                                  UInt32 surfaceId,
                                  bool sendToIlm)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, DeregisterSurface);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " DeregisterSurface"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->DeregisterSurface(requestArg,
            surfaceId,
            sendToIlm);
   }
}


// ------------------------------------------------------------------------
void ClientApi::ActivateApplication(const RequestArg& requestArg,
                                    UInt32 surfaceId,
                                    UInt32 entryCustomAnimationType,
                                    UInt32 exitCustomAnimationType)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, ActivateApplication);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " ActivateApplication"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->ActivateApplication(requestArg,
            surfaceId,
            entryCustomAnimationType,
            exitCustomAnimationType);
   }
}


// ------------------------------------------------------------------------
void ClientApi::StartAnimation(const RequestArg& requestArg,
                               UInt32 surfaceId,
                               UInt32 animationType,
                               UInt32 hint)
{
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, StartAnimation);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " StartAnimation"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->StartAnimation(requestArg,
            surfaceId,
            animationType,
            hint);
   }
}


// ------------------------------------------------------------------------
void ClientApi::StartLayerAnimation(const RequestArg& requestArg,
                                    UInt32 layerId,
                                    std::vector<UInt32> surfaceIdList,
                                    UInt32 animationType,
                                    UInt32 hint)
{
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, StartAnimation);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " StartLayerAnimation"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->StartLayerAnimation(requestArg,
            layerId,
            surfaceIdList,
            animationType,
            hint);
   }
}


// ------------------------------------------------------------------------
void ClientApi::ShowPopup(const RequestArg& requestArg,
                          UInt32 surfaceId,
                          const PopupPresentationArg& popupPresentationArg)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, ShowPopup);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " ShowPopup"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->ShowPopup(requestArg,
            surfaceId,
            popupPresentationArg);
   }
}


// ------------------------------------------------------------------------
void ClientApi::HidePopup(const RequestArg& requestArg,
                          UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, HidePopup);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " HidePopup"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->HidePopup(requestArg,
            surfaceId);
   }
}


// ------------------------------------------------------------------------
void ClientApi::CloseOnExternalTouch(const RequestArg& requestArg)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, CloseOnExternalTouch);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " CloseOnExternalTouch"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->CloseOnExternalTouch(requestArg);
   }
}


// ------------------------------------------------------------------------
void ClientApi::SetPopupFilter(const RequestArg& requestArg,
                               bool disableAll,
                               const PopupPresentationArg& popupPresentationArg)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, ShowPopup);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " ShowPopup"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->SetPopupFilter(requestArg,
            disableAll,
            popupPresentationArg);
   }
}


// ------------------------------------------------------------------------
void ClientApi::ClearPopupFilter(const RequestArg& requestArg)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, HidePopup);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " HidePopup"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->ClearPopupFilter(requestArg);
   }
}


// ------------------------------------------------------------------------
void ClientApi::KeyFeedback(UInt32 userData,
                            UInt32 keyCode,
                            KeyState::Enum keyState,
                            bool consumed)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, KeyFeedback);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " KeyFeedback"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->KeyFeedback(userData,
            keyCode,
            keyState,
            consumed);
   }
}


// ------------------------------------------------------------------------
void ClientApi::Action(UInt32 actionId, UInt32 actionData)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, Action);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " Action"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->Action(actionId, actionData);
   }
}


// ------------------------------------------------------------------------
void ClientApi::SetInputFocus(UInt32 surfaceId,
                              bool keyboardFocus,
                              bool pointerFocus,
                              Int32 priority,
                              UInt32 focusData)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, SetInputFocus);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " SetInputFocus"));
   }

   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->SetInputFocus(surfaceId,
            keyboardFocus,
            pointerFocus,
            priority,
            focusData);
   }
}


// ------------------------------------------------------------------------
void ClientApi::RequestCurrentStatus(UInt32 requestId)
{
   //SCREENBROKER_LOG_FN();
   bool ok = true;
   CHECK_SERVICE_PROXY(ok, RequestCurrentStatus);
   if (!ok)
   {
      ETG_TRACE_ERR_THR(("ServiceProxy not ready for use. Discarding request: " " RequestCurrentStatus"));
   }
   if (ok)
   {
      Client::GetInstance().GetServiceProxy()->RequestCurrentStatus(requestId);
   }
}


// ------------------------------------------------------------------------
void ClientApi::OnGetDisplayId(const RequestArg& requestArg,
                               UInt32 displayId)
{
   ETG_TRACE_SYS_THR(("Service response OnGetDisplayId({%u, %u, %u}, %u) ignored due to missing client implementation!",
                      requestArg.RequestId(),
                      requestArg.AppId(),
                      requestArg.UserData(),
                      displayId));
}


// ------------------------------------------------------------------------
void ClientApi::OnGetPreferredDimensions(const RequestArg& requestArg,
      const DimensionArg& dimensionArg)
{
   ETG_TRACE_SYS_THR(("Service response OnGetPreferredDimensions({%u, %u, %u}, {%u, %u}) ignored due to missing client implementation!",
                      requestArg.RequestId(),
                      requestArg.AppId(),
                      requestArg.UserData(),
                      dimensionArg.Width(),
                      dimensionArg.Height()));
}


// ------------------------------------------------------------------------
void ClientApi::OnSurfaceStateChanged(UInt32 userData,
                                      UInt32 surfaceId,
                                      UInt32 displayId,
                                      UInt32 displayAlias,
                                      SurfaceState::Enum surfaceState)
{
   ETG_TRACE_SYS_THR(("Service notification OnSurfaceStateChanged(%u, %u, %u, %u, %d) ignored due to missing client implementation!",
                      userData,
                      surfaceId,
                      displayId,
                      displayAlias,
                      surfaceState));
}


// ------------------------------------------------------------------------
void ClientApi::OnError(const RequestArg& requestArg,
                        Int32 systemErrno,
                        Int32 presentationErrno)
{
   ETG_TRACE_SYS_THR(("Service response OnError({%u, %u, %u}, %d, %d) ignored due to missing client implementation!",
                      requestArg.RequestId(),
                      requestArg.AppId(),
                      requestArg.UserData(),
                      systemErrno,
                      presentationErrno));
}


// ------------------------------------------------------------------------
void ClientApi::OnKeyEvent(UInt32 userData,
                           UInt32 surfaceId,
                           UInt32 keyCode,
                           KeyState::Enum keyState)
{
   ETG_TRACE_SYS_THR(("Service response OnKeyEvent(%u, %u, %u, %d) ignored due to missing client implementation!",
                      userData,
                      surfaceId,
                      keyCode,
                      keyState));
}


// ------------------------------------------------------------------------
void ClientApi::OnCurrentStatus(UInt32 requestId,
                                UInt32 surfaceId,
                                UInt32 status,
                                UInt32 userData)
{
   ETG_TRACE_SYS_THR(("Service response OnCurrentStatus(%u, %u, %u, %u) ignored due to missing client implementation!",
                      requestId,
                      surfaceId,
                      status,
                      userData));
}


// ------------------------------------------------------------------------
void ClientApi::OnLayerStateChanged(UInt32 requestId,
                                    UInt32 layerId,
                                    LayerState::Enum layerState)
{
   ETG_TRACE_SYS_THR(("Service response OnLayerStateChanged(%u, %u, %u) ignored due to missing client implementation!",
                      requestId,
                      layerId,
                      layerState));
}


// ------------------------------------------------------------------------
void ClientApi::OnCloseOnExternalTouch(bool closeOnExternalTouch)
{
   ETG_TRACE_SYS_THR(("Service response OnCloseOnExtneralTouch(%40s) ignored due to missing client implemenation!",
                      closeOnExternalTouch ? "Enabled" : "Disabled"));
}


void ClientApi::OnBuildInLayerInfo(const std::vector<BuildInLayerInfo>& info)
{
   (void) info;
   ETG_TRACE_SYS_THR(("Service response OnBuildInLayerInfo ignored due to missing client implemenation!"));
}


void ClientApi::OnServiceStatusChanged(bool available)
{
   ETG_TRACE_SYS_THR(("Service response OnServiceStatusChanged(Service %s) ignored due to missing client implemenation!",
                      available ? "available" : "unavailable"));
}


}
