/* ***************************************************************************************
* FILE:          SurfaceGeometryHelper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SurfaceGeometryHelper.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "SurfaceGeometryHelper.h"


#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include <ilm/ilm_input.h>
#endif

#include <cstdlib>

#include "hmibase/trace/HMIBaseTraceCmds_Trace.h"


#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_TRACEDAEMON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SurfaceGeometryHelper.cpp.trc.h"
#endif

namespace hmibase {
namespace trace {
namespace command {

int SurfaceGeometryHelper::_ilmInitStatus = -1;

SurfaceGeometryHelper::SurfaceGeometryHelper()
{
}


SurfaceGeometryHelper::~SurfaceGeometryHelper()
{
}


void SurfaceGeometryHelper::removeHiddenTouchArea(std::vector<surfaceTouchGeometry>& surfaceTouchVector)
{
   ::hmibase::util::geometry::Rect boundingRect;
   for (std::vector<surfaceTouchGeometry>::iterator iter = surfaceTouchVector.begin(); iter != surfaceTouchVector.end();)
   {
      if (boundingRect.contains(iter->touchableRect))
      {
         // this rectangle can be remove as it is fully covered by overlying surfaces
         ETG_TRACE_USR4(("removeHiddenTouchArea: surface %d is fully covered by overlying surfaces", iter->surfaceId));
         iter = surfaceTouchVector.erase(iter);
      }
      else
      {
         boundingRect.setBoundingRect(boundingRect, iter->touchableRect);
         ++iter;
      }
   }
}


bool SurfaceGeometryHelper::getSurfaceCluster(std::vector<surfaceTouchGeometry>& surfaceTouchVector)
{
   if (_ilmInitStatus == -1)
   {
      init();
   }

   if (_ilmInitStatus != ILM_SUCCESS)
   {
      ETG_TRACE_FATAL(("ilm was not correctly initialized, errorcode %d", _ilmInitStatus));
      return false;
   }

   //std::vector<surfaceTouchGeometry> surfaceTouchVector;
   t_ilm_uint numberOfScreenIds;
   t_ilm_uint* screenIds;

   ilmErrorTypes err = ilm_getScreenIDs(&numberOfScreenIds, &screenIds);

   if (err == ILM_SUCCESS)
   {
      for (unsigned int a = 0; a < numberOfScreenIds; a++)
      {
         ilmScreenProperties screenProperties;
         unsigned int screenId = screenIds[a];
         ETG_TRACE_USR4(("HMIBaseTraceCmds::getSurfaceCluster inspect screen %d", screenId));
         err = ilm_getPropertiesOfScreen(screenId, &screenProperties);

         if (err == ILM_SUCCESS)
         {
            for (unsigned int n = 0; n < screenProperties.layerCount; n++)
            {
               t_ilm_int surfaceCount = 0;
               t_ilm_surface* pAllSurfacesurfaces;

               unsigned int layerid = screenProperties.layerIds[n];
               ETG_TRACE_USR4(("HMIBaseTraceCmds::getSurfaceCluster inspect layer %d", layerid));
               // get all surfaces on the layer
               err = ilm_getSurfaceIDsOnLayer(layerid, &surfaceCount, &pAllSurfacesurfaces);

               if (err == ILM_SUCCESS)
               {
                  for (int p = 0; p < surfaceCount; p++)
                  {
                     ilmSurfaceProperties surfaceProperties;
                     unsigned int id = pAllSurfacesurfaces[p];
                     ETG_TRACE_USR4(("HMIBaseTraceCmds::getSurfaceCluster inspect surface %d", id));
                     err = ilm_getPropertiesOfSurface(id, &surfaceProperties);

                     if (err == ILM_SUCCESS)
                     {
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
                        if ((surfaceProperties.visibility)
                              && ((surfaceProperties.focus & ILM_INPUT_DEVICE_TOUCH) || (surfaceProperties.focus & ILM_INPUT_DEVICE_POINTER)))
#else
                        if ((surfaceProperties.visibility)
                              && ((surfaceProperties.inputDevicesAcceptance & ILM_INPUT_DEVICE_TOUCH) || (surfaceProperties.inputDevicesAcceptance & ILM_INPUT_DEVICE_POINTER)))
#endif
                        {
                           ETG_TRACE_USR4(("HMIBaseTraceCmds::getSurfaceCluster add surface %d to list of touchable rectangles", id));
                           surfaceTouchVector.insert(surfaceTouchVector.begin(), surfaceTouchGeometry(id,
                                                     ::hmibase::util::geometry::Rect(
                                                        ::hmibase::util::geometry::Point(static_cast<int>(surfaceProperties.destX), static_cast<int>(surfaceProperties.destY)),
                                                        ::hmibase::util::geometry::Size(static_cast<int>(surfaceProperties.destWidth), static_cast<int>(surfaceProperties.destHeight))
                                                     )
                                                                                                     )
                                                    );
                        }
                     }
                  }
               }

               if (pAllSurfacesurfaces != 0)
               {
                  free(pAllSurfacesurfaces);
               }
            }
         }
      }
   }

   if (screenIds != 0)
   {
      free(screenIds);
   }

   if (err != ILM_SUCCESS)
   {
      return false;
   }

   // don't remove hidden area as there is a possibility to touch surfaces that are hidden by others, which are fully transparent on certain pixels
   // this decision is normally taken by LayerManager. Removing the call here is not an overall fitting solution and will not simulate the whole
   // LayerManager functionality
   //removeHiddenTouchArea(surfaceTouchVector);
   return true;
}


int SurfaceGeometryHelper::getPidForSurfaceId(int id)
{
   init();
   if (_ilmInitStatus != ILM_SUCCESS)
   {
      ETG_TRACE_FATAL(("ilm was not correctly initialized, errorcode %d", _ilmInitStatus));
      return false;
   }

   int pid = 0;
   if (_ilmInitStatus == ILM_SUCCESS)
   {
      ilmSurfaceProperties surfaceProperties;
      ilmErrorTypes err = ilm_getPropertiesOfSurface(id, &surfaceProperties);

      if (err == ILM_SUCCESS)
      {
         pid = surfaceProperties.creatorPid;
      }

      ETG_TRACE_USR4(("PID for surface %d is %d", id, pid));
   }
   else
   {
      ETG_TRACE_ERR(("ILM not initialized successfully, error %s", ILM_ERROR_STRING(_ilmInitStatus)))
   }

   return pid;
}


void SurfaceGeometryHelper::init()
{
   if (_ilmInitStatus == -1)
   {
      _ilmInitStatus = ilm_init();

      if (_ilmInitStatus != ILM_SUCCESS)
      {
         ETG_TRACE_FATAL(("Failed to initialze ILM, error code %s", ILM_ERROR_STRING(_ilmInitStatus)));
      }
   }
}


}
}


}
