/* ***************************************************************************************
* FILE:          WidgetGestureConfig.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetGestureConfig is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once


#include <hmibase/util/ItemRegistry.h>
#include <FeatStd/Util/Rtti.h>


namespace hmibase {
namespace widget {

/* Factors used to adjust widget gesture configuration.
*/
struct WidgetGestureFactor
{
   public:
      WidgetGestureFactor() :
         TapHoldTime(1.0f), TapRepeatTime(1.0f), TapDoubleTapTime(1.0f), TapMaxDistance(1.0f),
         DragHoldTime(1.0f), DragHoldMaxDistance(1.0f), DragMinDistance(1.0f),
         SwipeMinSpeed(1.0f), SwipeMinDistance(1.0f),
         PinchSpreadMaxTime(1.0f), PinchSpreadMinSpreadFactor(1.0f), PinchSpreadMaxPinchFactor(1.0f),
         RotateMaxTime(1.0f), RotateMinAngle(1.0f)
      {
      }

      //Tap
      FeatStd::Float TapHoldTime;
      FeatStd::Float TapRepeatTime;
      FeatStd::Float TapDoubleTapTime;
      FeatStd::Float TapMaxDistance;

      //Drag
      FeatStd::Float DragHoldTime;
      FeatStd::Float DragHoldMaxDistance;
      FeatStd::Float DragMinDistance;

      //Swipe
      FeatStd::Float SwipeMinSpeed;
      FeatStd::Float SwipeMinDistance;

      //PinchSpread
      FeatStd::Float PinchSpreadMaxTime;
      FeatStd::Float PinchSpreadMinSpreadFactor;
      FeatStd::Float PinchSpreadMaxPinchFactor;

      //Rotate
      FeatStd::Float RotateMaxTime;
      FeatStd::Float RotateMinAngle;
};


typedef hmibase::util::ItemValRegistry<unsigned int, WidgetGestureFactor> DisplayGestureFactorRegistry;


/*
* The widget properties allow to enable various gestures or to specify the drag/swipe direction.
*
* But additional configuration is often required. For example timeout before press hold or repeat is detected. Or minimum distance and speed for movement gestures.
* The widgets provide default values which should fit the common situations. However in order to be able to implement a wider range of features,
* it should be possible to configure each widget independently or to provide the same configuration for a group of widgets.

* Additional configuration is possible by using the GestureConfigId property which allows to overwrite the default gesture configuration for a particular widget instance.
* Gesture configurations are registered at start-up and associated to numerical ids. Those ids are set as values for GestureConfigId property directly or with a databinding source.


//Add some gesture config id constants and a gesture config databinding source to the HMI contract.
<constants>
   <const name="GESTURE_CONFIG_ID_DRAG_SWIPE_PANEL" type="uint32" uid="" value="1001"/>
   <const name="GESTURE_CONFIG_ID_DOUBLE_TAP_PANEL" type="uint32" uid="" value="1002"/>
</constants>
<bindingSources>
   <bindingSource name="GestureConfigs" readonly="true" access="asynchronous" uid="">
      <item name="DragSwipePanel" type="Candera::UInt32" readonly="true" uid="" defaultValue="GESTURE_CONFIG_ID_DRAG_SWIPE_PANEL"/>
      <item name="DoubleTapPanel" type="Candera::Int32" readonly="true" uid=""defaultValue="GESTURE_CONFIG_ID_DOUBLE_TAP_PANEL"/>
   </bindingSource>
</bindingSources>


//Register the gesture configurations and send the databinding update at start-up.
#define HMIBASE_INCLUDE_WIDGET_UTIL
#include <hmibase/Widgets.h>

void onStartUp()
{
   // register a drag and swipe gesture config
   WidgetGestureConfigRegistry::registerItem(GESTURE_CONFIG_ID_DRAG_SWIPE_PANEL, WidgetGestureConfig(WidgetGestureConfig::Drag(true, 100, 50, 10), WidgetGestureConfig::Swipe(true, 20, 100)));

   // register a tap gesture config
   WidgetGestureConfigRegistry::registerItem(GESTURE_CONFIG_ID_DOUBLE_TAP_PANEL, WidgetGestureConfig(WidgetGestureConfig::Tap(true, 0, 0, 250)));

   //send data binding updates
   DataBindingItem<GestureConfigsDataBindingSource>().SendUpdate(true);
}


*/
class WidgetGestureConfig
{
   public:
      struct Tap
      {
         // gesture is enabled for this configuration. The Tap enabled flag of the widget is also checked. Both have to be true.
         bool Enabled;
         // Time for hold detection
         FeatStd::UInt32 HoldTime;
         // Time for repeat detection
         FeatStd::UInt32 RepeatTime;
         // Time in which double tab to be detected
         FeatStd::UInt32 DoubleTapTime;
         // maximum distance to move; when moving beyond this limit, the gesture is aborted
         FeatStd::Int32 MaxDistance;

         explicit Tap(bool enabled = false, FeatStd::UInt32 holdTime = 0, FeatStd::UInt32 repeatTime = 0, FeatStd::UInt32 doubleTapTime = 0, FeatStd::Int32 maxDistance = -1) :
            Enabled(enabled),
            HoldTime(holdTime),
            RepeatTime(repeatTime),
            DoubleTapTime(doubleTapTime),
            MaxDistance(maxDistance)
         {
         }
      };

      struct Drag
      {
         // gesture is enabled for this configuration. The Drag enabled flag of the widget is also checked. Both have to be true.
         bool Enabled;
         // Time to hold before dragging will start
         FeatStd::UInt32 HoldTime;
         // Pixel to move during hold time without canceling gesture detection, only used when DragHoldTime != 0
         FeatStd::Int32 HoldMaxDistance;
         // Minimum distance in pixel to move after hold time elapsed to detect gesture
         FeatStd::Int32 MinDistance;
         // Not removed from touch session when other drag gesture is detected. Use this in combination with HoldTime to have drag for 2 purposes: drag without hold for list scroll and drag with hold for gizmo movement.
         bool ParallelEnabled;

         explicit Drag(bool enabled = false, FeatStd::UInt32 holdTime = 0, FeatStd::Int32 holdMaxDistance = 0, FeatStd::Int32 minDistance = 0, bool parallelEnabled = false) :
            Enabled(enabled),
            HoldTime(holdTime),
            HoldMaxDistance(holdMaxDistance),
            MinDistance(minDistance),
            ParallelEnabled(parallelEnabled)
         {
         }
      };

      struct Swipe
      {
         // gesture is enabled for this configuration. The Swipe enabled flag of the widget is also checked. Both have to be true.
         bool Enabled;
         // Minimum movement speed in pixel/s
         FeatStd::Int32 MinSpeed;
         // Minimum distance of total movement in pixel
         FeatStd::Int32 MinDistance;

         explicit Swipe(bool enabled = false, FeatStd::Int32 minSpeed = 0, FeatStd::Int32 minDistance = 0) :
            Enabled(enabled),
            MinSpeed(minSpeed),
            MinDistance(minDistance)
         {
         }
      };

      struct PinchSpread
      {
         // gesture is enabled for this configuration. The PinchSpread enabled flag of the widget is also checked. Both have to be true.
         bool Enabled;
         // Time in which the second touch has to be detected
         FeatStd::UInt32 MaxTime;
         // minimum factor (distance now divided by distance at press time) for detecting a spread; should be = 1.0
         FeatStd::Float MinSpreadFactor;
         // maximum factor (distance now divided by distance at press time) for detecting a pinch; should be = 1.0
         FeatStd::Float MaxPinchFactor;

         explicit PinchSpread(bool enabled = false, FeatStd::UInt32 maxTime = 0, FeatStd::Float minSpreadFactor = 0.0f, FeatStd::Float maxPinchFactor = 0.0f) :
            Enabled(enabled),
            MaxTime(maxTime),
            MinSpreadFactor(minSpreadFactor),
            MaxPinchFactor(maxPinchFactor)
         {
         }
      };

      struct Rotate
      {
         // gesture is enabled for this configuration. The Rotate enabled flag of the widget is also checked. Both have to be true.
         bool Enabled;
         // Time in which the second touch has to be detected
         FeatStd::UInt32 MaxTime;
         // Minimum angle to rotate in degree
         FeatStd::Int32 MinAngle;

         explicit Rotate(bool enabled = false, FeatStd::UInt32 maxTime = 0, FeatStd::Int32 minAngle = 0) :
            Enabled(enabled),
            MaxTime(maxTime),
            MinAngle(minAngle)
         {
         }
      };

      struct RawTouch
      {
         // gesture is enabled for this configuration. The RawTouch enabled flag of the widget is also checked. Both have to be true.
         bool Enabled;

         explicit RawTouch(bool enabled = false) : Enabled(enabled) {}
      };

      WidgetGestureConfig() {}
      WidgetGestureConfig(const Tap& value) : _Tap(value) {}
      WidgetGestureConfig(const Drag& value) : _Drag(value) {}
      WidgetGestureConfig(const Swipe& value) : _Swipe(value) {}
      WidgetGestureConfig(const PinchSpread& value) : _PinchSpread(value) {}
      WidgetGestureConfig(const Rotate& value) : _Rotate(value) {}
      WidgetGestureConfig(const RawTouch& value) : _RawTouch(value) {}

      WidgetGestureConfig(const Drag& drag, const Swipe& swipe) : _Drag(drag), _Swipe(swipe) {}

      WidgetGestureConfig(const Tap& tap, const Drag& drag, const Swipe& swipe = Swipe(), const PinchSpread& pinchSpread = PinchSpread(), const Rotate& rotate = Rotate(), const RawTouch& rawTouch = RawTouch()) :
         _Tap(tap),
         _Drag(drag),
         _Swipe(swipe),
         _PinchSpread(pinchSpread),
         _Rotate(rotate),
         _RawTouch(rawTouch)
      {
      }

      WidgetGestureConfig& operator*=(const WidgetGestureFactor& factor)
      {
         _Tap.HoldTime = static_cast<FeatStd::UInt32>(static_cast<FeatStd::Float>(_Tap.HoldTime) * factor.TapHoldTime);
         _Tap.RepeatTime = static_cast<FeatStd::UInt32>(static_cast<FeatStd::Float>(_Tap.RepeatTime) * factor.TapRepeatTime);
         _Tap.DoubleTapTime = static_cast<FeatStd::UInt32>(static_cast<FeatStd::Float>(_Tap.DoubleTapTime) * factor.TapDoubleTapTime);
         _Tap.MaxDistance = static_cast<FeatStd::Int32>(static_cast<FeatStd::Float>(_Tap.MaxDistance) * factor.TapMaxDistance);

         _Drag.HoldTime = static_cast<FeatStd::UInt32>(static_cast<FeatStd::Float>(_Drag.HoldTime) * factor.DragHoldTime);
         _Drag.HoldMaxDistance = static_cast<FeatStd::Int32>(static_cast<FeatStd::Float>(_Drag.HoldMaxDistance) * factor.DragHoldMaxDistance);
         _Drag.MinDistance = static_cast<FeatStd::Int32>(static_cast<FeatStd::Float>(_Drag.MinDistance) * factor.DragMinDistance);

         _Swipe.MinSpeed = static_cast<FeatStd::Int32>(static_cast<FeatStd::Float>(_Swipe.MinSpeed) * factor.SwipeMinSpeed);
         _Swipe.MinDistance = static_cast<FeatStd::Int32>(static_cast<FeatStd::Float>(_Swipe.MinDistance) * factor.SwipeMinDistance);

         _PinchSpread.MaxTime = static_cast<FeatStd::UInt32>(static_cast<FeatStd::Float>(_PinchSpread.MaxTime) * factor.PinchSpreadMaxTime);
         _PinchSpread.MinSpreadFactor *= factor.PinchSpreadMinSpreadFactor;
         _PinchSpread.MaxPinchFactor *= factor.PinchSpreadMaxPinchFactor;

         _Rotate.MaxTime = static_cast<FeatStd::UInt32>(static_cast<FeatStd::Float>(_Rotate.MaxTime) * factor.RotateMaxTime);
         _Rotate.MinAngle = static_cast<FeatStd::Int32>(static_cast<FeatStd::Float>(_Rotate.MinAngle) * factor.RotateMinAngle);

         return *this;
      }

      WidgetGestureConfig operator*(const WidgetGestureFactor& factor) const
      {
         WidgetGestureConfig result(*this);
         result *= factor;
         return result;
      }

      const Tap& getTap() const
      {
         return _Tap;
      }
      const Drag& getDrag() const
      {
         return _Drag;
      }
      const Swipe& getSwipe() const
      {
         return _Swipe;
      }
      const PinchSpread& getPinchSpread() const
      {
         return _PinchSpread;
      }
      const Rotate& getRotate() const
      {
         return _Rotate;
      }
      const RawTouch& getRawTouch() const
      {
         return _RawTouch;
      }

      static const Tap& getDefaultTap()
      {
         return defaults()._Tap;
      }
      static const Drag& getDefaultDrag()
      {
         return defaults()._Drag;
      }
      static const Swipe& getDefaultSwipe()
      {
         return defaults()._Swipe;
      }
      static const PinchSpread& getDefaultPinchSpread()
      {
         return defaults()._PinchSpread;
      }
      static const Rotate& getDefaultRotate()
      {
         return defaults()._Rotate;
      }
      static const RawTouch& getDefaultRawTouch()
      {
         return defaults()._RawTouch;
      }

      static const WidgetGestureConfig& getDefaults()
      {
         return defaults();
      }
      static void setDefaults(const WidgetGestureConfig& config)
      {
         defaults() = config;
      }

#define WIDGET_GESTURE_CONFIG_SET(TData)\
   WidgetGestureConfig& set(const TData& value)\
   {\
      _##TData = value;\
      return *this;\
   }

      WIDGET_GESTURE_CONFIG_SET(Tap)
      WIDGET_GESTURE_CONFIG_SET(Drag)
      WIDGET_GESTURE_CONFIG_SET(Swipe)
      WIDGET_GESTURE_CONFIG_SET(PinchSpread)
      WIDGET_GESTURE_CONFIG_SET(Rotate)
      WIDGET_GESTURE_CONFIG_SET(RawTouch)

#undef WIDGET_GESTURE_CONFIG_SET

   private:
      static WidgetGestureConfig& defaults()
      {
         static WidgetGestureConfig defaultConfig(
            Tap(true, 3000/*hold time*/, 1000/*repeat time*/, 1000/*double tap time*/),
            Drag(true, 0, 0, 3/*min distance*/),
            Swipe(true, 100/*min velocity*/, 20/*min distance*/),
            PinchSpread(true),
            Rotate(true));
         return defaultConfig;
      }

      Tap _Tap;
      Drag _Drag;
      Swipe _Swipe;
      PinchSpread _PinchSpread;
      Rotate _Rotate;
      RawTouch _RawTouch;
};


typedef hmibase::util::ItemValRegistry<unsigned int, WidgetGestureConfig> WidgetGestureConfigRegistry;

class WidgetGestureConfigByTypeRegistry : public hmibase::util::ItemValRegistry<FeatStd::TypeId, WidgetGestureConfig>
{
   public:
      template <typename TWidget>
      static const WidgetGestureConfig* get(const TWidget& w)
      {
         for (typename ItemMapType::const_iterator it = getItemMap().begin(); it != getItemMap().end(); ++it)
         {
            if (w.IsTypeOf(it->first))
            {
               return &(it->second);
            }
         }
         return NULL;
      }
};


}
}


typedef ::hmibase::widget::WidgetGestureConfig WidgetGestureConfig;
typedef ::hmibase::widget::WidgetGestureFactor WidgetGestureFactor;
typedef ::hmibase::widget::WidgetGestureConfigRegistry WidgetGestureConfigRegistry;
typedef ::hmibase::widget::WidgetGestureConfigByTypeRegistry WidgetGestureConfigByTypeRegistry;
typedef ::hmibase::widget::DisplayGestureFactorRegistry DisplayGestureFactorRegistry;
