/* ***************************************************************************************
* FILE:          WidgetController.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetController.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once


#include <FeatStd/Util/Optional.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Courier/Visualization/IViewHandler.h>
#include "WidgetGestureConfig.h"

namespace Candera {
class WidgetBase;
}


namespace Courier {
class Message;
}


namespace hmibase {
namespace input {
namespace gesture {

struct GestureEvent;
}


}


namespace widget {

class Widget;
class WidgetCloneHelper;
class WidgetGestureConfig;

/*****************************************************************************/
/*
* Base class for the controller data.
* The widget controller instantiates it and sets it on the controllable widget.
*/
class WidgetControllerData
{
   public:
      CANDERA_RTTI_DECLARATION(WidgetControllerData);

      virtual ~WidgetControllerData() {}
};


/*****************************************************************************/
/*
* Base class for the controller events.
*/
class WidgetControllerEvent
{
   public:
      CANDERA_RTTI_DECLARATION(WidgetControllerEvent);

      virtual ~WidgetControllerEvent() {}
};


/*****************************************************************************/
/*
* Provides support to delegate updates and messages to a widget controller.
* Widget controllers are registered at start-up and attached to widgets using the bindable property ControllerId.


* How to use widget controllers:

* - implement the widget controller by extending WidgetController
class MyGizmoWidgetController : public ::hmibase::widget::WidgetController
{
public:
virtual void Update(DelegateWidget& widget);
virtual bool OnMessage(DelegateWidget& widget, const Courier::Message& msg);
};


* - add some widget controller id constants in the hmi contract
<constants>
<const name="CONTROLLER_ID_GIZMO_CONTROLLER" type="int32" uid="" value="1"/>
<const name="CONTROLLER_ID_MY_CONTROLLER_1" type="int32" uid="" value="100"/>
<const name="CONTROLLER_ID_MY_CONTROLLER_2" type="int32" uid="" value="200"/>
</constants>

* - add a widget controller databinding source the hmi contract
<bindingSources>
<bindingSource name="WidgetControllers" readonly="true" access="asynchronous" uid="">
<item name="Gizmo" type="Candera::Int32" readonly="true" uid="" defaultValue="CONTROLLER_ID_GIZMO_CONTROLLER"/>
<item name="MyController1" type="Candera::Int32" readonly="true" uid="" defaultValue="CONTROLLER_ID_MY_CONTROLLER_1"/>
<item name="MyController2" type="Candera::Int32" readonly="true" uid="" defaultValue="CONTROLLER_ID_MY_CONTROLLER_2"/>
</bindingSource>
</bindingSources>

* - register the widget controller at application startup
void main()
{
...
static MyGizmoWidgetController gizmoController;
::hmibase::widget::WidgetControllerRegistry::registerItem(CONTROLLER_ID_GIZMO_CONTROLLER, &gizmoController);
...
}


* - in scene composer bind the ControllerId property of the delegate widget to an item in the widget controllers databinding source
WidgetControllers@/ControllerId/Gizmo
*/
class WidgetController
{
   public:
      typedef Candera::WidgetBase DelegateWidget;

      CANDERA_RTTI_DECLARATION(WidgetController);

      virtual ~WidgetController() {}

      /*****************************************************************************/
      /* Attaches the controller to the widget. Creates and sets the controller data on the widget. */
      virtual void Attach(DelegateWidget& widget);

      /* Detaches the controller from the widget. Destroys the controller data from the widget. */
      virtual void Detach(DelegateWidget& widget);

      /* Creates a new controller data. */
      virtual WidgetControllerData* CreateData(DelegateWidget& /*widget*/)
      {
         return NULL;
      }

      /* Destroys the controller data. */
      virtual void DestroyData(DelegateWidget& widget, WidgetControllerData* data);

      /*****************************************************************************/
      /* Delegate methods */

      /* Update */
      virtual void Update(DelegateWidget& /*widget*/) {}

      /* Messages */
      virtual bool OnMessage(DelegateWidget& /*widget*/, const Courier::Message& /*msg*/)
      {
         return false;
      }

      /* Event */
      virtual bool OnEvent(DelegateWidget& /*widget*/, const WidgetControllerEvent& /*e*/)
      {
         return false;
      }

      /* Called at the beginning of the touch session when this widget's node is visible and has been touched. Return true if the widget should be added to the touch session. */
      virtual bool OnTouch(DelegateWidget& /*widget*/, const Candera::Camera2D& /*camera*/, const Candera::Vector2& /*touchedPoint*/)
      {
         return true;
      }

      /* Gestures */
      virtual FeatStd::Optional<WidgetGestureConfig> GetDefaultGestureConfig(const DelegateWidget& /*widget*/)
      {
         return FeatStd::Optional<WidgetGestureConfig>();
      }

      virtual void CheckGestureConfig(const DelegateWidget& /*widget*/, WidgetGestureConfig&)
      {
      }

      virtual bool OnGesture(DelegateWidget& /*widget*/, const hmibase::input::gesture::GestureEvent& /*gestureData*/)
      {
         return false;
      }

      /* Change notifications */
      //virtual void OnNodeChanged(DelegateWidget& /*widget*/) {}
      virtual void OnChanged(DelegateWidget& /*widget*/, FeatStd::UInt32 /*propertyId*/) {}

      /* Parent view */
      //virtual void OnParentViewActivate(DelegateWidget& /*widget*/, bool /*activate*/) {}
      //virtual void OnParentViewRenderingEnabled(DelegateWidget& /*widget*/, bool /*enable*/) {}
      //virtual void OnParentViewLoad(DelegateWidget& /*widget*/, bool /*load*/) {}

      /* Clone */
      virtual bool CloneFrom(DelegateWidget& /*widget*/, const Widget& /*originalWidget*/, WidgetCloneHelper& /*cloneHelper*/)
      {
         return true;
      }

      /* Get access to touch info of the current session */
      Courier::TouchInfo GetCurrentSessionTouchInfo(const DelegateWidget& w) const;
      Courier::TouchInfo GetCurrentSessionStartTouchInfo(const DelegateWidget& w) const;
};


typedef hmibase::util::ItemPtrRegistry<unsigned int, WidgetController> WidgetControllerRegistry;

#define WIDGET_CONTROLLER_ON_GESTURE() \
virtual bool OnGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData) \
{ \
   switch (gestureData._gestureType) \
   { \
   case hmibase::input::gesture::GestureEvent::GT_PressHoldRepeatTap: \
      return OnTapGesture(widget, gestureData); \
   case hmibase::input::gesture::GestureEvent::GT_DragNudge: \
      return OnDragGesture(widget, gestureData); \
   case hmibase::input::gesture::GestureEvent::GT_SwipeFling: \
      return OnSwipeGesture(widget, gestureData); \
   case hmibase::input::gesture::GestureEvent::GT_PinchSpread: \
      return OnPinchSpreadGesture(widget, gestureData); \
   case hmibase::input::gesture::GestureEvent::GT_Rotate: \
      return OnRotateGesture(widget, gestureData); \
   default: \
      return false; \
   } \
}


}
}


HMIBASE_DEPRECATED_TYPE_ALIAS(WidgetController, hmibase::widget::WidgetController)
HMIBASE_DEPRECATED_TYPE_ALIAS(WidgetControllerData, hmibase::widget::WidgetControllerData)
HMIBASE_DEPRECATED_TYPE_ALIAS(WidgetControllerEvent, hmibase::widget::WidgetControllerEvent)
HMIBASE_DEPRECATED_TYPE_ALIAS(WidgetControllerRegistry, hmibase::widget::WidgetControllerRegistry)

//deprecated type required for old widget controllers
HMIBASE_DEPRECATED("use type DelegateWidget for the methods in your controller", typedef Candera::WidgetBase DelegateWidget2D);
