/* ***************************************************************************************
* FILE:          TouchSessionBase.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TouchSessionBase.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(TouchHandling_TouchSessionBase_h)
#define TouchHandling_TouchSessionBase_h

#include <Courier/Visualization/IViewHandler.h>
#include <Courier/Visualization/ViewHandlerSession.h>
#include <Courier/Util/FixedSizeBitVector.h>
#include <Courier/Foundation/FoundationMsgs.h>
#include <FeatStd/Container/Vector.h>

namespace Courier {
class TouchMsg;
struct TouchInfo;
}

namespace hmibase {
   namespace input {

#if ! defined(COURIER_DEFAULT_TOUCHSOURCE_COUNT)
static const Courier::UInt32 cCOURIER_DEFAULT_TOUCHSOURCE_COUNT = 2;
#define COURIER_DEFAULT_TOUCHSOURCE_COUNT Courier::cCOURIER_DEFAULT_TOUCHSOURCE_COUNT
#else
   static const Courier::UInt32 cCOURIER_DEFAULT_TOUCHSOURCE_COUNT = COURIER_DEFAULT_TOUCHSOURCE_COUNT;
#endif

#if ! defined(COURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT)
static const Courier::UInt32 cCOURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT = 10;
#define COURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT Courier::cCOURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT
#else
   static const Courier::UInt32 cCOURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT = COURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT;
#endif


class TouchAbort;
namespace gesture{
   class GestureTimerExpiredMsg;
}

///
class TouchSessionBase : public Courier::ViewHandlerSession
{
   public:
      /** The following value is not allowed to be used as surface id. It indicates an invalid surface id. */
      static const FeatStd::UInt32 InvalidSourceId = 0xFFFFFFFF;

      ///
      TouchSessionBase();

      ///
      virtual ~TouchSessionBase();

      ///
      virtual bool IsSessionStarted() const;

      ///
      virtual void OnSessionStart(const Courier::TouchInfo& touchInfo) = 0;

      ///
      virtual void OnSessionStop(const Courier::TouchInfo& touchInfo) = 0;

      ///
      virtual bool OnMessage(const Courier::TouchMsg* touchMsgPtr) = 0;

      ///
      virtual bool OnMessage(const TouchAbort* msg) = 0;

      ///
      virtual bool OnMessage(const gesture::GestureTimerExpiredMsg* msg) = 0;

      ///
      virtual bool ProcessMessage(const Courier::Message& msg);

      ///
      bool IsPointerDown(FeatStd::UInt32 sourceId, Courier::PointerId pointerId) const
      {
         const DownState* downState = TryGetDownState(sourceId);
         return (0 != downState) && downState->mDownState.Test(pointerId);
      }

      bool ResetDownState(unsigned int sourceId);

      virtual void printInfo();

      /// Returns the touch info associated with the current touch message
      const Courier::TouchInfo& GetTouchInfo() const
      {
         return _touchInfo;
      }

      /// Returns the touch info associated with the last started touch session
      const Courier::TouchInfo& GetStartTouchInfo() const
      {
         return _startTouchInfo;
      }

   private:
      ///
      typedef Courier::Internal::FixedSizeBitVector<cCOURIER_DEFAULT_MAX_TOUCHPOINTER_COUNT> PointerIdBitVector;

      ///
      struct DownState
      {
         explicit DownState(FeatStd::UInt32 sourceId) : mSourceId(sourceId) { }

         FeatStd::UInt32 mSourceId;
         PointerIdBitVector mDownState;
      };

      ///
      const DownState* TryGetDownState(FeatStd::UInt32 sourceId) const;

      ///
      DownState* GetDownState(FeatStd::UInt32 sourceId);

      ///
      bool CheckSessionStart(const Courier::TouchMsg* touchMsgPtr, bool& msgConsumed);

      ///
      bool CheckSessionStop(const Courier::TouchMsg* touchMsgPtr, bool& msgConsumed);

      ///
      bool IsSurfaceValid(FeatStd::UInt32 surfaceId) const;

      ///
      FeatStd::Internal::Vector<DownState> mDownStates;

      /// current touch info
      Courier::TouchInfo _touchInfo;

      /// touch info when the session starts
      Courier::TouchInfo _startTouchInfo;
};


}
}


#endif
