/* ***************************************************************************************
* FILE:          TouchSessionBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TouchSessionBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "lint_deactivation.h"
#include "TouchSessionBase.h"
#include <Courier/Diagnostics/Log.h>
#include <Courier/Visualization/IViewHandler.h>

#include "BaseContract/generated/BaseTouchSessionMsgs.h"

#ifndef WIN32
#include "CanderaPlatform/Device/Genivi/Target/ILM/GeniviSurface.h"
#include "AppBase/ILM_Accessor.h"
#endif

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TouchSessionBase.cpp.trc.h"
#endif

using namespace Courier;

namespace hmibase {
namespace input {


void callback(unsigned int id)
{
   Courier::Message* msg = COURIER_MESSAGE_NEW(TouchAbort)(id);
   if (msg)
   {
      msg->Post();
   }
}


// ------------------------------------------------------------------------
TouchSessionBase::TouchSessionBase() : _touchInfo(0, 0, 0, 0, 0), _startTouchInfo(0, 0, 0, 0, 0)
{
   (void)mDownStates.Reserve(cCOURIER_DEFAULT_TOUCHSOURCE_COUNT);
}


// ------------------------------------------------------------------------
TouchSessionBase::~TouchSessionBase()
{
}


bool TouchSessionBase::ResetDownState(unsigned int sourceId)
{
   bool success = false;
   DownState* downState = GetDownState(sourceId);
   if (downState)
   {
      ETG_TRACE_USR1_THR(("TouchSessionBase::ResetDownState: reset downstate for sourceId %u", sourceId));
      if (downState->mDownState.GetFirstSetBit() != downState->mDownState.EndIndex())
      {
         success = true;
      }
      downState->mDownState.Reset();
      downState->mSourceId = InvalidSourceId;
   }
   return success;
}


// ------------------------------------------------------------------------
bool TouchSessionBase::IsSessionStarted() const
{
   for (FeatStd::SizeType i = 0; i < mDownStates.Size(); ++i)
   {
      if ((mDownStates[i].mSourceId != InvalidSourceId) && (mDownStates[i].mDownState.EndIndex() != mDownStates[i].mDownState.GetFirstSetBit()))
      {
         return true;
      }
   }
   return false;
}


// ------------------------------------------------------------------------
bool TouchSessionBase::ProcessMessage(const Message& msg)
{
   bool msgConsumed = false;
   switch (msg.GetId())
   {
      case Courier::TouchMsg::ID:
      {
         const Courier::TouchMsg* touchMsgPtr = Courier::message_cast<const Courier::TouchMsg*>(&msg);
         if (touchMsgPtr)
         {
            _touchInfo = TouchInfo(touchMsgPtr->GetXPos(), touchMsgPtr->GetYPos(), touchMsgPtr->GetTimeStamp(), touchMsgPtr->GetPointerId(), touchMsgPtr->GetSourceId());

            if (!IsSessionStarted())
            {
               _startTouchInfo = _touchInfo;
               // Down: If session handler is installed, session is always active after between 'dn' and 'up'
               if (!CheckSessionStart(touchMsgPtr, msgConsumed))
               {
                  // session couldn't be started, so reset touch info
                  _startTouchInfo = TouchInfo(0, 0, 0, 0, 0);
               }
            }
            else
            {
               // Move or Up ? CheckSessionStop reacts on up
               bool shallStop = CheckSessionStop(touchMsgPtr, msgConsumed);
               if (shallStop)
               {
                  _touchInfo = TouchInfo(0, 0, 0, 0, 0);
                  _startTouchInfo = TouchInfo(0, 0, 0, 0, 0);

                  // up detected
                  return msgConsumed;
               }

               UInt32 sourceId = touchMsgPtr->GetSourceId();
               PointerId pointerId = touchMsgPtr->GetPointerId();
               DownState* downState = GetDownState(sourceId);

               if (0 != downState)
               {
                  if (touchMsgPtr->GetState() == TouchMsgState::Down)
                  {
                     // down is normally no a valid state
                     if (downState->mDownState.Test(pointerId))
                     {
                        ETG_TRACE_ERR_THR(("TouchSessionBase::ProcessMessage [%25s]: Already received TouchMsg(Down,%u)", hmibase::trace::getAppName().c_str(), pointerId));
                     }
                     ETG_TRACE_USR4_THR(("TouchSessionBase::ProcessMessage [%25s]: Set sourceId=%u, pointerId=%u, address=%p", hmibase::trace::getAppName().c_str(), sourceId, pointerId, downState));
                     downState->mDownState.Set(pointerId);
                     msgConsumed = OnMessage(touchMsgPtr);
                  }
                  else if (touchMsgPtr->GetState() == TouchMsgState::Move)
                  {
                     if (!downState->mDownState.Test(pointerId))
                     {
                        ETG_TRACE_ERR_THR(("TouchSessionBase::ProcessMessage [%25s]: Cant Move: No related TouchMsg(Down,%u)", hmibase::trace::getAppName().c_str(), pointerId));
                     }
                     msgConsumed = OnMessage(touchMsgPtr);
                  }
                  else /* up */
                  {
                     /* MISRA Rule 5-0-21 else- branch must have a comment or action */
                     // sat2hi: do not assert here, as it is a valid use case that 'up' is received and session is not stopped before
                     // due to the fact, that the 'up' can be from different pointerIDs (multi finger touch)
                     //COURIER_DEBUG_FAIL();
                  }
               }
            }
         }
      }
      break;
      case TouchAbort::ID:
      {
         const TouchAbort* touchAbortMsgPtr = Courier::message_cast<const TouchAbort*>(&msg);
         if (0 != touchAbortMsgPtr)
         {
            ETG_TRACE_USR4_THR(("TouchSessionBase::OnMessage(TouchAbort*) [%25s]: TouchAbort for surface %d received", hmibase::trace::getAppName().c_str(), touchAbortMsgPtr->GetSurfaceId()));

            if (IsSessionStarted())
            {
               ETG_TRACE_USR4_THR(("TouchSessionBase::OnMessage(TouchAbort*) [%25s]: clear session for surface id %d", hmibase::trace::getAppName().c_str(), touchAbortMsgPtr->GetSurfaceId()));

               if (ResetDownState(touchAbortMsgPtr->GetSurfaceId()))
               {
                  // down state for this surface was reseted, so proceed further
                  msgConsumed = OnMessage(touchAbortMsgPtr);

                  if (!IsSessionStarted())
                  {
                     ETG_TRACE_USR4_THR(("TouchSessionBase::OnMessage(TouchAbort*) [%25s]: no session active anymore, so remove all registered widgets", hmibase::trace::getAppName().c_str()));

                     // remove all registered widgets
                     OnSessionStop(TouchInfo(0, 0, 0, 0, touchAbortMsgPtr->GetSurfaceId()));
                  }
                  else
                  {
                     ETG_TRACE_USR4_THR(("TouchSessionBase::OnMessage(TouchAbort*) [%25s]: session still active", hmibase::trace::getAppName().c_str()));
                  }
               }
               else
               {
                  // there was no down detected for this surface, so do nothing
                  msgConsumed = true;
               }
            }
         }
      }
      break;
      case hmibase::input::gesture::GestureTimerExpiredMsg::ID:
      {
         const hmibase::input::gesture::GestureTimerExpiredMsg* gestureTimerExpiredMsg = Courier::message_cast<const hmibase::input::gesture::GestureTimerExpiredMsg*>(&msg);
         msgConsumed = OnMessage(gestureTimerExpiredMsg);
      }
      break;
      default:
         break;
   }
   return msgConsumed;
}


// ------------------------------------------------------------------------
const TouchSessionBase::DownState* TouchSessionBase::TryGetDownState(UInt32 sourceId) const
{
   if (sourceId != InvalidSourceId)
   {
      for (FeatStd::SizeType i = 0; i < mDownStates.Size(); ++i)
      {
         const DownState* currentDownState = &mDownStates[i];
         if (sourceId == currentDownState->mSourceId)
         {
            return currentDownState;
         }
      }
   }
   return 0;
}


// ------------------------------------------------------------------------
TouchSessionBase::DownState* TouchSessionBase::GetDownState(UInt32 sourceId)
{
   DownState* downState = 0;
   if (sourceId != InvalidSourceId)
   {
      for (FeatStd::SizeType i = 0; i < mDownStates.Size(); ++i)
      {
         DownState* currentDownState = &mDownStates[i];
         if (currentDownState->mSourceId == sourceId)
         {
            downState = currentDownState;
            break;
         }
         else if ((0 == downState) && (InvalidSourceId == currentDownState->mSourceId))
         {
            downState = currentDownState;
            //fet2hi: don't set the source id here because we will end up with having multiple entries in the vector with the same source id
            //downState->mSourceId = sourceId;
         }
         else
         {
            // MISRA Rule 6-4-2, 'else' at end of 'if ... else if' chain required
         }
      }
      if (0 != downState)
      {
         downState->mSourceId = sourceId;
      }
      else if (mDownStates.Add(DownState(sourceId)))
      {
         downState = &mDownStates[mDownStates.Size() - 1];
      }
      else
      {
         // nothing to do
      }
   }
   return downState;
}


// ------------------------------------------------------------------------
bool TouchSessionBase::CheckSessionStart(const Courier::TouchMsg* touchMsgPtr, bool& msgConsumed)
{
   COURIER_DEBUG_ASSERT(touchMsgPtr != 0);
   COURIER_DEBUG_ASSERT(! IsSessionStarted());

   msgConsumed = false;
   if ((0 != touchMsgPtr) && (touchMsgPtr->GetState() == TouchMsgState::Down))
   {
      UInt32 sourceId = touchMsgPtr->GetSourceId();
      PointerId pointerId = touchMsgPtr->GetPointerId();
      DownState* downState = GetDownState(sourceId);

      if (IsSurfaceValid(sourceId) == false)
      {
         ETG_TRACE_ERR_THR(("TouchSessionBase::CheckSessionStart [%25s]: Surface %u has become inaccessible meanwhile, skip touch event", hmibase::trace::getAppName().c_str(), sourceId));
         return false;
      }

      if (0 != downState)
      {
         ETG_TRACE_USR1_THR(("TouchSessionBase::CheckSessionStart [%25s]: Set sourceId=%u, pointerId=%u, address=%p", hmibase::trace::getAppName().c_str(), sourceId, pointerId, downState));
         downState->mDownState.Set(pointerId);
         // Send startevent with TouchInfo round
#ifndef WIN32
         Candera::GeniviSurface::SetUnloadCallback(callback);
#endif
         OnSessionStart(TouchInfo(touchMsgPtr->GetXPos(), touchMsgPtr->GetYPos(), touchMsgPtr->GetTimeStamp(), pointerId, sourceId));
         // and registered widgets should get the touch message
         // if no widget was hit (consumed msg), forward this msg to next receiver
         msgConsumed = OnMessage(touchMsgPtr);
         return true;
      }
   }
   return false;
}


// ------------------------------------------------------------------------
bool TouchSessionBase::CheckSessionStop(const Courier::TouchMsg* touchMsgPtr, bool& msgConsumed)
{
   COURIER_DEBUG_ASSERT(touchMsgPtr != 0);
   COURIER_DEBUG_ASSERT(IsSessionStarted());

   msgConsumed = false;
   if ((0 != touchMsgPtr) && (touchMsgPtr->GetState() == TouchMsgState::Up))
   {
      UInt32 sourceId = touchMsgPtr->GetSourceId();
      PointerId pointerId = touchMsgPtr->GetPointerId();
      DownState* downState = GetDownState(sourceId);
      if (0 != downState)
      {
         if (!downState->mDownState.Test(pointerId))
         {
            ETG_TRACE_ERR_THR(("TouchSessionBase::CheckSessionStop [%25s]: Never received related TouchMsg(Down,%u)", hmibase::trace::getAppName().c_str(), pointerId));
         }
         else
         {
            ETG_TRACE_USR4_THR(("TouchSessionBase::CheckSessionStop [%25s]: Reset sourceId=%u, pointerId=%u, address=%p", hmibase::trace::getAppName().c_str(), sourceId, pointerId, downState));
            downState->mDownState.Reset(pointerId);
            if (downState->mDownState.EndIndex() == downState->mDownState.GetFirstSetBit())
            {
               downState->mSourceId = InvalidSourceId;
            }
            msgConsumed = OnMessage(touchMsgPtr);
         }
         if (!IsSessionStarted())
         {
            // remove all registered widgets
            OnSessionStop(TouchInfo(touchMsgPtr->GetXPos(), touchMsgPtr->GetYPos(), touchMsgPtr->GetTimeStamp(), pointerId, sourceId));
            return true;
         }
      }
   }
   return false;
}


void TouchSessionBase::printInfo()
{
   ETG_TRACE_FATAL_THR(("[%25s] Session started %d ", hmibase::trace::getAppName().c_str(), IsSessionStarted()));
   if (mDownStates.Size() == 0)
   {
      ETG_TRACE_FATAL_THR(("[%25s] no down states", hmibase::trace::getAppName().c_str()));
   }
   else
   {
      for (FeatStd::SizeType i = 0; i < mDownStates.Size(); ++i)
      {
         ETG_TRACE_FATAL_THR(("[%25s] down state source id %d", hmibase::trace::getAppName().c_str(), mDownStates[i].mSourceId));

         for (FeatStd::UInt32 j = 0; j < mDownStates[i].mDownState.Capacity(); ++j)
         {
            ETG_TRACE_FATAL_THR(("[%25s]\t bit %u state %d", hmibase::trace::getAppName().c_str(), j, mDownStates[i].mDownState.Test(j)));
         }
      }
   }
}


bool TouchSessionBase::IsSurfaceValid(FeatStd::UInt32 surfaceId) const
{
   bool valid = false;
   if (surfaceId < SURFACEID_VIRTUAL_SURFACE_OFFSET)
   {
      // check if surface is known to ILM, if not it was most likely destroyed meanwhile
#ifdef WIN32
      valid = true;
#else
      std::vector<unsigned int> surfaceIds;
      ILM_Accessor::getSurfaceIds(surfaceIds);

      for (std::vector<unsigned int>::iterator it = surfaceIds.begin(); it != surfaceIds.end(); ++it)
      {
         if (*it == surfaceId)
         {
            valid = true;
            break;
         }
      }
#endif
   }
   else
   {
      valid = true;
   }
   return valid;
}


}
}
