/* ***************************************************************************************
* FILE:          TouchSession.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TouchSession.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "lint_deactivation.h"
#include "TouchSession.h"
#include "View/IMessageSystem.h"

#include "TouchGestures/DragGestureDetector.h"
#include "TouchGestures/LegacyTouchDetector.h"
#include "TouchGestures/RawGestureDetector.h"
#include "TouchGestures/RotateGestureDetector.h"
#include "TouchGestures/SpreadGestureDetector.h"
#include "TouchGestures/SwipeGestureDetector.h"
#include "TouchGestures/TapGestureDetector.h"

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TouchSession.cpp.trc.h"
#endif

class WidgetComparerByTouchPriority
{
   public:
      bool operator()(const hmibase::input::gesture::GestureListener* left, const hmibase::input::gesture::GestureListener* right) const
      {
         FEATSTD_DEBUG_ASSERT(0 != left);
         FEATSTD_DEBUG_ASSERT(0 != right);
         if ((0 != left) && (0 != right))
         {
            FeatStd::UInt32 leftTouchPriority = left->getGesturePriorizationParameter().effectiveTouchPriority;
            FeatStd::UInt32 rightTouchPriority = right->getGesturePriorizationParameter().effectiveTouchPriority;
            if (leftTouchPriority > rightTouchPriority)
            {
               return true;
            }
            if (leftTouchPriority < rightTouchPriority)
            {
               return false;
            }
            return left->getGesturePriorizationParameter().effectiveRenderOrderRank > right->getGesturePriorizationParameter().effectiveRenderOrderRank;
         }
         return false;
      }
};


using namespace Courier;

namespace hmibase {
namespace input {

#ifdef VARIANT_S_FTR_ENABLE_DEPRECATED_GESTURE_PRIORITY
bool TouchSession::_isDeprecatedGesturePriorityEnabled = true;
#else
bool TouchSession::_isDeprecatedGesturePriorityEnabled = false;
#endif


// ------------------------------------------------------------------------
TouchSession::TouchSession() : _legacyTouchInfo(0)
{
}


// ------------------------------------------------------------------------
TouchSession::~TouchSession()
{
}


// ------------------------------------------------------------------------
void TouchSession::enableDeprecatedGesturePriority(bool enable)
{
   ETG_TRACE_USR4_THR(("TouchSession::enableDeprecatedGesturePriority [%25s]: enable=%u", hmibase::trace::getAppName().c_str(), enable));

   _isDeprecatedGesturePriorityEnabled = enable;
}


// ------------------------------------------------------------------------
void TouchSession::OnSessionStart(const TouchInfo& touchInfo)
{
   const TouchSessionStartEvent startEvent(*this, touchInfo);
   if (GetViewHandler() != 0)
   {
      (void)GetViewHandler()->OnMessageViewTreeOnly(startEvent);
   }

   WidgetComparerByTouchPriority objSort;
   std::sort(_gestureListener.begin(), _gestureListener.end(), objSort);
   registerGestures();
}


void TouchSession::registerGestures()
{
   ETG_TRACE_USR4_THR(("TouchSession::registerGestures [%25s]: %d listener", hmibase::trace::getAppName().c_str(), _gestureListener.size()));

   //
   size_t normalizedPrio = (_gestureListener.size() > 0) ? (_gestureListener.size() - 1) : 0;
   FeatStd::Int64 ancestorPrio = FeatStd::Internal::NativeTypeLimit<FeatStd::Int64>::Min();
   for (GestureListenerList::iterator it = _gestureListener.begin(); it != _gestureListener.end(); ++it)
   {
      hmibase::input::gesture::GestureListener* listener = *it;

      // calculate gesture priority based on gesture type and effective touch priority
      FeatStd::Int64 prio = 0;
      if (listener)
      {
         ::hmibase::input::gesture::GestureListener::GesturePriorizationParameter gesturePriorizationParameter = listener->getGesturePriorizationParameter();
         if (isDeprecatedGesturePriorityEnabled())
         {
            prio = static_cast<FeatStd::Int64>(gesturePriorizationParameter.effectiveRenderOrderRank);
         }
         else
         {
            prio = (static_cast<FeatStd::Int64>(gesturePriorizationParameter.effectiveTouchPriority) << (sizeof(FeatStd::UInt32) * 8)) + static_cast<FeatStd::Int64>(gesturePriorizationParameter.effectiveRenderOrderRank);
         }
      }

      // if prio of ancestor is bigger than actual one, reduce the normalized value by one
      if (ancestorPrio > prio)
      {
         --normalizedPrio;
      }
      ancestorPrio = prio;
      //

      if (listener)
      {
         hmibase::widget::Widget::Gestures gestures = listener->GetGestureList();
         ETG_TRACE_USR4_THR(("TouchSession::registerGestures [%25s]: listener supporting %d gestures", hmibase::trace::getAppName().c_str(), gestures.size()));

         for (hmibase::widget::Widget::Gestures::iterator gesture = gestures.begin(); gesture != gestures.end(); ++gesture)
         {
            char gesturePriority = gesture::GestureHandler::s_getInstance().getGesturePriority(gesture->_type);
            int priority = 0;

            if (gesturePriority != static_cast<char>(gesture::GestureDetector::PRIORITY_INVALID))
            {
               // shift normalized render order rank to upper bytes and add gesture priority
               priority = ((static_cast<unsigned int>(normalizedPrio) << (sizeof(gesturePriority) * 8)) + gesturePriority);
            }
            else
            {
               ETG_TRACE_ERR_THR(("TouchSession::registerGestures [%25s]: invalid gesture priority received for gesture type %d", hmibase::trace::getAppName().c_str(), gesture->_type));
            }

            ETG_TRACE_USR4_THR(("TouchSession::registerGestures [%25s]: register gesture %d with prio %d", hmibase::trace::getAppName().c_str(), ETG_CENUM(GestureEvent::GestureType, gesture->_type), priority));

            switch (gesture->_type)
            {
               case gesture::GestureEvent::GT_PressHoldRepeatTap:
               {
                  hmibase::widget::Widget::GestureConfig::TapPressHoldRepeatConfig config = gesture->_config.tapPressHoldRepeatGestureParameter;
                  gesture::TapGestureDetector::registerGesture(listener,
                        priority,
                        config.doubleTapEnabled,
                        gesture->_parallelEnabled,
                        gesture->_parallelType,
                        config.doubleTapTime,
                        config.maxDistance,
                        config.holdTime,
                        config.repeatTime);
               }
               break;
               case gesture::GestureEvent::GT_DragNudge:
               {
                  hmibase::widget::Widget::GestureConfig::DragNudgeConfig config = gesture->_config.dragGestureParameter;
                  gesture::DragGestureDetector::registerGesture(listener,
                        priority,
                        config.direction,
                        gesture->_parallelEnabled,
                        gesture->_parallelType,
                        config.holdTime,
                        config.holdMaxDistance,
                        config.minDistance);
               }
               break;
               case gesture::GestureEvent::GT_SwipeFling:
               {
                  hmibase::widget::Widget::GestureConfig::SwipeFlingConfig config = gesture->_config.swipeGestureParameter;
                  gesture::SwipeGestureDetector::registerGesture(listener,
                        priority,
                        config.direction,
                        gesture->_parallelEnabled,
                        gesture->_parallelType,
                        config.minSpeed,
                        config.minDistance);
               }
               break;
               case gesture::GestureEvent::GT_PinchSpread:
               {
                  hmibase::widget::Widget::GestureConfig::PinchSpreadConfig config = gesture->_config.pinchGestureParameter;
                  gesture::SpreadGestureDetector::registerGesture(listener,
                        priority,
                        gesture->_parallelEnabled,
                        gesture->_parallelType,
                        config.doubleTouchMaxTime,
                        config.spreadMinFactor,
                        config.pinchMaxFactor);
               }
               break;
               case gesture::GestureEvent::GT_Rotate:
               {
                  hmibase::widget::Widget::GestureConfig::RotateConfig config = gesture->_config.rotateGestureParameter;
                  gesture::RotateGestureDetector::registerGesture(listener,
                        priority,
                        gesture->_parallelEnabled,
                        gesture->_parallelType,
                        config.doubleTouchMaxTime,
                        config.minRotationAngle);
               }
               break;
               case gesture::GestureEvent::GT_RawTouchData:
               {
                  gesture::RawGestureDetector::registerGesture(listener, priority);
               }
               break;
               case gesture::GestureEvent::GT_LegacyTouch:
               {
                  gesture::LegacyTouchDetector::registerGesture(listener, priority);
               }
               break;
               default:
                  break;
            }
         }
      }
   }
}


// ------------------------------------------------------------------------
void TouchSession::OnSessionStop(const TouchInfo& touchInfo)
{
   const TouchSessionStopEvent stopEvent(*this, touchInfo);
   ETG_TRACE_USR4_THR(("TouchSession::OnSessionStop [%25s]: inform %d widgets", hmibase::trace::getAppName().c_str(), _gestureListener.size()));

   if (GetViewHandler() != 0)
   {
      (void)GetViewHandler()->OnMessageViewTreeOnly(stopEvent);
   }

   Cleanup();
}


// ------------------------------------------------------------------------
bool TouchSession::OnMessage(const Courier::TouchMsg* touchMsgPtr)
{
   COURIER_DEBUG_ASSERT(touchMsgPtr != 0);
   bool lMsgConsumed = false;
   if (touchMsgPtr != 0)
   {
      _legacyTouchInfo = const_cast<Courier::TouchMsg*>(touchMsgPtr);
      lMsgConsumed = gesture::GestureHandler::s_getInstance().onTouchEvent(touchMsgPtr->GetXPos(), touchMsgPtr->GetYPos(), static_cast<int>(touchMsgPtr->GetState()), touchMsgPtr->GetPointerId(), touchMsgPtr->GetTimeStamp());
      _legacyTouchInfo = 0;
   }
   return lMsgConsumed;
}


bool TouchSession::OnMessage(const TouchAbort* msg)
{
   COURIER_DEBUG_ASSERT(msg != 0);
   bool lMsgConsumed = false;
   if (msg != 0)
   {
      _legacyTouchInfo = const_cast<TouchAbort*>(msg);
      gesture::GestureHandler::s_getInstance().onTouchAbort();
      _legacyTouchInfo = 0;
   }
   return lMsgConsumed;
}


bool TouchSession::OnMessage(const hmibase::input::gesture::GestureTimerExpiredMsg* msg)
{
   return gesture::GestureHandler::s_getInstance().OnMessage(*msg);
}


// ------------------------------------------------------------------------
bool TouchSession::Register(hmibase::input::gesture::GestureListener* listener)
{
   bool registered = false;
   if (listener != 0)
   {
      _gestureListener.push_back(listener);
      ETG_TRACE_USR4_THR(("TouchSession::Register [%25s]: gestureListener registered", hmibase::trace::getAppName().c_str()));
      registered = true;
   }
   return registered;
}


// ------------------------------------------------------------------------
bool TouchSession::Deregister(const hmibase::input::gesture::GestureListener* listener)
{
   if (listener != 0)
   {
      for (GestureListenerList::iterator it = _gestureListener.begin(); it != _gestureListener.end(); ++it)
      {
         if (*it == listener)
         {
            gesture::GestureHandler::s_getInstance().removeDetectors(*it);
            _gestureListener.erase(it);
            ETG_TRACE_USR4_THR(("TouchSession::Deregister [%25s]: widget deregistered", hmibase::trace::getAppName().c_str()));
            return true;
         }
      }
   }
   ETG_TRACE_USR4_THR(("TouchSession::Deregister [%25s]: %d widgets registered", hmibase::trace::getAppName().c_str(), _gestureListener.size()));

   return false;
}


// ------------------------------------------------------------------------
void TouchSession::Cleanup()
{
   ETG_TRACE_USR4_THR(("TouchSession::Cleanup [%25s]", hmibase::trace::getAppName().c_str()));
   gesture::GestureHandler::s_getInstance().removeAllGestures();
   _gestureListener.clear();
}


void TouchSession::printInfo()
{
   TouchSessionBase::printInfo();
   ETG_TRACE_FATAL_THR(("[%25s] %d widgets registered", hmibase::trace::getAppName().c_str(), _gestureListener.size()));
   gesture::GestureHandler::s_getInstance().printInfo();
}


}
}


#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {

template<> UInt32 StringBufferAppender< hmibase::input::TouchSession::Ref >::Append(StringBuffer&, ::hmibase::input::TouchSession::Ref const&)
{
   // nothing to append for TouchSession::Ref
   return 0;
}


template<> UInt32 StringBufferAppender< Courier::TouchInfo >::Append(StringBuffer& stringBuffer, Courier::TouchInfo const& object)
{
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("{");
   tcharCount += stringBuffer.Append("PointerId = ");
   tcharCount += stringBuffer.AppendObject(object.mPointerId);
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("SourceId = ");
   tcharCount += stringBuffer.AppendObject(object.mSourceId);
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("X = ");
   tcharCount += stringBuffer.AppendObject(object.mX);
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("Y = ");
   tcharCount += stringBuffer.AppendObject(object.mY);
   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


} //namespace FeatStd
#endif
