/* ***************************************************************************************
* FILE:          TouchEventPreprocessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TouchEventPreprocessor.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "lint_deactivation.h"
#include "TouchEventPreprocessor.h"
#include <Courier/Diagnostics/Log.h>
#include <Courier/Platform/CriticalSectionLocker.h>
#include <Courier/Foundation/FoundationMsgs.h>
#include <Courier/Messaging/MessagingMsgs.h>
#include <Courier/Visualization/ComponentWakeupHelper.h>

namespace Courier {
namespace Internal {
static bool forceInitViewComponentWakeupHelper = ComponentWakeupHelper<ComponentId(ComponentType::View)>::Init();
}


}

//////// TRACE IF ///////////////////////////////////////////////////
#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TouchEventPreprocessor.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace Courier {
namespace TouchHandling {

using namespace Diagnostics;

// ------------------------------------------------------------------------
TouchEventPreprocessor::TouchEventPreprocessor() : mOccurredTouchEvents(0), mTouchEventThreshold(10)
{
}


// ------------------------------------------------------------------------
TouchEventPreprocessor::~TouchEventPreprocessor()
{
}


// ------------------------------------------------------------------------
bool TouchEventPreprocessor::Process()
{
   Platform::CriticalSectionLocker csLock(&mMsgListCs);
   while (! mMsgList.IsEmpty())
   {
      MessageList::Iterator firstMsgRef = mMsgList.Begin();
      MessageReferrer msgRef(*firstMsgRef);
      (void)mMsgList.Remove(*firstMsgRef);

      if (! msgRef.GetMessage()->Post())
      {
         ETG_TRACE_ERR_THR(("TouchMsg could not be posted by TouchEventPreprocessor"));
      }
   }

   mOccurredTouchEvents = 0;

   return true;
}


// ------------------------------------------------------------------------
bool TouchEventPreprocessor::Receive(Message* msgPtr)
{
   if (msgPtr == 0)
   {
      ETG_TRACE_ERR_THR(("No Message received!"));
      return false;
   }

   TouchMsg* touchMsg = Courier::message_cast<TouchMsg*>(msgPtr);
   if (touchMsg == 0)
   {
      ETG_TRACE_ERR_THR(("No TouchMsg received!"));
      return false;
   }

   Platform::CriticalSectionLocker csLock(&mMsgListCs);
   MessageReferrer msgRef(touchMsg);

   ++mOccurredTouchEvents;

   MessageList::Iterator it = mMsgList.Begin();
   MessageList::Iterator itLastMove = mMsgList.End();
   while (it != mMsgList.End())
   {
      TouchMsg* queuedMsg = Courier::message_cast<TouchMsg*>((*it).GetMessage());
      COURIER_DEBUG_ASSERT(queuedMsg != 0);
      if (queuedMsg != 0)
      {
         if ((touchMsg->GetPointerId() == queuedMsg->GetPointerId()) && (touchMsg->GetSourceId() == queuedMsg->GetSourceId()))
         {
            if ((touchMsg->GetState() == TouchMsgState::Move) && (queuedMsg->GetState() == TouchMsgState::Move))
            {
               itLastMove = it;
            }
            else
            {
               ETG_TRACE_USR1_THR(("TouchMsg State %u for Source %u Pointer %u already queued", touchMsg->GetState(), touchMsg->GetSourceId(), touchMsg->GetPointerId()));
               itLastMove = mMsgList.End();
               break;
            }
         }
      }
      ++it;
   }

   if ((itLastMove != mMsgList.End()) && (touchMsg->GetState() == TouchMsgState::Move))
   {
      TouchMsg* queuedMsg = Courier::message_cast<TouchMsg*>((*itLastMove).GetMessage());

      if (queuedMsg != 0)
      {
         queuedMsg->SetXPos(touchMsg->GetXPos());
         queuedMsg->SetYPos(touchMsg->GetYPos());
         queuedMsg->SetTimeStamp(touchMsg->GetTimeStamp());
         TriggerMessageReceiverWakeup();
         return true;
      }
   }

   bool rc = mMsgList.Append(msgRef);

   TriggerMessageReceiverWakeup();

   return rc;
}


// ------------------------------------------------------------------------
size_t TouchEventPreprocessor::GetSize() const
{
   // avoid changing const interface, therefore cast const away to access CS
   TouchEventPreprocessor* non_const_this = const_cast<TouchEventPreprocessor*>(this);
   Platform::CriticalSectionLocker csLock(&(non_const_this->mMsgListCs));

   return mMsgList.GetSize();
}


// ------------------------------------------------------------------------
bool TouchEventPreprocessor::Exists(const TouchMsg* touchMsg) const
{
   // avoid changing const interface, therefore cast const away to access CS
   TouchEventPreprocessor* non_const_this = const_cast<TouchEventPreprocessor*>(this);
   Platform::CriticalSectionLocker csLock(&(non_const_this->mMsgListCs));

   MessageList::ConstIterator it = mMsgList.Begin();
   while (it != mMsgList.End())
   {
      TouchMsg* queuedMsg = Courier::message_cast<TouchMsg*>((*it).GetMessage());
      COURIER_DEBUG_ASSERT(queuedMsg != 0);
      if (queuedMsg != 0)
      {
         if (touchMsg->GetPointerId() == queuedMsg->GetPointerId())
         {
            TouchMsgState::Enum tState = touchMsg->GetState();
            TouchMsgState::Enum qState = queuedMsg->GetState();
            if ((tState == TouchMsgState::Down) && (qState == TouchMsgState::Down))
            {
               return true;
            }
            if ((tState == TouchMsgState::Up) && (qState == TouchMsgState::Up))
            {
               return true;
            }
            if ((tState == TouchMsgState::Move) && (qState == TouchMsgState::Move))
            {
               if ((queuedMsg->GetXPos() == touchMsg->GetXPos()) && (queuedMsg->GetYPos() == touchMsg->GetYPos()))
               {
                  return true;
               }
            }
         }
      }
      ++it;
   }
   return false;
}


// ------------------------------------------------------------------------
void TouchEventPreprocessor::TriggerMessageReceiverWakeup()
{
   bool wakeup = false;

   if (mOccurredTouchEvents > mTouchEventThreshold)
   {
      wakeup = true;
      mOccurredTouchEvents = 0;
   }

   // wakeup when there is only one motion event in queue as it seemed to happen that down was already processed
   // and thread fall asleep again before motion event was pushed
   if(mOccurredTouchEvents == 1)
   {
      MessageList::Iterator it = mMsgList.Begin();
      if(it != mMsgList.End())
      {
         TouchMsg* queuedMsg = Courier::message_cast<TouchMsg*>((*it).GetMessage());
         if((queuedMsg != 0) && (queuedMsg->GetState() == TouchMsgState::Move))
         {
            wakeup = true;
         }
      }
   }

   if (! wakeup)
   {
      MessageList::Iterator it = mMsgList.Begin();
      while (it != mMsgList.End())
      {
         TouchMsg* queuedMsg = Courier::message_cast<TouchMsg*>((*it).GetMessage());
         COURIER_DEBUG_ASSERT(queuedMsg != 0);
         if (queuedMsg != 0)
         {
            TouchMsgState::Enum qState = queuedMsg->GetState();
            if ((qState == TouchMsgState::Down) || (qState == TouchMsgState::Up))
            {
               wakeup = true;
               break;
            }
         }
         ++it;
      }
   }
   if (wakeup)
   {
      Internal::ComponentWakeupHelper<ComponentId(ComponentType::View)>::Wakeup();
   }
}


// ------------------------------------------------------------------------
void TouchEventPreprocessor::SetWakeupTouchEventThreshold(UInt32 touchEventThreshold)
{
   // avoid changing const interface, therefore cast const away to access CS
   TouchEventPreprocessor* non_const_this = const_cast<TouchEventPreprocessor*>(this);
   Platform::CriticalSectionLocker csLock(&(non_const_this->mMsgListCs));
   mTouchEventThreshold = touchEventThreshold;
}


}
}
