/* ***************************************************************************************
* FILE:          WaylandInputHandler.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandInputHandler.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(InputHandling_Wayland_WaylandInputHandler_h)
#define InputHandling_Wayland_WaylandInputHandler_h

#include <View/CGI/InputHandling/InputHandler.h>
#include <View/CGI/InputHandling/Wayland/WaylandContext.h>
#include <Courier/Platform/CriticalSection.h>
#include <Courier/Platform/CriticalSectionLocker.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <CanderaPlatform/Device/Common/Base/Display.h>

// Forward declarations
namespace Courier {
class Message;
}


//lint info 1768 - has an access (public) different from the access (protected) in the base class
//lint -esym(1768, Courier::InputHandling::Wayland::UnloadSynchronizedDisplay::UploadInternal)
//lint -esym(1768, Courier::InputHandling::Wayland::UnloadSynchronizedDisplay::UnloadInternal)
//lint -esym(1768, Courier::InputHandling::Wayland::Internal::DisplayHelper::UnloadInternal)
//lint -esym(1768, Courier::InputHandling::Wayland::Internal::DisplayHelper::UploadInternal)
namespace Courier {
namespace InputHandling {
namespace Wayland {

class WaylandEventHook;

namespace Internal {
class DisplayHelper : public Candera::Display
{
   public:
      virtual ~DisplayHelper() {}
      virtual void UnloadInternal() = 0;
      virtual bool UploadInternal(const Candera::Display::CommonSettings& settings) = 0;
};


}

class UnloadSynchronizedDisplay : public Candera::Display
{
   public:
      static Candera::Display* CreateDisplay(Int displayId)
      {
         Candera::Display* display = Candera::DevicePackageInterface::CreateDisplay(displayId);
         return (0 != display) ? FEATSTD_NEW(UnloadSynchronizedDisplay)(display) : 0;
      }

      virtual ~UnloadSynchronizedDisplay()
      {
         if (0 != mDisplay)
         {
            FEATSTD_DELETE(static_cast<Courier::InputHandling::Wayland::Internal::DisplayHelper*>(mDisplay));
            mDisplay = 0;
         }
      }

      virtual bool AttachNativeHandle(Int intendedWidth, Int intendedHeight, void* handle)
      {
         return (0 != mDisplay) && mDisplay->AttachNativeHandle(intendedWidth, intendedHeight, handle);
      }

      virtual Candera::SimulatedDisplay* ToSimulatedDisplay()
      {
         return (0 != mDisplay) ? mDisplay->ToSimulatedDisplay() : 0;
      }

      virtual void* GetNativeHandle()
      {
         return (0 != mDisplay) ? mDisplay->GetNativeHandle() : 0;
      }

      virtual bool IsValid()
      {
         return (0 != mDisplay) && mDisplay->IsValid();
      }

      virtual bool UploadInternal(const CommonSettings& settings)
      {
         mUploaded = true;
         return (0 != mDisplay) && static_cast<Courier::InputHandling::Wayland::Internal::DisplayHelper*>(mDisplay)->UploadInternal(settings);
      }

      virtual void UnloadInternal()
      {
         mUploaded = false;
         if (0 != mDisplay)
         {
            Courier::Platform::CriticalSectionLocker lock(&mCriticalSection);
            static_cast<Courier::InputHandling::Wayland::Internal::DisplayHelper*>(mDisplay)->UnloadInternal();
         }
      }

      const Courier::Platform::CriticalSection& GetCriticalSection()
      {
         return mCriticalSection;
      }

      Candera::Display* GetDelegateDisplay()
      {
         return mDisplay;
      }

      bool IsDelegateDisplayUploaded() const
      {
         return mUploaded;
      }

   private:
      explicit UnloadSynchronizedDisplay(Candera::Display* display) :
         Display((0 != display) ? display->GetDisplayId() : 0),
         mUploaded(false),
         mDisplay(display)
      {
         FEATSTD_DEBUG_ASSERT(display != 0);
      }

      bool mUploaded;
      Candera::Display* mDisplay;
      Courier::Platform::CriticalSection mCriticalSection;
};


/// @addtogroup COURIER_PLATFORM_WAYLAND
/// @{
class WaylandInputHandler : public InputHandler
{
      typedef InputHandler Base;
   public:
      ///
      WaylandInputHandler();

      ///
      virtual ~WaylandInputHandler() {}

      ///
      bool InitContext();

      ///
      virtual bool Listen();

      ///
      const WaylandContext& GetContext() const
      {
         return mContext;
      }

   protected:
      ///
      virtual bool LockDisplayForListen()
      {
         return true;
      }
      ///
      virtual void UnlockDisplayForListen() { }

   private:
      WaylandContext mContext;
};


/// @}
}


}
}


#endif
