/* ***************************************************************************************
* FILE:          SurfaceInputRegionManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SurfaceInputRegionManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "SurfaceInputRegion.h"
#include "SurfaceInputRegionManager.h"

#include <View/CGI/CgiExtensions/SurfaceUtils.h>
#include <View/CGI/CgiExtensions/ViewScene2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/SurfaceInputRegionManager.cpp.trc.h>
#endif

namespace hmibase {
namespace input {


/*****************************************************************************/
class SurfaceInputRegionContainer
{
   public:
      /*****************************************************************************/
      ~SurfaceInputRegionContainer()
      {
         for (MapType::iterator it = _map.begin(); it != _map.end(); ++it)
         {
            SurfaceInputRegion* inputRegion = it->second;
            if (inputRegion != NULL)
            {
               SurfaceInputRegion::destroy(inputRegion);
            }
         }
      }

      /*****************************************************************************/
      SurfaceInputRegion* getOrCreate(Candera::RenderTarget* rt)
      {
         //get the existing entry or insert a new one if no entry was there already
         SurfaceInputRegion*& inputRegion = _map[rt];
         if (inputRegion == NULL)
         {
            inputRegion = SurfaceInputRegion::create(rt);
         }
         return inputRegion;
      }

      /*****************************************************************************/
      void flushAll()
      {
         for (MapType::iterator it = _map.begin(); it != _map.end(); ++it)
         {
            SurfaceInputRegion* inputRegion = it->second;
            if (inputRegion != NULL)
            {
               inputRegion->flush();
               SurfaceInputRegion::destroy(inputRegion);
            }
         }
         _map.clear();
      }

   private:
      typedef std::map<const Candera::RenderTarget*, SurfaceInputRegion*> MapType;
      MapType _map;
};


/*****************************************************************************/
class SurfaceInputRegionViewConfig
{
   public:
      typedef SurfaceInputRegionViewType ViewType;
      typedef SurfaceInputRegionWidgetType WidgetType;

      /*****************************************************************************/
      SurfaceInputRegionViewConfig(ViewType* view) : _view(view), _isEntireViewportTouchable(true)
      {
      }

      /*****************************************************************************/
      ~SurfaceInputRegionViewConfig()
      {
         _view = NULL;
      }

      /*****************************************************************************/
      ViewType* getView() const
      {
         return _view;
      }

      /*****************************************************************************/
      bool isEntireViewTouchable() const
      {
         return _isEntireViewportTouchable;
      }

      /*****************************************************************************/
      void setEntireViewTouchable(bool value)
      {
         _isEntireViewportTouchable = value;
      }

      /*****************************************************************************/
      void clearInputRegionWidgets()
      {
         if (_widgets.size() > 0)
         {
            ETG_TRACE_USR4_THR(("clearInputRegionWidgets view=%s", _view != NULL ? _view->GetId().CStr() : "<null>"));
         }
         _widgets.clear();
      }

      /*****************************************************************************/
      void addInputRegionWidget(WidgetType* widget, SurfaceInputRegionWidgetConfig* widgetConfig)
      {
         if ((widget != NULL) && (widgetConfig != NULL))
         {
            ETG_TRACE_USR4_THR(("addInputRegionWidget widget=%s", widget->GetLegacyName()));

            _widgets.push_back(WidgetTuple(widget, widgetConfig));
         }
      }

      /*****************************************************************************/
      void processInputRegionWidgets(SurfaceInputRegionContainer& inputRegionContainer);

   private:
      ViewType* _view;
      bool _isEntireViewportTouchable;

      typedef hmibase::util::ObjectAccessor<WidgetType> WidgetAccessor;
      struct WidgetTuple
      {
         WidgetTuple(WidgetType* widget, SurfaceInputRegionWidgetConfig* widgetConfig) : _widget(widget), _widgetConfig(widgetConfig)
         {
         }

         ~WidgetTuple()
         {
            _widgetConfig = NULL;
         }

         WidgetAccessor _widget;
         SurfaceInputRegionWidgetConfig* _widgetConfig;
      };

      typedef std::vector<WidgetTuple> WidgetsType;
      WidgetsType _widgets;
};


/*****************************************************************************/


/*****************************************************************************/
void SurfaceInputRegionViewConfig::processInputRegionWidgets(SurfaceInputRegionContainer& inputRegionContainer)
{
   Courier::ViewScene2D* view = (_view != NULL) ? _view->ToViewScene2D() : NULL;
   if (view == NULL)
   {
      ETG_TRACE_USR1_THR(("processInputRegionWidgets invalid view=%s (only 2D views are supported)",
                          (_view != NULL) ? _view->GetId().CStr() : "<null>"));
      return;
   }

   ETG_TRACE_USR4_THR(("processInputRegionWidgets entireViewportTouchable=%u, count=%u, view=%s",
                       _isEntireViewportTouchable, _widgets.size(), view->GetId().CStr()));

   Courier::ViewScene2D::CameraPtrVector& cameras(view->GetCameraPtrVector());
   for (size_t i = 0; i < cameras.Size(); ++i)
   {
      Candera::Camera2D* camera = cameras[i];
      if ((camera != NULL) && (camera->GetRenderTarget() != NULL))
      {
         Candera::RenderTarget* rt = camera->GetRenderTarget();
         SurfaceInputRegion* inputRegion = inputRegionContainer.getOrCreate(rt);
         if (inputRegion != NULL)
         {
            //entire viewport is touchable=>use camera viewport as input region
            if (_isEntireViewportTouchable)
            {
               Candera::Rectangle viewport(camera->GetViewport());
               if (viewport.GetWidth() <= 0.0f)
               {
                  viewport.SetWidth(static_cast<FeatStd::Float>(rt->GetWidth()));
               }
               if (viewport.GetHeight() <= 0.0f)
               {
                  viewport.SetHeight(static_cast<FeatStd::Float>(rt->GetHeight()));
               }

               inputRegion->add(static_cast<int>(viewport.GetLeft()), static_cast<int>(viewport.GetTop()), static_cast<int>(viewport.GetWidth()), static_cast<int>(viewport.GetHeight()));
            }

            for (WidgetsType::iterator it = _widgets.begin(); it != _widgets.end(); ++it)
            {
               WidgetType* widget = (*it)._widget.getObjectSafely();
               SurfaceInputRegionWidgetConfig* widgetConfig = (*it)._widgetConfig;
               if ((widget != NULL) && (widget->GetParentView() == view) && (widget->IsEffectiveVisible())
                     && (widgetConfig != NULL) && (widgetConfig->getInputRegionOperation() != hmibase::enSurfaceInputRegionOperation::Ignore))
               {
                  Candera::Rectangle rect(widgetConfig->getInputRegionRectangle());

                  Candera::Vector2 coord(rect.GetPosition());
                  hmibase::view::util::SurfaceUtils::transformSceneToSurface(*camera, coord);
                  rect.SetPosition(coord);

                  if (widgetConfig->getInputRegionOperation() == hmibase::enSurfaceInputRegionOperation::Add)
                  {
                     inputRegion->add(static_cast<int>(rect.GetLeft()), static_cast<int>(rect.GetTop()), static_cast<int>(rect.GetWidth()), static_cast<int>(rect.GetHeight()));
                  }
                  else //if (widgetConfig->getInputRegionOperation() == hmibase::enSurfaceInputRegionOperation::Subtract)
                  {
                     inputRegion->subtract(static_cast<int>(rect.GetLeft()), static_cast<int>(rect.GetTop()), static_cast<int>(rect.GetWidth()), static_cast<int>(rect.GetHeight()));
                  }
               }
            }
         }
      }
   }
}


/*****************************************************************************/
SurfaceInputRegionManager::SurfaceInputRegionManager()
{
}


/*****************************************************************************/
SurfaceInputRegionManager::~SurfaceInputRegionManager()
{
}


/*****************************************************************************/
SurfaceInputRegionManager& SurfaceInputRegionManager::getInstance()
{
   static SurfaceInputRegionManager _instance;
   return _instance;
}


#ifdef VARIANT_S_FTR_ENABLE_SURFACE_INPUT_REGION_MANAGER
/*****************************************************************************/
SurfaceInputRegionViewConfig* SurfaceInputRegionManager::getViewConfig(ViewType* view)
{
   if (view != NULL)
   {
      ViewConfigMapType::iterator it = _allViewConfigs.find(view);
      if (it != _allViewConfigs.end())
      {
         return it->second;
      }
   }

   return NULL;
}


/*****************************************************************************/
SurfaceInputRegionViewConfig* SurfaceInputRegionManager::registerView(ViewType* view)
{
   if (view != NULL)
   {
      SurfaceInputRegionViewConfig*& viewConfig = _allViewConfigs[view];
      if (viewConfig == NULL)
      {
         ETG_TRACE_USR4_THR(("registerView view=%s", view->GetId().CStr()));

         viewConfig = FEATSTD_NEW(SurfaceInputRegionViewConfig)(view);
      }
      return viewConfig;
   }

   return NULL;
}


/*****************************************************************************/
void SurfaceInputRegionManager::unregisterView(ViewType* view)
{
   if (view != NULL)
   {
      //find and erase from active views
      ViewConfigMapType::iterator it = _activeViewConfigs.find(view);
      if (it != _activeViewConfigs.end())
      {
         _activeViewConfigs.erase(it);
      }

      //find and erase from all views
      it = _allViewConfigs.find(view);
      if (it != _allViewConfigs.end())
      {
         ETG_TRACE_USR4_THR(("unregisterView view=%s", view->GetId().CStr()));

         if (it->second != NULL)
         {
            FEATSTD_DELETE(it->second);
         }
         _allViewConfigs.erase(it);
      }
   }
}


/*****************************************************************************/
void SurfaceInputRegionManager::onViewActivated(ViewType* view, bool isEntireViewportTouchable)
{
   if (view != NULL)
   {
      SurfaceInputRegionViewConfig* viewConfig = getViewConfig(view);
      if (viewConfig != NULL)
      {
         ETG_TRACE_USR4_THR(("onViewActivated entireViewportTouchable=%u, view=%s", isEntireViewportTouchable, view->GetId().CStr()));

         viewConfig->setEntireViewTouchable(isEntireViewportTouchable);
         _activeViewConfigs[view] = viewConfig;
      }
   }
}


/*****************************************************************************/
void SurfaceInputRegionManager::onViewDeactivated(ViewType* view)
{
   if (view != NULL)
   {
      ViewConfigMapType::iterator it = _activeViewConfigs.find(view);
      if (it != _activeViewConfigs.end())
      {
         ETG_TRACE_USR4_THR(("onViewDeactivated view=%s", view->GetId().CStr()));

         _activeViewConfigs.erase(it);
      }
   }
}


/*****************************************************************************/
void SurfaceInputRegionManager::clearTouchableWidgets(ViewType* view)
{
   SurfaceInputRegionViewConfig* viewConfig = getViewConfig(view);
   if (viewConfig != NULL)
   {
      viewConfig->clearInputRegionWidgets();
   }
}


/*****************************************************************************/
void SurfaceInputRegionManager::addInputRegionWidget(WidgetType* widget, SurfaceInputRegionWidgetConfig* widgetConfig)
{
   if ((widget != NULL) && (widgetConfig != NULL))
   {
      SurfaceInputRegionViewConfig* viewConfig = getViewConfig(widget->GetParentView());
      if (viewConfig != NULL)
      {
         viewConfig->addInputRegionWidget(widget, widgetConfig);
      }
   }
}


/*****************************************************************************/
void SurfaceInputRegionManager::sendInputRegions()
{
   ETG_TRACE_USR4_THR(("sendInputRegions app=%s", hmibase::trace::getAppName().c_str()));

   SurfaceInputRegionContainer inputRegionContainer;
   for (ViewConfigMapType::iterator it = _activeViewConfigs.begin(); it != _activeViewConfigs.end(); ++it)
   {
      SurfaceInputRegionViewConfig* viewConfig = it->second;
      if (viewConfig != NULL)
      {
         viewConfig->processInputRegionWidgets(inputRegionContainer);
      }
   }
   inputRegionContainer.flushAll();
}


#else
/*****************************************************************************/
SurfaceInputRegionViewConfig* SurfaceInputRegionManager::getViewConfig(ViewType*)
{
   return NULL;
}


SurfaceInputRegionViewConfig* SurfaceInputRegionManager::registerView(ViewType*)
{
   return NULL;
}


void SurfaceInputRegionManager::unregisterView(ViewType*)
{
}


void SurfaceInputRegionManager::onViewActivated(ViewType*, bool)
{
}


void SurfaceInputRegionManager::onViewDeactivated(ViewType*)
{
}


void SurfaceInputRegionManager::clearTouchableWidgets(ViewType*)
{
}


void SurfaceInputRegionManager::addInputRegionWidget(WidgetType*, SurfaceInputRegionWidgetConfig*)
{
}


void SurfaceInputRegionManager::sendInputRegions()
{
}


#endif
}


}
