/* ***************************************************************************************
* FILE:          SurfaceInputRegion.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SurfaceInputRegion is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "SurfaceInputRegion.h"

#ifdef VARIANT_S_FTR_ENABLE_SURFACE_INPUT_REGION_MANAGER
#include <wayland-client.h>
#include <CanderaPlatform/Device/Genivi/GeniviWindowSurface.h>
#include <CanderaPlatform/Device/Genivi/Target/ILM/GeniviTargetDisplay.h>
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/SurfaceInputRegion.cpp.trc.h>
#endif

namespace hmibase {
namespace input {

/*****************************************************************************/
SurfaceInputRegion::SurfaceInputRegion(unsigned int surfaceId, wl_surface* wlSurface, wl_region* wlRegion) : _surfaceId(surfaceId), _wlSurface(wlSurface), _wlRegion(wlRegion)
{
}


/*****************************************************************************/
void SurfaceInputRegion::destroy(SurfaceInputRegion* handle)
{
   if (handle != NULL)
   {
      ETG_TRACE_USR4_THR(("destroy input region handle for surface=%u, app=%s",
                          handle->getSurfaceId(), hmibase::trace::getAppName().c_str()));
      FEATSTD_DELETE(handle);
   }
}


#ifdef VARIANT_S_FTR_ENABLE_SURFACE_INPUT_REGION_MANAGER
/*****************************************************************************/
SurfaceInputRegion::~SurfaceInputRegion()
{
   _wlSurface = NULL;

   if (_wlRegion != NULL)
   {
      wl_region_destroy(_wlRegion);
      _wlRegion = NULL;
   }
}


/*****************************************************************************/
SurfaceInputRegion* SurfaceInputRegion::create(Candera::RenderTarget* renderTarget)
{
   Candera::GraphicDeviceUnit* gdu = (renderTarget != NULL) ? renderTarget->GetGraphicDeviceUnit() : NULL;
   if (gdu == NULL)
   {
      ETG_TRACE_ERR_THR(("create - null rendertarget (%p) or gdu (%p) for app=%s",
                         renderTarget, gdu, hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   if (gdu->ToWindow() == NULL)
   {
      ETG_TRACE_USR4_THR(("render target '%50s' is not a window surface for app=%s",
                          gdu->GetName(), hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   Candera::GeniviSurface* geniviSurface = dynamic_cast<Candera::GeniviSurface*>(gdu->ToWindow());
   wl_surface* wlSurface = (geniviSurface != NULL) ? geniviSurface->GetWaylandSurface() : NULL;
   if (wlSurface == NULL)
   {
      ETG_TRACE_ERR_THR(("create - null window surface (%p), properties (%p) or native handle (%p) for displayId=%d, gdu=%30s, app=%s",
                         geniviSurface, (geniviSurface != NULL) ? geniviSurface->GetProperties() : NULL,
                         (geniviSurface != NULL) ? geniviSurface->GetWaylandSurface() : NULL,
                         gdu->GetDisplay(), gdu->GetName(), hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   Candera::GeniviTargetDisplay* geniviDisplay = dynamic_cast<Candera::GeniviTargetDisplay*>(Candera::GeniviDevicePackageInterface::GetInstance().GetDisplay(gdu->GetDisplay()));
   wl_compositor* wlCompositor = ((geniviDisplay != NULL) && (geniviDisplay->GetWaylandContext() != NULL)) ? geniviDisplay->GetWaylandContext()->GetCompositor() : NULL;
   if (wlCompositor == NULL)
   {
      ETG_TRACE_ERR_THR(("create - null display (%p), context (%p) or compositor (%p) for displayId=%d, gdu=%30s, app=%s",
                         geniviDisplay, (geniviDisplay != NULL) ? geniviDisplay->GetWaylandContext() : NULL, wlCompositor,
                         gdu->GetDisplay(), gdu->GetName(), hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   wl_region* wlRegion = wl_compositor_create_region(wlCompositor);
   if (wlRegion == NULL)
   {
      ETG_TRACE_ERR_THR(("create - failed to create input region for displayId=%d, gdu=%30s, app=%s",
                         gdu->GetDisplay(), gdu->GetName(), hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   unsigned int surfaceId = geniviSurface->GetProperties()->GetSurfaceId();
   ETG_TRACE_USR4_THR(("create input region handle for surface=%u, app=%s",
                       surfaceId, hmibase::trace::getAppName().c_str()));

   return FEATSTD_NEW(SurfaceInputRegion)(surfaceId, wlSurface, wlRegion);
}


/*****************************************************************************/
void SurfaceInputRegion::flush()
{
   if ((_wlSurface != NULL) && (_wlRegion != NULL))
   {
      ETG_TRACE_USR4_THR(("input region flush for surfaceId=%u, app=%s", _surfaceId, hmibase::trace::getAppName().c_str()));

      wl_surface_set_input_region(_wlSurface, _wlRegion);
      wl_region_destroy(_wlRegion);
   }
   _wlSurface = NULL;
   _wlRegion = NULL;
}


/*****************************************************************************/
void SurfaceInputRegion::add(int x, int y, int width, int height)
{
   if (_wlRegion != NULL)
   {
      ETG_TRACE_USR4_THR(("input region add [%d,%d,%d,%d] for surfaceId=%u, app=%s",
                          x, y, width, height, _surfaceId, hmibase::trace::getAppName().c_str()));

      wl_region_add(_wlRegion, x, y, width, height);
   }
}


/*****************************************************************************/
void SurfaceInputRegion::subtract(int x, int y, int width, int height)
{
   if (_wlRegion != NULL)
   {
      ETG_TRACE_USR4_THR(("input region subtract [%d,%d,%d,%d] for surfaceId=%u, app=%s",
                          x, y, width, height, _surfaceId, hmibase::trace::getAppName().c_str()));

      wl_region_subtract(_wlRegion, x, y, width, height);
   }
}


#else
/*****************************************************************************/
SurfaceInputRegion::~SurfaceInputRegion()
{
   _wlSurface = NULL;
   _wlRegion = NULL;
}


/*****************************************************************************/
SurfaceInputRegion* SurfaceInputRegion::create(Candera::RenderTarget* renderTarget)
{
   Candera::GraphicDeviceUnit* gdu = (renderTarget != NULL) ? renderTarget->GetGraphicDeviceUnit() : NULL;
   if (gdu == NULL)
   {
      ETG_TRACE_ERR_THR(("create - null rendertarget (%p) or gdu (%p) for app=%s",
                         renderTarget, gdu, hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   if (gdu->ToWindow() == NULL)
   {
      ETG_TRACE_USR4_THR(("render target '%50s' is not a window surface for app=%s",
                          gdu->GetName(), hmibase::trace::getAppName().c_str()));
      return NULL;
   }

   ETG_TRACE_USR4_THR(("create simulation input region handle for rt=%50s, app=%s",
                       gdu->GetName(), hmibase::trace::getAppName().c_str()));

   return FEATSTD_NEW(SurfaceInputRegion)(0, NULL, NULL);
}


/*****************************************************************************/
void SurfaceInputRegion::flush()
{
   ETG_TRACE_USR4_THR(("simulate input region flush for app=%s", hmibase::trace::getAppName().c_str()));
}


/*****************************************************************************/
void SurfaceInputRegion::add(int x, int y, int width, int height)
{
   ETG_TRACE_USR4_THR(("simulate input region add [%d,%d,%d,%d] for app=%s",
                       x, y, width, height, hmibase::trace::getAppName().c_str()));
}


/*****************************************************************************/
void SurfaceInputRegion::subtract(int x, int y, int width, int height)
{
   ETG_TRACE_USR4_THR(("simulate input region subtract [%d,%d,%d,%d] for app=%s",
                       x, y, width, height, hmibase::trace::getAppName().c_str()));
}


#endif
}


}
