/* ***************************************************************************************
* FILE:          DisplayInputContext.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DisplayInputContext.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "Courier/Version.h"
#include "DisplayInputContext.h"
#include <Courier/Diagnostics/Log.h>
#include <View/CGI/InputHandling/InputHandler.h>
#include <View/CGI/InputHandling/InputThread.h>
#include <CanderaPlatform/Device/Common/Base/Display.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>


#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DisplayInputContext.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace Courier {
namespace InputHandling {
///
DisplayInputContext::DisplayInputContext() : mInputHandler(0), mInputThread(0), mDisplayConfiguration(0), mDisplay(0), mContextResourcePool(0), mNativeHandle(0)
{
}


// ------------------------------------------------------------------------
void DisplayInputContext::SetNativeHandle(void* nativeHandle)
{
   if (0 == mNativeHandle)
   {
      mNativeHandle = nativeHandle;
   }
   else
   {
      ETG_TRACE_ERR_THR(("Native handle was already set and cannot be changed!"));
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::SetDisplay(Candera::Display* display)
{
   if (0 == mDisplay)
   {
      mDisplay = display;
   }
   else
   {
      ETG_TRACE_ERR_THR(("Display was already set and cannot be changed!"));
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::SetDisplayConfiguration(DisplayConfiguration* displayConfiguration)
{
   if (0 == mDisplayConfiguration)
   {
      mDisplayConfiguration = displayConfiguration;
   }
   else
   {
      ETG_TRACE_ERR_THR(("Display configuration was already set and cannot be changed!"));
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::SetInputHandler(InputHandler* inputHandler,
      void* inputEventHook)
{
   if (0 == mInputHandler)
   {
      mInputHandler = inputHandler;
      if (0 != mInputHandler)
      {
         // Use input handler only with blocking mode
         mInputHandler->SetMode(Courier::InputHandling::InputHandlerMode::Blocking);
         // Set application specific input event handling for window events
         mInputHandler->SetEventHook(inputEventHook);
         // Set input context relation in input handler
         mInputHandler->SetInputContext(this);
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("Input handler was already set and cannot be changed!"));
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::SetInputThread(InputThread* inputThread,
      const Char* inputThreadName,
      InputThreadHook* inputThreadHook)
{
   if (0 == mInputThread)
   {
      mInputThread = inputThread;
      if (0 != mInputThread)
      {
         if (0 != inputThreadName)
         {
            mInputThread->SetName(inputThreadName);
         }
         (void)mInputThread->SetHook(inputThreadHook);
         FEATSTD_LINT_NEXT_EXPRESSION(925, "pointer to pointer cast is required here")
         (void)mInputThread->SetData(reinterpret_cast<void*>(this));
         mInputThread->SetInputContext(this);
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("Input thread was already set and cannot be changed!"));
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::Run()
{
   FEATSTD_DEBUG_ASSERT(0 != mInputThread);
   if (0 != mInputThread)
   {
      if (!(mInputThread->Run()))
      {
         ETG_TRACE_ERR_THR(("Starting of input handler thread '%s' failed!", mInputThread->GetName()));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("Calling Run() without having input thread set!"));
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::Terminate()
{
   if (0 != mInputThread)
   {
      mInputThread->Terminate();
      mInputThread = 0;
   }
}


// ------------------------------------------------------------------------
void DisplayInputContext::Reset()
{
   Terminate();
   if (0 != mInputHandler)
   {
      mInputHandler->Reset();
      mInputHandler = 0;
   }
   if (0 != mDisplay)
   {
      Int lId = mDisplay->GetDisplayId();
      // Destroy display
      Candera::DevicePackageInterface::DestroyDisplay(mDisplay);
      // Destroy only the created resource context pool (displayId != 0)
      if (0 != lId)
      {
         FEATSTD_SAFE_DELETE(mContextResourcePool);
      }
      mDisplay = 0;
   }
   mDisplayConfiguration = 0;
   mNativeHandle = 0;
}


}
}
