/* ***************************************************************************************
* FILE:          AppEnvironmentPlatform.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppEnvironmentPlatform.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#ifdef WIN32
# error This is the Target/LSIM implementation
#endif

#include <FeatStd/Diagnostics/ErrorHandling.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <View/CGI/TouchHandling/TouchEventPreprocessor.h>

#include "AppEnvironmentPlatform.h"

#include "View/CGI/CgiExtensions/MemoryMappedAssetRepository.h"

#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"


#include "AppBase/IApplicationSettings.h"

#if defined(FEATSTD_DLT_ENABLED)
#include "FeatStd/Diagnostics/DltAppender.h"
#endif
#include "Trace/TTFisAppender.h"

#include "View/CGI/CgiExtensions/DataBindingRealm.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_CGI
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AppEnvironmentPlatform.cpp.trc.h"
#endif

// Test define for 2nd display support
//#define HAS_2ND_DISPLAY 1

// Test define for terminal input support
//#define HAS_TERMINAL_INPUT

//! [AssetRepository]
#include "View/CGI/CgiExtensions/MemoryMappedAssetRepository.h"


//! [AssetRepository]

namespace AppPlatform {
namespace Impl {


AppEnvironment::AppEnvironment(const IApplicationSettings& settings): Base(settings)
{
   mWindowInputEventHook.initHook(&(getKeyMapping()));
   mWindowInputEventHook.setUserInputObserver(_appSettings.getUserInputObserver());

   SetupTraceForFramework(settings);
}


void AppEnvironment::SetupTraceForFramework(const IApplicationSettings& settings)
{
   FeatStd::Diagnostics::LogControl::SetAppender(0);

   // set default trace level to FATAL, use TTFis command HMI_SET_CGITRACELEVEL to adopt it later on
   // this is majorly required to improve performance
   struct stat buf;
   if (stat("/opt/bosch/hmibase/cgi_debug.txt", &buf) == 0)
   {
      ETG_TRACE_FATAL_THR(("CGI Debug traces enabled"));
      Courier::Diagnostics::LogControl::SetLogLevel(Courier::Diagnostics::LogLevel::Debug);
   }
   else
   {
      Courier::Diagnostics::LogControl::SetLogLevel(Courier::Diagnostics::LogLevel::Fatal);
   }

   if (getenv("CGI_DEBUG_FILE") != NULL)
   {
      FeatStd::Diagnostics::LogControl::SetLogLevel(Courier::Diagnostics::LogLevel::Debug);

      char buffer[100];
      snprintf(buffer, sizeof(buffer), "/var/opt/bosch/dynamic/hmi/cgi_debug_%s.txt", hmibase::trace::getAppName().c_str());

      static FeatStd::Diagnostics::FileAppender fileAppender;
      if (fileAppender.Open(buffer))
      {
         FeatStd::Diagnostics::LogControl::AppendAppender(&(fileAppender));
         ETG_TRACE_FATAL_THR(("CGI Debug traces to file enabled"));
      }
      else
      {
         ETG_TRACE_FATAL_THR(("fopen %s failed", buffer));
      }
   }
   else if (getenv("CGI_DEBUG_CONSOLE") != NULL)
   {
      FeatStd::Diagnostics::LogControl::SetLogLevel(Courier::Diagnostics::LogLevel::Debug);
      FeatStd::Diagnostics::LogControl::AppendAppender(&(FeatStd::Diagnostics::ConsoleAppender::GetInstance()));
      ETG_TRACE_FATAL_THR(("CGI Debug traces to console enabled"));
   }

   // Setup Widgets trace
   hmibase::trace::widget::setUserTraceClass(_appSettings.getCourierTraceGroup().traceClass_Widget);
   ::hmibase::view::DataBindingRealm::setTraceClass(_appSettings.getCourierTraceGroup().traceClass_CourierPayload_DataBinding);
   Courier::Diagnostics::TraceControl::SetEnabled(false);

#if defined(FEATSTD_DLT_ENABLED)
   std::string str = hmibase::trace::getAppName();
   size_t pos = str.find("_");
   std::string AppId;
   if (pos < 0)
   {
      AppId = str.substr(0, 4);
   }
   else
   {
      AppId = (++pos < str.length()) ? str.substr(pos, 4) : str.substr(0, 4);
   }
   FeatStd::Diagnostics::DltAppender& lDltAppender = FeatStd::Diagnostics::DltAppender::GetInstance();
   lDltAppender.Register((ScreenBroker::Char*)(AppId.c_str()), (ScreenBroker::Char*)(HmiAsfComponentBase::getAppName()));
   FeatStd::Diagnostics::LogControl::AppendAppender(&lDltAppender);
#else
   // Setup appender for TTFIS
   hmibase::trace::TTFisAppender& ttfisAppender = hmibase::trace::TTFisAppender::GetInstance();
   ttfisAppender.setTraceClasses(settings.getCourierTraceGroup());
   FeatStd::Diagnostics::LogControl::AppendAppender(&ttfisAppender);
#endif
}


bool AppEnvironment::BeforeInputContextSetup()
{
   bool lRc = ScreenBrokerClient::GetInstance().Init(static_cast<unsigned int>(getpid()),
              _appSettings.getScreenBrokerSettings().ids.mainSurfaces,
              _appSettings.getScreenBrokerSettings().ids.videoSurfaces,
              _appSettings.getScreenBrokerSettings().ids.popupSurfaces,
              _appSettings.getScreenBrokerSettings().scaling,
              _appSettings.getScreenBrokerSettings().appViewSettings,
//              &(_appSettings.getKeyMapping()),
              & mWindowInputEventHook);
   FEATSTD_DEBUG_ASSERT(lRc);
   ScreenBrokerClient::GetInstance().Synchronize();
   return lRc;
}


void AppEnvironment::SetupInputHandler(int display)
{
   mDisplayInputContexts[display].SetInputHandler(&mMainWindowInputHandler, &mWindowInputEventHook);
}


void AppEnvironment::SetupInputThread(int display)
{
   mDisplayInputContexts[display].SetInputThread(&mMainWindowInputThread, "MainWdwInput", 0);
}


bool AppEnvironment::SyncWindowCreation()
{
   return false;
}


bool AppEnvironment::Cleanup(const std::vector<std::string>& cAssetFileNames)
{
   ScreenBrokerClient::GetInstance().Terminate();

#if !defined(FEATSTD_DLT_ENABLED)
   // Finalize screen broker client logging as it is not done via DltAppender

#endif

   return Base::Cleanup(cAssetFileNames);
}


Candera::AssetRepository* AppEnvironment::GetAssetRepo(const char* assetFileName)
{
   static std::map<std::string, Candera::MemoryMappedAssetRepository*> assetRepos;
   std::map<std::string, Candera::MemoryMappedAssetRepository*>::iterator it = assetRepos.find(assetFileName);

   if (it == assetRepos.end())
   {
      assetRepos[assetFileName] = CANDERA_NEW(Candera::MemoryMappedAssetRepository)(assetFileName);
   }

   return assetRepos.at(assetFileName);
}


void AppEnvironment::SetupTouchPreprocessor()
{
   mWindowInputEventHook.SetTouchEventPreprocessor(&mTouchEventPreprocessor);
}


}
}
