/* ***************************************************************************************
* FILE:          AppEnvironment.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppEnvironment.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "Courier/Visualization/AssetAccessor.h"
#include "AppBase/IApplicationSettings.h"
#include "AppEnvironment.h"
#include "AppBase/ILM_Accessor.h"

#if defined(FEATSTD_DLT_ENABLED)
#include "FeatStd/Diagnostics/DltAppender.h"
#endif

#include "Trace/HMIAlert.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_CGI
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AppEnvironment.cpp.trc.h"
#endif

namespace AppPlatform {

FEATSTD_RTTI_DEFINITION(AppEnvironment::AssetRepositoryEvent, FeatStd::Event)


Courier::InputHandling::DisplayConfiguration AppEnvironment::mDisplayConfigurations[] =
{
   {
      // Main display
      0,                                                      // Display ID in Asset
      "Main window",                                          // Display name
      {
         800, 0, 0, 0, 480, 0, 0, 0, 0, 0, 0.0F, false

         , 0, 0
      },      // Display settings
   }
#if defined(HAS_2ND_DISPLAY)
   ,
   {
      // 2nd display
      0,                                                      // Display ID in Asset
      "2nd window",                                           // Display name
      { 100, 0, 0, 0, 60, 0, 0, 0, 0, 0, 0.0F, false },       // Display settings
   }
#endif
};


static const FeatStd::Int cMainWindow = 0;
#if defined(HAS_2ND_DISPLAY)
static const FeatStd::Int c2ndWindow = 1;
#endif

namespace AppDisplayInputContext {


enum Enum
{
   MainWindow,
#if defined(HAS_2ND_DISPLAY)
   SecondWindow,
#endif
#if defined(HAS_TERMINAL_INPUT)
   Terminal,
#endif
   COUNT
};


}

_TODO("TODO: HAS_2ND_DISPLAY cleanup")

Courier::InputHandling::DisplayInputContext AppEnvironment::mDisplayInputContexts[AppDisplayInputContext::COUNT];

#if defined(FEATSTD_DLT_ENABLED)
AppEnvironment::AppEnvironment(FeatStd::Char* cDltRegistration, FeatStd::Char* cDltApplication, const IApplicationSettings& settings)
#else
AppEnvironment::AppEnvironment(const IApplicationSettings& settings) : _appSettings(settings)
#endif
{
   memset(mDisplayInputContexts, 0, sizeof(mDisplayInputContexts));
}


AppEnvironment::~AppEnvironment()
{
#if defined(FEATSTD_DLT_ENABLED)
   FeatStd::Diagnostics::DltAppender& lDltAppender = FeatStd::Diagnostics::DltAppender::GetInstance();
   lDltAppender.Unregister();
#endif
   Courier::Diagnostics::LogControl::SetAppender(0);
}


FeatStd::EventSource& AppEnvironment::GetEventSource()
{
   FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::EventSource, s_eventSource);
   return s_eventSource;
}


Candera::AssetRepository* AppEnvironment::GetAssetRepo(const char* assetFileName)
{
   static std::map<std::string, Candera::FileAssetRepository*> assetRepos;
   std::map<std::string, Candera::FileAssetRepository*>::iterator it = assetRepos.find(assetFileName);

   if (it == assetRepos.end())
   {
      assetRepos[assetFileName] = CANDERA_NEW(Candera::FileAssetRepository)(assetFileName);
   }

   return assetRepos.at(assetFileName);
}


bool AppEnvironment::Setup(const std::vector<std::string>& cAssetFileNames)
{
   bool lRc = true;

   // ----- Asset configuration
   for (std::vector<std::string>::const_iterator it = cAssetFileNames.begin(); it != cAssetFileNames.end(); ++it)
   {
      const char* assetFileName = (*it).c_str();
      ETG_TRACE_USR1_THR_DCL((_appSettings.getCourierTraceGroup().traceClass_Debug, "ASSET FILE NAME to load : %s", assetFileName));
      Candera::AssetRepository* assetRepo = GetAssetRepo(assetFileName);

      lRc = lRc && (assetRepo != NULL) && mAssetConfiguration.Add(assetRepo);
      if (lRc)
      {
         AssetRepositoryEvent evt(assetRepo, assetFileName, AssetRepositoryEvent::Added);
         GetEventSource().DispatchEvent(evt);
      }
   }
   //lRc = lRc && mAssetConfiguration.Add(&GetAsset(cAssetFileName));
   FEATSTD_DEBUG_ASSERT(lRc);

   // Call dummy call for not let the linker omit the message factory configuration
   //AppPlatform::LoadMessageFactoryConfiguration();

   //! [COURIER_MESSAGE_FACTORY_CONF_CALL]

   // Call setup of platform specific implementation first
   lRc = lRc && BeforeInputContextSetup();
   //bool lRc = Base::Setup(cAssetFileName);
   HMI_ALERT_CONDITIONAL(lRc, "Platform specific setup failed!");

   // ----- Main window event input handling
   // Wire input handler with display and input thread
   mDisplayConfigurations[cMainWindow].displayId = _appSettings.getDisplayParameters().id;
   mDisplayConfigurations[cMainWindow].displaySettings.width = _appSettings.getDisplayParameters().width;
   mDisplayConfigurations[cMainWindow].displaySettings.height = _appSettings.getDisplayParameters().height;
   SetupInputHandler(static_cast<int>(AppDisplayInputContext::MainWindow));
   mDisplayInputContexts[AppDisplayInputContext::MainWindow].SetDisplayConfiguration(&mDisplayConfigurations[cMainWindow]);

   // Check application display configuration against the asset's display configuration
   CheckAssetDisplayConfiguration();

   SetupInputThread(static_cast<int>(AppDisplayInputContext::MainWindow));
   // Create display (via ThreadHook) and start listening input events in separate thread
   mDisplayInputContexts[AppDisplayInputContext::MainWindow].Run();

#if defined(HAS_2ND_DISPLAY)
   // ----- 2nd window event input handling
   // Wire input handler with display and input thread
   SetupInputHandler(AppDisplayInputContext::SecondWindow);
   //mDisplayInputContexts[AppDisplayInputContext::SecondWindow].SetInputHandler(&m2ndWindowInputHandler, &mWindowInputEventHook);
   mDisplayInputContexts[AppDisplayInputContext::SecondWindow].SetDisplayConfiguration(&mDisplayConfigurations[c2ndWindow]);
   SetupInputThread(AppDisplayInputContext::SecondWindow);
   //mDisplayInputContexts[AppDisplayInputContext::SecondWindow].SetInputThread(&m2ndWindowInputThread, "2nd window input", 0);

   // Create display (via ThreadHook) and start listening input events in separate thread
   mDisplayInputContexts[AppDisplayInputContext::SecondWindow].Run();
#endif
   // todo sat2hi
   //mWindowInputEventHook.SetTouchEventPreprocessor(&mTouchEventPreprocessor);
   SetupTouchPreprocessor();

   if (lRc && SyncWindowCreation())
   {
      // Wait until input thread is running and native displays are created and attached to input handler
      while ((0 == mDisplayInputContexts[AppDisplayInputContext::MainWindow].GetNativeHandle())
#if defined(HAS_2ND_DISPLAY)
             || (0 == mDisplayInputContexts[AppDisplayInputContext::SecondWindow].GetNativeHandle())
#endif
            )
      {
         (void)FeatStd::Internal::Thread::Yield();
      }
   }

   // ----- Create all defined displays
   for (FeatStd::Int i = 0; lRc && (i < GetDisplayInputContextCount()); i++)
   {
      // Read and check external input context
      Courier::InputHandling::DisplayInputContext* lInputContext = const_cast<Courier::InputHandling::DisplayInputContext*>(GetDisplayInputContext(i));
      lRc = lRc && (0 != lInputContext);
      HMI_ALERT_CONDITIONAL(lRc, "Invalid input context (%d)", i);

      // Check if a display is attached to external input context
      Courier::InputHandling::DisplayConfiguration* lDisplayConfiguration = lInputContext->GetDisplayConfiguration();
      // ----- Create and initialize Candera display
      if (0 != lDisplayConfiguration)
      {
         // ----- Create Candera display
         Candera::Display* lDisplay = Candera::DevicePackageInterface::CreateDisplay(lDisplayConfiguration->displayId);
         lRc = lRc && (0 != lDisplay);
         HMI_ALERT_CONDITIONAL(lRc, "Failed to create Candera display!");

         // Update display in input context
         lInputContext->SetDisplay(lDisplay);

         // ----- Initialize Candera display
         // Native display was already created (and attached to the input handler) simply attach it to Candera display instance
         if (0 != lInputContext->GetNativeHandle())
         {
            // Attach manually created display to Candera display
            lRc = lRc && lDisplay->AttachNativeHandle(lDisplayConfiguration->displaySettings.width,
                  lDisplayConfiguration->displaySettings.height,
                  lInputContext->GetNativeHandle());
            HMI_ALERT_CONDITIONAL(lRc, "Attach native display handle to Candera display failed!");
         }
         // Native display not yet exists but shall be created automatically by Candera
         else
         {
            // Upload display configuration for automatic display creation.
            lRc = lRc && lDisplay->Upload(lDisplayConfiguration->displaySettings);
            HMI_ALERT_CONDITIONAL(lRc, "Upload of display settings to Candera display failed!");

            // Attach native display handle to window input handler, for let it start listening.
            if (lRc)
            {
               lInputContext->SetNativeHandle(lDisplay->GetNativeHandle());
            }
         }

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
         if (lDisplay)
         {
            hmibase::ILM_Accessor::init(lDisplay->GetNativeHandle());
         }
#endif
      }
   }

   return lRc;
}


// ------------------------------------------------------------------------
bool AppEnvironment::Cleanup(const std::vector<std::string>& cAssetFileNames)
{
   bool lRc = true;

   // Clear asset configuration
   /// mAssetConfiguration.Clear(); since 2.11.2 not longer supported

   // Cleanup file asset (not dependent on previous cleanups)
   //lRc = GetAsset(cAssetFileName).Finalize() && lRc;
   (void)cAssetFileNames;
   for (FeatStd::Int index = static_cast<FeatStd::Int>(mAssetConfiguration.GetAssetRepositoryCount()) - 1; index >= 0; --index)
   {
      Candera::AssetRepository* assetRepo = mAssetConfiguration.GetAssetRepository(static_cast<FeatStd::UInt16>(index));
      if (assetRepo != NULL)
      {
         lRc = assetRepo->Finalize() && lRc;
         mAssetConfiguration.Remove(assetRepo);
         if (lRc)
         {
            const char* assetFileName = (cAssetFileNames.size() > static_cast<size_t>(index)) ? cAssetFileNames[static_cast<size_t>(index)].c_str() : "";
            AssetRepositoryEvent evt(assetRepo, assetFileName, AssetRepositoryEvent::Removed);
            GetEventSource().DispatchEvent(evt);
         }

         // delete assetRepo; is static data
      }
   }

   // Cleanup main window input handling
   mDisplayInputContexts[AppDisplayInputContext::MainWindow].Terminate();
   mDisplayInputContexts[AppDisplayInputContext::MainWindow].Reset();

#if defined(HAS_2ND_DISPLAY)
   // Cleanup 2nd window input handling
   mDisplayInputContexts[AppDisplayInputContext::SecondWindow].Terminate();
   mDisplayInputContexts[AppDisplayInputContext::SecondWindow].Reset();
#endif

#if defined(HAS_TERMINAL_INPUT)
   // Cleanup terminal input handling
   mDisplayInputContexts[AppDisplayInputContext::Terminal].Terminate();
   mDisplayInputContexts[AppDisplayInputContext::Terminal].Reset();
#endif

   // Dispose all defined external input contexts
   for (FeatStd::Int i = 0; lRc && (i < GetDisplayInputContextCount()); i++)
   {
      // Read external input context
      Courier::InputHandling::DisplayInputContext* lInputContext = const_cast<Courier::InputHandling::DisplayInputContext*>(GetDisplayInputContext(i));

      // Terminate input threads and reset contexts (incl. destroying displays)
      if (lInputContext)
      {
         lInputContext->Terminate();
         lInputContext->Reset();
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
void AppEnvironment::CheckAssetDisplayConfiguration()
{
   // ----- Initialize asset accessor for reading display metadata
   Courier::AssetAccessor& lAssetAccessor = Courier::AssetAccessor::GetInstance();
   // version fr83hi widget version
   unsigned int validationFlags = 0;
   validationFlags |=  Candera::AssetValidation::Flag<Candera::CffVersionAttribute, Candera::ErrorLevel>();
   validationFlags |=  Candera::AssetValidation::Flag<Candera::FractionalNumberRepresentationAttribute, Candera::IgnoreLevel>();
   validationFlags |=  Candera::AssetValidation::Flag<Candera::PlatformInstanceNameAttribute, Candera::IgnoreLevel>();
   validationFlags |=  Candera::AssetValidation::Flag<Candera::WidgetSetHashAttribute, Candera::IgnoreLevel>();
//   validationFlags |= Candera::AssetValidation::Flag<Candera::WidgetSetVersionAttribute, Candera::ErrorLevel>();
   validationFlags |= Candera::AssetValidation::Flag<Candera::WidgetSetVersionAttribute, Candera::IgnoreLevel>();

   bool lRc = lAssetAccessor.Init(const_cast<Courier::AssetConfiguration*>(GetAssetConfiguration()), validationFlags);

   HMI_ALERT_CONDITIONAL(lRc, "Asset configuration initialization failed! In Most cases the ASSET has to be rebuild with the latest SCHOST.");

   // Loop over platform display configuration (bound to external input context)
   Courier::AssetAccessor::DisplayData lDisplayData;
   FeatStd::Int lDisplayCount = 0;
   for (FeatStd::Int i = 0; lRc && (i < GetDisplayInputContextCount()); i++)
   {
      // Read external input context
      Courier::InputHandling::DisplayInputContext* lInputContext = const_cast<Courier::InputHandling::DisplayInputContext*>(GetDisplayInputContext(i));

      // Check if display is attached to external input context
      Courier::InputHandling::DisplayConfiguration* lDisplayConfiguration = lInputContext->GetDisplayConfiguration();
      // ----- Create and initialize Candera display
      if (0 != lDisplayConfiguration)
      {
         // Display found, increase counter
         lDisplayCount++;

         // Loop over asset display configuration for checking against platform display configuration
         FeatStd::Int j = 0;
         while ((j < lAssetAccessor.GetDisplayCount()))
         {
            // Read asset's display data
            (void)lAssetAccessor.GetDisplayData(j, lDisplayData);

            // Display ID match
            if (lDisplayConfiguration->displayId == lDisplayData.mDisplayId)
            {
               ETG_TRACE_USR1_THR(("Display '%30s' (ID:%d) found in Asset!", lDisplayConfiguration->displayName, lDisplayConfiguration->displayId));

               // Check dimension of platform and asset display configuration
               if ((lDisplayConfiguration->displaySettings.width == -1) || (lDisplayConfiguration->displaySettings.height == -1))
               {
                  lDisplayConfiguration->displaySettings.width = lDisplayData.mWidth;
                  lDisplayConfiguration->displaySettings.height = lDisplayData.mHeight;
                  ETG_TRACE_USR1_THR(("Display '%30s' (ID:%d): resolution read from Asset (%dx%d)", lDisplayConfiguration->displayName, lDisplayConfiguration->displayId, lDisplayConfiguration->displaySettings.width, lDisplayConfiguration->displaySettings.height));
               }
               else if ((lDisplayConfiguration->displaySettings.width != lDisplayData.mWidth) || (lDisplayConfiguration->displaySettings.height != lDisplayData.mHeight))
               {
                  ETG_TRACE_FATAL_THR(("Dimension of display '%30s' (ID:%d) in application platform (%dx%d) and in asset (%dx%d) doesn't match!",
                                       lDisplayConfiguration->displayName, lDisplayConfiguration->displayId,
                                       lDisplayConfiguration->displaySettings.width, lDisplayConfiguration->displaySettings.height,
                                       lDisplayData.mWidth, lDisplayData.mHeight));
               }
               break;
            }
            j++;
         }
      }
   }

   // Check display count of asset and application platform implementation and inform about inconsistencies
   if (lRc && (lAssetAccessor.GetDisplayCount() != lDisplayCount))
   {
      ETG_TRACE_FATAL_THR(("Number of displays in application platform (%d) and in asset (%d) doesn't match!", lDisplayCount, lAssetAccessor.GetDisplayCount()));
   }
}


// ------------------------------------------------------------------------
FeatStd::Int AppEnvironment::GetDisplayInputContextCount() const
{
   return sizeof(mDisplayInputContexts) / sizeof(mDisplayInputContexts[0]);
}


}
