/* ***************************************************************************************
* FILE:          VSTraceHelper.hpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  VSTraceHelper.hpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2019 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined (VSTraceHelper_h)
#define VSTraceHelper_h

#include "Cit/VisualStateGlueLayer/DeductResult.h"

#include "hmi_trace_if.h"

namespace Cit {
namespace Internal {

template<typename StateMachineType>
class VSTraceHelper
{
   public:

      VSTraceHelper() : _sm(0)
      {
      }
      VSTraceHelper(StateMachineType* sm) : _sm(sm)
      {
      }

      void init(StateMachineType* sm)
      {
         _sm = sm;
      }

      unsigned char SEM_Deduct(typename StateMachineType::VS_Type_Event EventNo)
      {
         const char* txt_event;
         if (_sm->SEM_NameAbs(StateMachineType::EVENT_TYPE, EventNo, &txt_event) == StateMachineType::VS_Ret_Ok)
         {
            HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_EVENTS: Evaluate event %s", checkLength(txt_event).c_str());
         }
         else
         {
            HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_EVENTS: Evaluate event %i", EventNo);
         }
         return _sm->SEM_Deduct(EventNo);
      }

      DeductResult::Enum  DeductEvent(::Courier::Message const& message, Courier::UInt32& EventNo)
      {
         const char* txt_event;

         if (_sm->SEM_NameAbs(StateMachineType::EVENT_TYPE, static_cast<typename StateMachineType::VS_Type_Explanation>(EventNo), &txt_event) == StateMachineType::VS_Ret_Ok)
         {
            HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_EVENTS: Evaluate event %s with data ", checkLength(txt_event).c_str());
         }
         else
         {
            HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_EVENTS: Evaluate event %i with data", EventNo);
         }
         HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_4, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_EVENTS: \tpayload: %s", CourierMessageMapTrace::getMessagePayload(message).c_str());

         return _sm->DeductEvent(message, EventNo);
      }

      void VSAction(typename StateMachineType::VS_Type_ActionExpression i)
      {
         _sm->VSAction(i);

         // ==== state info start ===
         const char* txt_curr_state;
         if (_sm->SEM_NameAbs(StateMachineType::STATE_TYPE, _sm->GetCurrentState(), &txt_curr_state) == StateMachineType::VS_Ret_Ok)
         {
            HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_ACTIONS:    -> SM=%u, current state=%s", _sm->GetCurrentSM(), checkLength(txt_curr_state).c_str());
         }
         else
         {
            HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_ACTIONS:      -> SM=%u, current state=%u", _sm->GetCurrentSM(), _sm->GetCurrentState());
         }
         // ====  state info end  ===
      }

      unsigned char SEM_NextStateChg()
      {
         unsigned char retval = 0;
         typename StateMachineType::VS_Type_State* smList;

         int numberOfSM = _sm->GetNumberOfStatemachines();
         typename StateMachineType::VS_Type_State currState;
         unsigned char ret1;

         smList = new typename StateMachineType::VS_Type_State[numberOfSM];
         if (smList == 0)
         {
            return retval;
         }

         for (int i = 0u; i < numberOfSM; i++)
         {
            ret1 = _sm->SEM_State(static_cast<typename StateMachineType::VS_Type_Statemachine>(i), &(smList[i]));
            if (ret1 != StateMachineType::VS_Ret_Found)
            {
               HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_ERRORS, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_STATES:        ERROR: SEM_NextState() old state %i not valid ", i);
            }
         }
         retval = _sm->SEM_NextStateChg();
         for (int i = 0u; i < numberOfSM; i++)
         {
            ret1 = _sm->SEM_State(static_cast<typename StateMachineType::VS_Type_Statemachine>(i), &currState);
            if (ret1 != StateMachineType::VS_Ret_Found)
            {
               HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_ERRORS, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_STATES:        ERROR: SEM_NextState() new state %i not valid ", i);
            }
            if (smList[i] != currState)
            {
               const char* txt_old_state;
               const char* txt_new_state;

               if (_sm->SEM_NameAbs(StateMachineType::STATE_TYPE, smList[i], &txt_old_state) == StateMachineType::VS_Ret_Ok)
               {
                  HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_STATES:        SM: %i Leaving state %s", i, checkLength(txt_old_state).c_str());
               }
               else
               {
                  HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_STATES:        SM: %i Leaving state %i", i, smList[i]);
               }
               if (_sm->SEM_NameAbs(StateMachineType::STATE_TYPE, currState, &txt_new_state) == StateMachineType::VS_Ret_Ok)
               {
                  HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_STATES:        SM: %i Entering state %s", i, checkLength(txt_new_state).c_str());
               }
               else
               {
                  HMIBASE_TRACE_VARG_CLS_DCL(ETG_LEVEL_USER_1, TR_CLASS_HMI_SM, _sm->GetTraceClass(), "SM_STATES:        SM: %i Entering state %i", i, currState);
               }
            }
         }
         delete[] smList;
         return retval;
      }

   private:
      std::string checkLength(const char* text)
      {
         if (strlen(text) > 150)
         {
            std::string s = "...";
            s += text + strlen(text) - 150;
            return s;
         }
         return text;
      }

      StateMachineType* _sm;
};


}
}


#endif
