/* ***************************************************************************************
* FILE:          ViewControllerBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewControllerBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "ViewControllerBase.h"
#include "Focus/FManager.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"

namespace hmibase {
namespace view {
ViewControllerBase::OnParentViewLoadFct ViewControllerBase::_onParentViewLoadHook = 0;

ViewControllerBase::ViewControllerBase() :
   _focuspriority(-1)
{
}


ViewControllerBase::ViewControllerBase(signed int priority) :
   _focuspriority(priority)
{
}


ViewControllerBase::~ViewControllerBase()
{
}


void ViewControllerBase::OnParentViewActivate(bool activate)
{
   if (GetView() != NULL)
   {
      Focus::FManager::getInstance().onViewActivate(GetView()->GetId(), activate);
   }

   if (activate && _focuspriority >= 0)
   {
      Courier::RTPtrVector RTVectorLoc;
      GetView()->GetRenderTargetPtrVector(RTVectorLoc, true);
      Courier::RTPtrVector::Iterator iter = RTVectorLoc.Begin();

      //assumption is that view are mapped to only one RenderTarget or first RenderTarget is the main one
      unsigned int surfaceId = getSurfaceId(*iter);

      if (surfaceId != 0)
      {
         ScreenBrokerClient::GetInstance().SetInputFocusPublic(
            surfaceId,
            true,
            true,
            _focuspriority,
            0);
      }
   }
}


void ViewControllerBase::OnParentViewLoad(bool load)
{
   if (_onParentViewLoadHook)
   {
      _onParentViewLoadHook(load, GetView());
   }
}


bool ViewControllerBase::onCourierMessage(const TransitionMediatorOnDataCompleteMsg& oMsg)
{
   _dataCompletionHandler.notifyDataComplete(oMsg.GetHint());
   return true;
}


bool ViewControllerBase::onCourierMessage(const TimerExpiredMsg& oMsg)
{
   return _dataCompletionHandler.checkTimeout(oMsg.GetTimer());
}


DataCompletionHandler* ViewControllerBase::getDataCompletionHandler()
{
   return &_dataCompletionHandler;
}


unsigned int ViewControllerBase::getSurfaceId(const Candera::Surface* surface) const
{
   unsigned int surfaceId;
   if (0 == surface)
   {
      return 0;
   }
   Candera::GraphicDeviceUnit* gdu = surface->GetGraphicDeviceUnit();
   if (0 == gdu)
   {
      return 0;
   }
   const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(gdu->GetUnitType());
   if (0 == metaInfo)
   {
      return 0;
   }
   Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
   if (0 == propMetaInfo)
   {
      return 0;
   }

   Candera::Char buffer[32];
   buffer[0] = '\0';
   if (!propMetaInfo->Get(gdu, buffer, 32))
   {
      return false;
   }
   ::sscanf(buffer, "%u", &surfaceId);
   return surfaceId;
}


/************************************************************************/
void ViewControllerBase::RegisterMessageHandler(MessageHandler& handler)
{
   _messageHandlers.push_back(&handler);
}


/************************************************************************/
void ViewControllerBase::UnregisterMessageHandler(MessageHandler& handler)
{
   MessageHandlersType::iterator it = std::find(_messageHandlers.begin(), _messageHandlers.end(), &handler);
   if (it != _messageHandlers.end())
   {
      _messageHandlers.erase(it);
   }
}


/************************************************************************/
bool ViewControllerBase::DispatchToMessageHandlers(const Courier::Message& msg)
{
   for (MessageHandlersType::iterator it = _messageHandlers.begin(); it != _messageHandlers.end(); ++it)
   {
      MessageHandler* handler = *it;
      if ((handler != NULL) && handler->OnMessage(*this, msg))
      {
         return true;
      }
   }
   return false;
}


}
}
