/* ***************************************************************************************
* FILE:          SurfaceUtils.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SurfaceUtils is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "SurfaceUtils.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"

//#include <Courier/DataBinding/TypeConverter.h>
//#include <Candera/EngineBase/DataTypes/ColorDataType.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/SurfaceUtils.cpp.trc.h>
#endif

namespace hmibase {
namespace view {
namespace util {

/************************************************************************/
SurfaceIdType SurfaceUtils::getSurfaceId(const Candera::Surface& surface)
{
   FeatStd::UInt32 id = 0;

   Candera::GraphicDeviceUnit* gdu = surface.GetGraphicDeviceUnit();
   if (gdu != NULL)
   {
      const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(gdu->GetUnitType());
      if (metaInfo != NULL)
      {
         Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
         //if surface's gdu has the property surface id get its value
         if (propMetaInfo != NULL)
         {
            FeatStd::Char buffer[32];
            buffer[0] = '\0';
            if (propMetaInfo->Get(gdu, buffer, 32))
            {
               ::sscanf(buffer, "%u", &id);
            }
         }
         //otherwise check the owner gdu
         else
         {
            Candera::Internal::GraphicDeviceUnitOwnerAccess gduAccess(gdu);
            Candera::GraphicDeviceUnit* ownerGdu = gduAccess.GetGraphicDeviceUnitOwner();
            if (ownerGdu != NULL)
            {
               Candera::Surface* ownerRT = ownerGdu->ToRenderTarget3D();
               if ((ownerRT != NULL) && (&surface != ownerRT))
               {
                  id = getSurfaceId(*ownerRT);
               }
            }
         }
      }
   }

   return id;
}


/************************************************************************/
DisplayIdType SurfaceUtils::getDisplayId(SurfaceIdType surfaceId)
{
   return ScreenBrokerClient::GetInstance().GetDisplayID(surfaceId);
}


/************************************************************************/
DisplayIdType SurfaceUtils::getDisplayId(const Candera::Surface& surface)
{
   return getDisplayId(getSurfaceId(surface));
}


/************************************************************************/
bool SurfaceUtils::transformSceneToDisplay(Candera::Camera2D& camera, Candera::Vector2& coordinate)
{
   if (camera.GetRenderTarget() == NULL)
   {
      return false;
   }

   coordinate = Candera::Math2D::TransformSceneToViewport(camera, coordinate);
   coordinate = Candera::Math2D::TransformViewportToRenderTarget(camera, coordinate);
   coordinate = Candera::Math2D::TransformRenderTargetToScreen(*(camera.GetRenderTarget()), coordinate);
   return true;
}


/************************************************************************/
bool SurfaceUtils::transformSceneToSurface(Candera::Camera2D& camera, Candera::Vector2& coordinate)
{
   if (camera.GetRenderTarget() == NULL)
   {
      return false;
   }

   coordinate = Candera::Math2D::TransformSceneToViewport(camera, coordinate);
   coordinate = Candera::Math2D::TransformViewportToRenderTarget(camera, coordinate);
   return true;
}


/************************************************************************/
void SurfaceUtils::sendTouchableRectangle(SurfaceIdType surfaceId, const Candera::Rectangle& rect)
{
   ETG_TRACE_USR4_CLS_THR((TR_CLASS_HMI_FW_INPUT, "sendTouchableRectangle SurfaceId=%u, Rectangle=[%d,%d,%d,%d] App=%s",
                           surfaceId,
                           static_cast<int>(rect.GetLeft()),
                           static_cast<int>(rect.GetTop()),
                           static_cast<int>(rect.GetWidth()),
                           static_cast<int>(rect.GetHeight()),
                           hmibase::trace::getAppName().c_str()));
}


}
}


}
