/* ***************************************************************************************
* FILE:          Renderer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Renderer.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "lint_deactivation.h"
#include "Renderer.h"

#include <CanderaPlatform/Device/Common/Base/RenderTarget.h>

using namespace Candera;
using namespace Courier;

namespace hmibase {
namespace view {

// ------------------------------------------------------------------------
Renderer::Renderer() : _observer(0)
{
}


// ------------------------------------------------------------------------
Renderer::~Renderer()
{
   _observer = 0;
}


// ------------------------------------------------------------------------
Gdu* Renderer::GetGdu(Candera::RenderTarget* renderTarget)
{
   Gdu* gdu = Base::GetGdu(renderTarget);

   if ((renderTarget != 0) && (gdu == 0)
         && (renderTarget->GetGraphicDeviceUnit() != 0)//NCG3D-67115 - prevent crash during drunken monkey tests
         && (renderTarget->GetGraphicDeviceUnit()->ToRenderTarget2D() != 0
             || renderTarget->GetGraphicDeviceUnit()->ToRenderTarget3D() != 0))
   {
      gdu = FEATSTD_NEW(Gdu)(renderTarget->GetGraphicDeviceUnit(), 0);
      if (gdu != 0)
      {
         // NCG3D-133579: first add gdu to vector to prevent that Upload will trigger another allocation and so on -> endless loop
         if (mGDU.Add(gdu))
         {
            // sucessfully added to vector, now try to upload
            if (gdu->Upload(this) == false)
            {
               // upload failed, so remove gdu from vector
               mGDU.Remove(mGDU.Size() - 1);
               FEATSTD_DELETE(gdu);
               gdu = 0;
            }
         }
      }
   }

   return gdu;
}


bool Renderer::AddGdu(GraphicDeviceUnit* inGdu)
{
   if (NULL != inGdu)
   {
      Gdu* gduObj = FEATSTD_NEW(Gdu)(inGdu, mGDU.Size());
      FEATSTD_LINT_NEXT_EXPRESSION(774, "allocation may fail")

      if ((0 != gduObj) && mGDU.Add(gduObj))
      {
         gduObj->Log();

         if (mRenderConfiguration.ShallLoadAllRenderTargetsAtStartup())
         {
            Gdu* lookupGdu = GetGdu(gduObj->GetRenderTarget());

            if ((0 != lookupGdu) && (!lookupGdu->IsLoaded()))
            {
               EnableLayer(true, lookupGdu->GetRenderTarget(), true);
            }
         }
         return true;
      }
   }
   return false;
}


bool Renderer::LoadGdu(GraphicDeviceUnit* inGdu)
{
   if (NULL != inGdu)
   {
      Gdu* lookupGdu = GetGdu(inGdu->ToRenderTarget3D());
      if ((0 != lookupGdu) && (!lookupGdu->IsLoaded()))
      {
         EnableLayer(true, lookupGdu->GetRenderTarget(), true);
      }
      return true;
   }
   return false;
}


bool Renderer::RemoveGdu(GraphicDeviceUnit* inGdu)
{
   bool result = false;

   if ((NULL != inGdu) && (mGDU.Size() > 0))
   {
      Candera::SizeType i = mGDU.Size() - 1;

      while (true)
      {
         Gdu* gdu = mGDU[i];

         if (gdu != 0)
         {
            if (inGdu->ToRenderTarget3D() == gdu->GetRenderTarget())
            {
               if (mGDU.Remove(i))
               {
                  result = true;
               }
               else
               {
                  //FEATSTD_LOG_ERROR("Removing the GDU from GDU vector failed!");
               }
            }
         }

         if (i == 0)
         {
            break;
         }
         else
         {
            --i;
         }
      }
   }
   return result;
}


bool Renderer::Render(Courier::RenderHint* renderHint)
{
   if (_observer)
   {
      _observer->onPreRender();
   }

   bool ret = Base::Render(renderHint);

   if (_observer)
   {
      _observer->onPostRender(ret);
   }
   return ret;
}


const Renderer::tLayerSurfaceMap& Renderer::GetLayerSurfaceMap()
{
   if (_layerSurfaceMap.empty())
   {
      for (FeatStd::SizeType i = 0; i < mGDU.Size(); i++)
      {
         Gdu* gdu = mGDU[i];
         if (gdu != 0)
         {
            Candera::Surface* surface = gdu->GetRenderTarget();
            if (0 != surface)
            {
               Candera::GraphicDeviceUnit* graphicDeviceUnit = surface->GetGraphicDeviceUnit();
               if (0 != graphicDeviceUnit)
               {
                  const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(graphicDeviceUnit->GetUnitType());
                  if (0 != metaInfo)
                  {
                     Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
                     if (0 != propMetaInfo)
                     {
                        unsigned int id = 0;
                        Char buffer[32];
                        buffer[0] = '\0';
                        if (propMetaInfo->Get(graphicDeviceUnit, buffer, 32))
                        {
                           ::sscanf(buffer, "%u", &id);
                           std::vector<unsigned int>* surfacelist = &_layerSurfaceMap[gdu->GetLayerId()];
                           surfacelist->push_back(id);
                        }
                     }
                  }
               }
            }
         }
      }
   }

   return _layerSurfaceMap;
}


}
} // namespace hmibase
