/* ***************************************************************************************
* FILE:          RenderJobStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RenderJobStrategy.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "RenderJobStrategy.h"
#include "View/IMessageSystem.h"
#include <View/CGI/CgiExtensions/ViewScene2D.h>

#include "EGL/egl.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_VIEW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RenderJobStrategy.cpp.trc.h"
#endif

namespace hmibase {
namespace view {

RenderJobStrategy::CallbackFunction RenderJobStrategy::s_callback = 0;
std::set<RenderJobStrategy::RenderJobListener*> RenderJobStrategy::s_renderJobListener;

RenderJobStrategy::RenderJobStrategy()
{
}


RenderJobStrategy::~RenderJobStrategy()
{
}


void RenderJobStrategy::SetEarlySwap(bool value)
{
   mEarlySwap = value;
}


bool RenderJobStrategy::OnSwapBuffer(Courier::Gdu&)
{
   //default value of mEarlySwap in order to preserve the behavior from the base class
   return false;
}


void RenderJobStrategy::OnPostSwapBuffer(Courier::Gdu& gdu)
{
   unsigned int surfaceId = 0;
   std::map<const Courier::Gdu*, unsigned int>::const_iterator cit = _gdus.find(&gdu);

   if (cit != _gdus.end())
   {
      surfaceId = cit->second;
   }
   else
   {
      surfaceId = GetSurfaceId(gdu);
      _gdus[&gdu] = surfaceId;
   }

   std::map<unsigned int, unsigned int>::iterator it = _waitForNativeAfterSwap.find(surfaceId);
   if (it != _waitForNativeAfterSwap.end())
   {
      EGLBoolean success = eglWaitNative(EGL_CORE_NATIVE_ENGINE);

      if (it->second > 0)
      {
         it->second = it->second - 1;
      }

      if (s_callback)
      {
         s_callback(surfaceId, (success > 0) ? false : true, it->second);
      }
      else
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(EglWaitNativeResMsg)(surfaceId, (success > 0) ? false : true, it->second);
         if (msg)
         {
            msg->Post();
         }
      }

      if (it->second == 0)
      {
         _waitForNativeAfterSwap.erase(it);
      }
   }

   // if a listener is registered, we have to guarantee that all OpenGl commands are executed on the graphics card,
   // before we inform the listener that swap has finished (useful for direct texture use case, where drm buffer is used)
   if (s_renderJobListener.empty() == false)
   {
#ifdef WIN32
      EGLBoolean success = true;
#else
      EGLBoolean success = eglWaitNative(EGL_CORE_NATIVE_ENGINE);
#endif

      if (success)
      {
         if (s_renderJobListener.size() > 0)
         {
            // create a copy of lstener set to iterate through as a listener can deregister during iteration
            std::set<RenderJobListener*> workingCopy = s_renderJobListener;
            for (std::set<RenderJobListener*>::iterator it = workingCopy.begin(); it != workingCopy.end(); ++it)
            {
               if (*it)
               {
                  (*it)->OnPostSwapBuffer(gdu);
               }
            }
         }
      }
   }
   ETG_TRACE_USR4_CLS_THR((TR_CLASS_HMI_PERFORMANCE_MP, "Buffer swapped, layer id %d", gdu.GetLayerId()));
}


void RenderJobStrategy::TransformDirtyAreaToInvalidationDependency(const FeatStd::Optional<Candera::Rectangle>& dirtyArea, FeatStd::Optional<Candera::Rectangle>& transformedDirtyArea,
      const Courier::View& view, const Courier::RenderJobStrategy::InvalidationDependencyData& invalidationDependencyData)
{
   if ((*dirtyArea).GetSize() == Candera::Vector2())
   {
      Base::TransformDirtyAreaToInvalidationDependency(dirtyArea, transformedDirtyArea, view, invalidationDependencyData);
   }
   else
   {
      // for dirty area repainting to work together with invalidation dependencies it may be necessary to translate the dirty area if the viewport of the cameras from the 2 scenes don't have the same position
      // we use InvalidationDependency.DirtyArea.Position to translate the dirty area so that we can have the same rectangle in both scenes

      Candera::Vector2 translate((*invalidationDependencyData.GetInvalidationDependency().GetDirtyArea()).GetPosition());
      if (translate != Candera::Vector2())
      {
         Candera::Rectangle area(*dirtyArea);
         ETG_TRACE_USR4_THR(("Adjust dirtyArea=[%d,%d,%d,%d] translate=[%d,%d] for view '%s'",
                             static_cast<int>(area.GetLeft()), static_cast<int>(area.GetTop()), static_cast<int>(area.GetWidth()), static_cast<int>(area.GetHeight()),
                             static_cast<int>(translate.GetX()), static_cast<int>(translate.GetY()), view.GetId().CStr()));

         area.SetPosition(area.GetPosition() + translate);
         transformedDirtyArea = area;
      }
      else
      {
         transformedDirtyArea = dirtyArea;
      }
   }
}


FeatStd::UInt8 RenderJobStrategy::GetEffectiveRenderCounter(const Courier::ViewScene* view, const Candera::CanderaObject& camera, const Courier::Gdu& gdu, FeatStd::UInt8 renderCounter)
{
   const hmibase::view::ViewScene2D* viewScene2D = dynamic_cast<const hmibase::view::ViewScene2D*>(view);
   // a view is independent if the content rendered for it is not affected by the content rendered for other views, also there is no invalidation dependency.
   // this feature is required for wait animations rendered into separate views and separate window surfaces where we need max fps with min cpu load.
   if ((viewScene2D != NULL) && viewScene2D->IsIndependentUpdateEnabled())
   {
      return 1;
   }
   return Base::GetEffectiveRenderCounter(view, camera, gdu, renderCounter);
}


bool RenderJobStrategy::operator()(const Courier::RenderJob& a, const Courier::RenderJob& b) const
{
   //overrides the default sorting from Courier to not consider if the jobs are 2D or 3D
   const Courier::RenderConfiguration& renderConfiguration = Courier::Renderer::GetRenderConfiguration();
   const bool c_aIsOffscreen = const_cast<Courier::RenderJob&>(a).GetGdu()->IsOffscreen();
   const bool c_bIsOffscreen = const_cast<Courier::RenderJob&>(b).GetGdu()->IsOffscreen();
   return ((c_aIsOffscreen && (!c_bIsOffscreen)) || ((c_aIsOffscreen == c_bIsOffscreen) && (
              (a.IsClear() && (!b.IsClear())) ||
              ((a.IsClear() == b.IsClear()) &&
               (renderConfiguration.ShallUseGlobalCameraSequenceNumber() && (a.GetSequenceNumber() < b.GetSequenceNumber()))
              )
           )));
}


void RenderJobStrategy::WaitForNativeAfterSwap(unsigned int surfaceId, unsigned int count)
{
   if (count != 0)
   {
      _waitForNativeAfterSwap[surfaceId] = count;
   }
   else
   {
      _waitForNativeAfterSwap.erase(surfaceId);
   }
}


unsigned int RenderJobStrategy::GetSurfaceId(Courier::Gdu& gdu) const
{
   unsigned int surfaceId = 0;
   Candera::GraphicDeviceUnit* canderaGdu = gdu.GetGdu();

   if (canderaGdu)
   {
      // get surface id and store it for next time
      const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(canderaGdu->GetUnitType());
      if (0 != metaInfo)
      {
         Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
         if (0 != propMetaInfo)
         {
            char buffer[32];
            buffer[0] = '\0';
            if (propMetaInfo->Get(canderaGdu, buffer, 32))
            {
               ::sscanf(buffer, "%u", &surfaceId);
            }
         }
      }
   }

   return surfaceId;
}


void RenderJobStrategy::RegisterListener(RenderJobListener* listener)
{
   s_renderJobListener.insert(listener);
}


void RenderJobStrategy::DeregisterListener(RenderJobListener* listener)
{
   std::set<RenderJobListener*>::iterator it = s_renderJobListener.find(listener);

   if (it != s_renderJobListener.end())
   {
      s_renderJobListener.erase(listener);
   }
}


}
}
