/* ***************************************************************************************
* FILE:          ImageUploadCache.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageUploadCache.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(_IMAGE_UPLOAD_CACHE_H)
#define _IMAGE_UPLOAD_CACHE_H

#include <FeatStd/Event/EventListener.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>

//define this symbol in the configuration/project files (or define it from here but only for debug purpose)
//#define VARIANT_S_FTR_ENABLE_IMAGE_UPLOAD_CACHE

#if defined(VARIANT_S_FTR_ENABLE_IMAGE_UPLOAD_CACHE)

namespace hmibase {
namespace view {
namespace utils {

class ImageUploadCache : public FeatStd::EventListener
#ifdef CANDERA_CUSTOMIZATION_BITMAP_BRUSH
   , private Candera::BitmapBrush::UploadListener
#endif
{
   public:
      /////////////////////////////////////////////////////////////////////////
      typedef Candera::MemoryManagement::SharedPointer<Candera::Image2D> ImageSharedPointer;

      /////////////////////////////////////////////////////////////////////////
      static ImageUploadCache& getInstance()
      {
         FEATSTD_UNSYNCED_STATIC_OBJECT(ImageUploadCache, _instance);
         return _instance;
      }

      // Enables the image caching and registers the cache as bitmap upload listener.
      void enable();
      // Disables the image caching and resets it.
      void disable();
      // resets the cache by removing all the entries
      void reset(bool isError = false);

      size_t getActiveImagesCount() const;
      size_t getActiveImagesSize() const;

      size_t getCachedImagesCount() const;
      size_t getCachedImagesSize() const;

      size_t getTotalImagesCount() const;
      size_t getTotalImagesSize() const;

      size_t getMaximumImageSize() const;
      void setMaximumImageSize(size_t value);

      size_t getMaximumCachedImagesSize() const;
      void setMaximumCachedImagesSize(size_t value);

      size_t getMaximumTotalImagesSize() const;
      void setMaximumTotalImagesSize(size_t value);

      /////////////////////////////////////////////////////////////////////////
      // Maximum size of images.
      static const size_t DefaultMaximumImageSize;
      static const size_t DefaultMaximumCachedImagesSize;
      static const size_t DefaultMaximumTotalImagesSize;

   private:
      /////////////////////////////////////////////////////////////////////////
      class ImageInfo;
      class ImageInfoList;

      /////////////////////////////////////////////////////////////////////////
      typedef const Candera::Char* ImageName;
      typedef const Candera::Image2D* ImageId;
      //typedef Candera::Id ImageId;
      typedef size_t ImageSize;
      typedef std::map<ImageId, ImageInfo* > ImageInfoMap;

      /////////////////////////////////////////////////////////////////////////
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ImageUploadCache);
      ImageUploadCache();

      static int getAppId();

      /////////////////////////////////////////////////////////////////////////
      // overrides from Candera::BitmapBrush::UploadListener
      virtual void OnPreUnload(const ImageSharedPointer& image);
      virtual void OnPostUpload(const ImageSharedPointer& image);

      // overrides from FeatStd::EventListener
      virtual FeatStd::EventResult::Enum OnEvent(const FeatStd::Event& evt);

      /////////////////////////////////////////////////////////////////////////
      void deleteImageInfo(ImageInfo* imageInfo, bool isError = false);
      void ensureMaximumSize();
      static ImageName getImageName(const ImageSharedPointer& image);
      static ImageId getImageId(const ImageSharedPointer& image);
      static ImageSize getImageSize(const ImageSharedPointer& image);

      /////////////////////////////////////////////////////////////////////////
      class ImageInfo
      {
            friend class ImageInfoList;

         public:
            ImageInfo(const ImageSharedPointer& image);
            ~ImageInfo();

            void updateInfo();
            bool isActive() const;
            unsigned int getOwnerAppendIndex() const
            {
               return _ownerAppendIndex;
            }

            ImageSharedPointer _image;

            // Cached id to avoid errors if the actual id changes dynamically
            ImageId _imageId;

            // Cached size to avoid errors if the actual size changes dynamically
            size_t _imageSize;

            int _uploadCounter;

         private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(ImageInfo);

            // List access members
            ImageInfoList* _ownerList;
            ImageInfo* _next;
            ImageInfo* _previous;
            unsigned int _ownerAppendIndex;
      };

      /////////////////////////////////////////////////////////////////////////
      // Custom double linked list which allows us to avoid keeping list iterators in the entries map
      class ImageInfoList
      {
         public:
            ImageInfoList();
            ~ImageInfoList();

            ImageInfo* getOldestImage()
            {
               return _oldestImage;
            }

            size_t getImagesCount() const
            {
               return _imagesCount;
            }

            size_t getImagesSize() const
            {
               return _imagesSize;
            }

            bool removeImage(ImageInfo* imageInfo);
            bool appendImage(ImageInfo* imageInfo);
            void reset();

         private:
            FEATSTD_MAKE_CLASS_UNCOPYABLE(ImageInfoList);

            ImageInfo* _oldestImage;
            size_t _imagesCount;
            size_t _imagesSize;
            // incremented every time append is called, this means that oldest image will have the smallest index
            unsigned int _appendIndex;
      };

      /////////////////////////////////////////////////////////////////////////
      ImageInfoMap _images;
      ImageInfoList _cachedImages;

      //statistics
      typedef FeatStd::UInt64 StatsSize;
      size_t _hitCount;
      StatsSize _hitSize;
      size_t _missCount;
      StatsSize _missSize;

      size_t _activeImagesSize;

      size_t _maximumImageSize;
      size_t _maximumCachedImagesSize;
      size_t _maximumTotalImagesSize;
};


}//utils
}//view
}//hmibase

#endif//VARIANT_S_FTR_ENABLE_IMAGE_UPLOAD_CACHE

#endif//_IMAGE_UPLOAD_CACHE_H
