/* ***************************************************************************************
* FILE:          ImageLoader.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageLoader.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "ImageLoader.h"

#include <Courier/Version.h>
#include <Courier/Platform/CriticalSectionLocker.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetIdMacros.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

#include <IL/il.h>
#ifdef WIN32
#include "Win32/IL/ilu.h"
#else
#include "IL/ilu.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_IMAGE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ImageLoader.cpp.trc.h"
#endif

namespace hmibase {
namespace view {

static ILuint createILImage()
{
   static bool isILInit = false;
   if (!isILInit)
   {
      isILInit = true;
      ilInit();
      iluInit();
   }

   ILuint handle = 0;
   ilGenImages(1, &handle);
   if (handle != 0)
   {
      ilBindImage(handle);
   }
   return handle;
}


static void destroyILImage(ILuint handle)
{
   if (handle != 0)
   {
      ilDeleteImages(1, &handle);
   }
}


static bool loadILImageFile(const char* fileName)
{
   return ilLoadImage(fileName) == IL_TRUE;
}


static bool loadILImageData(ILenum type, const void* data, ILuint dataSize)
{
   return ilLoadL(type, data, dataSize) == IL_TRUE;
}


static Candera::Bitmap::SharedPointer createBitmap()
{
   Candera::Bitmap::SharedPointer bitmap;

   ilEnable(IL_ORIGIN_SET);
   ilOriginFunc(IL_ORIGIN_LOWER_LEFT);

   if (ilConvertImage(IL_RGBA, IL_UNSIGNED_BYTE) == IL_TRUE)
   {
      ILuint format = static_cast<ILuint>(ilGetInteger(IL_IMAGE_FORMAT));
      ILuint type = static_cast<ILuint>(ilGetInteger(IL_IMAGE_TYPE));
      ILuint width = static_cast<ILuint>(ilGetInteger(IL_IMAGE_WIDTH));
      ILuint height = static_cast<ILuint>(ilGetInteger(IL_IMAGE_HEIGHT));
      ILuint depth = static_cast<ILuint>(ilGetInteger(IL_IMAGE_DEPTH));
      ILuint bpp = static_cast<ILuint>(ilGetInteger(IL_IMAGE_BPP));

      if ((format == IL_RGBA) && (type == IL_UNSIGNED_BYTE))
      {
         size_t bufferSize = static_cast<size_t>(width * height * bpp);
         FeatStd::UInt8* buffer = CANDERA_NEW_ARRAY(FeatStd::UInt8, bufferSize);
         if (buffer != NULL)
         {
            ilCopyPixels(0, 0, 0, width, height, depth, format, type, (void*)buffer);
            bitmap = Candera::Bitmap::Create(static_cast<FeatStd::UInt16>(width),
                                             static_cast<FeatStd::UInt16>(height),
                                             Candera::Bitmap::RgbaUnsignedBytePixelFormat,
                                             Candera::Bitmap::PackAlignment1,
                                             buffer,
                                             Candera::Bitmap::Disposer::Dispose);
         }
      }
   }
   return bitmap;
}


//use this lock for external access to DevIL
static Courier::Platform::CriticalSection& getLock()
{
   static Courier::Platform::CriticalSection _lock;
   return _lock;
}


Candera::Bitmap::SharedPointer ImageLoader::loadBitmapFile(const Candera::Char* fileName)
{
   Candera::Bitmap::SharedPointer bitmap;
   if (fileName == NULL)
   {
      ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapFile fileName=NULL!"));
      return bitmap;
   }

   ETG_TRACE_USR4_THR(("ImageLoader::loadBitmapFile fileName=[%s]", fileName));

   getLock().Obtain();

   ILuint handle = createILImage();
   if (handle != 0)
   {
      if (loadILImageFile(fileName) != 0)
      {
         bitmap = createBitmap();
         if (bitmap.PointsToNull())
         {
            ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapFile createBitmap failed for fileName=[%s]!", fileName));
         }
         else
         {
            ETG_TRACE_USR4_THR(("ImageLoader::loadBitmapFile width=%u, height=%u, fileName=[%s]",
                                bitmap->GetWidth(), bitmap->GetHeight(), fileName));
         }
      }
      else
      {
         ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapFile loadILImageFile failed for fileName=[%s]!", fileName));
      }
      destroyILImage(handle);
   }
   else
   {
      ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapFile createILImage failed for fileName=[%s]!", fileName));
   }

   getLock().Release();

   return bitmap;
}


Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> ImageLoader::loadBitmapImageFile(const Candera::Char* fileName)
{
   Candera::Bitmap::SharedPointer bmp = loadBitmapFile(fileName);
   if (!bmp.PointsToNull())
   {
      Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image = Candera::BitmapImage2D::Create();
      if (!image.PointsToNull())
      {
         image->SetBitmap(bmp);
      }
      return image;
   }
   return Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D>();
}


Candera::Bitmap::SharedPointer ImageLoader::loadBitmapData(const void* data, size_t dataSize)
{
   Candera::Bitmap::SharedPointer bitmap;
   if (data == NULL)
   {
      ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapData data=NULL!"));
      return bitmap;
   }

   ETG_TRACE_USR4_THR(("ImageLoader: loadBitmapData dataSize=%u, data=%p", dataSize, data));

   getLock().Obtain();

   ILuint handle = createILImage();
   if (handle != 0)
   {
      if (loadILImageData(0, data, static_cast<ILuint>(dataSize)))
      {
         bitmap = createBitmap();
         if (bitmap == NULL)
         {
            ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapData createBitmap failed!"));
         }
         else
         {
            ETG_TRACE_USR4_THR(("ImageLoader::loadBitmapData width=%u, height=%u, dataSize=%u, data=%p",
                                bitmap->GetWidth(), bitmap->GetHeight(), dataSize, data));
         }
      }
      destroyILImage(handle);
   }
   else
   {
      ETG_TRACE_ERR_THR(("ImageLoader::loadBitmapData createILImage failed!"));
   }

   getLock().Release();

   return bitmap;
}


// DEPRECATED
Candera::Bitmap* ImageLoader::loadBitmapResource(const Candera::Char* /*canderaName*/)
{
   ETG_TRACE_FATAL_THR(("ImageLoader: loadBitmapResource This method is not thread safe, it should not be used!"));
   return NULL;

   /*
      if (canderaName == NULL)
      {
      return NULL;
      }

      Candera::Bitmap* bitmap = NULL;
      Candera::DefaultAssetProvider& assetProvider = Candera::DefaultAssetProvider::GetInstance();
      Candera::ResourceHandle resHandle = assetProvider.OpenRawResource(canderaName);
      if (resHandle != NULL)
      {
      size_t dataSize = static_cast<size_t>(assetProvider.GetResourceSize(resHandle));
      void* data = CANDERA_NEW_ARRAY(FeatStd::UInt8, dataSize);
      if ((data != NULL) && (assetProvider.ReadResource(resHandle, data, 0, dataSize) == dataSize))
      {
      bitmap = loadBitmapData(data, dataSize);
      CANDERA_DELETE_ARRAY(data);
      }

      assetProvider.CloseResource(resHandle);
      }
      return bitmap;
      */
}


// DEPRECATED
Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> ImageLoader::createImage(Candera::Bitmap* bitmap)
{
   Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image = Candera::BitmapImage2D::Create();
   if (!image.PointsToNull())
   {
      image->SetBitmap(Candera::MemoryManagement::SharedPointer<Candera::Bitmap>(bitmap));
   }
   return image;
}


Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> ImageLoader::createImage(Candera::MemoryManagement::SharedPointer<Candera::Bitmap> bmp)
{
   Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> image = Candera::BitmapImage2D::Create();
   if (!image.PointsToNull())
   {
      image->SetBitmap(bmp);
   }
   return image;
}


class AssetIdUtils
{
   public:
      //calculates the hash of the specified string
      //the same function is used by the asset ids plugin
      static unsigned int CalculateHash(const char* name)
      {
         unsigned int hash = 0;

         if (name != NULL)
         {
            while (*name != 0)
            {
               hash += (unsigned int)(*name);
               hash += (hash << 10);
               hash ^= (hash >> 6);

               ++name;
            }
         }

         hash += (hash << 3);
         hash ^= (hash >> 11);
         hash += (hash << 15);
         return hash;
      }
};


class DelayedBitmapImage2D : public Candera::BitmapImage2D
{
      typedef Candera::BitmapImage2D Base;

   public:
      explicit DelayedBitmapImage2D(const Candera::Char* bitmapCanderaName) :
         _bitmapCanderaName(bitmapCanderaName), _initialized(false)
      {
      }

   protected:
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3)))
      virtual bool UploadInternal(Candera::DeviceObject2D::LoadingHint loadingHint)
#else
      virtual bool UploadInternal()
#endif
      {
         if (!_initialized)
         {
            Candera::MemoryManagement::SharedPointer<Candera::Bitmap> bitmap;

            ////calculate the lib item id based on the candera name
            //unsigned int libItemId = AssetIdUtils::CalculateHash(_bitmapCanderaName.c_str());

            ////if candera name contains a scl name (scl name appears in candera name before '##')
            //size_t sclDelimiterPos = _bitmapCanderaName.find("##");
            //if (sclDelimiterPos != std::string::npos)
            //{
            //   //calculate the scl id based on scl name
            //   unsigned int sclId = AssetIdUtils::CalculateHash(_bitmapCanderaName.substr(0, sclDelimiterPos).c_str());

            //   //search the bitmap based on scl id and lib item id
            //   bitmap = tryGetBitmap(CDA_LIBRARY_ASSETID(sclId, libItemId));
            //}

            //if (bitmap.PointsToNull())
            {
               ////search the bitmap based only on lib item id
               //bitmap = tryGetBitmap(CDA_LIBRARY_ASSETID(0, libItemId));

               //if (bitmap.PointsToNull())
               {
                  //search the bitmap using the default function (which iterated through all the bitmaps and compares strings)
                  Candera::Id id = Candera::Internal::AssetProviderFunctions::GetIdByName(&Candera::DefaultAssetProvider::GetInstance(), Candera::BitmapLib, _bitmapCanderaName.c_str());
                  bitmap = tryGetBitmap(id);

                  if (bitmap.PointsToNull())
                  {
                     ETG_TRACE_ERR_THR(("ImageLoader:DelayedBitmapImage2D Asset image [%s] not found!", _bitmapCanderaName.c_str()));
                  }
               }
            }

            SetBitmap(bitmap);

            _initialized = true;
         }

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3)))
         return Base::UploadInternal(loadingHint);
#else
         return Base::UploadInternal();
#endif
      }

   private:

      Candera::MemoryManagement::SharedPointer<Candera::Bitmap> tryGetBitmap(const Candera::Id& id)
      {
         Candera::MemoryManagement::SharedPointer<Candera::Bitmap> bitmap = Candera::DefaultAssetProvider::GetInstance().GetBitmapById(id);
         //if a bitmap is found but the name does not match it means that we have found a wrong bitmap
         if (!bitmap.PointsToNull() && (Candera::StringPlatform::CompareStrings(_bitmapCanderaName.c_str(), bitmap->GetName()) != 0))
         {
            bitmap.Release();
         }
         return bitmap;
      }

      std::string _bitmapCanderaName;
      bool _initialized;
};


Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> ImageLoader::getAssetBitmapImage(const Candera::Char* canderaName)
{
   if ((canderaName == NULL) || (canderaName[0] == 0))
   {
      return Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D>();
   }
   return Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D>(CANDERA_NEW(DelayedBitmapImage2D)(canderaName));
}


bool ImageLoader::saveImage(void* data, uint32_t size, uint32_t width, uint32_t height, const char* filePath, bool fileOverwrite)
{
   bool success = false;
   if (filePath == 0)
   {
      return false;
   }

   getLock().Obtain();

   ILuint handle = createILImage();

   if (handle != 0)
   {
      if (ilLoadDataL(data, size, width, height, 1, 4))
      {
         if (iluFlipImage())
         {
            if (fileOverwrite)
            {
               ilEnable(IL_FILE_OVERWRITE);;
            }

            if (ilSaveImage(filePath))
            {
               success = true;
               ETG_TRACE_USR4_THR(("Image saved to %s", filePath));
            }
         }
      }
   }

   if (success == false)
   {
      ETG_TRACE_ERR_THR(("ImageLoader::saveImage: IL error code %d", ilGetError()));
   }
   destroyILImage(handle);

   getLock().Release();

   return success;
}


}
}
