/* ***************************************************************************************
* FILE:          HMIRenderViewVisitor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIRenderViewVisitor.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "gui_std_if.h"

#include "HMIRenderViewVisitor.h"
#include "Trace/StartupInvestigation.h"
#include "hmibase/util/TraceUtils.h"
#include "CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h"
#include <algorithm>

#if defined(CANDERA_2D_ENABLED)
#include <Courier/Visualization/ViewScene2D.h>
#endif
#if defined(CANDERA_3D_ENABLED)
#include <Courier/Visualization/ViewScene3D.h>
#endif

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIRenderViewVisitor.cpp.trc.h"
#endif

namespace hmibase {
namespace view {

#if defined(CANDERA_2D_ENABLED)
void HMIRenderViewVisitor::Visit(Courier::ViewScene2D* view)
{
   updateActiveViewsContainer(static_cast<Courier::ViewScene*>(view));
}


#endif

#if defined(CANDERA_3D_ENABLED)
void HMIRenderViewVisitor::Visit(Courier::ViewScene3D* view)
{
   updateActiveViewsContainer(static_cast<Courier::ViewScene*>(view));
}


#endif

void HMIRenderViewVisitor::OnEnd()
{
   if ((isNewView) && (m_vecActiveViews.size() > 0))
   {
      tViewInfoVector::iterator it = m_vecActiveViews.begin();

      for (; it != m_vecActiveViews.end(); ++it)
      {
         if (it->update)
         {
            Candera::String viewname = it->viewName.c_str();
            for (std::vector<unsigned int>::const_iterator cit = it->surfaceIds.begin(); cit != it->surfaceIds.end(); ++cit)
            {
               ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_PERFORMANCE_MP, "[%25s] Scene [%50s] initially rendered on surface %d",
                                       hmibase::trace::getAppName().c_str(),
                                       it->viewName.c_str(),
                                       *cit));
               Courier::Message* postMsg = COURIER_MESSAGE_NEW(RenderingCompleteMsg)(*cit, viewname, it->rtUnitType, it->viewState);
               if (postMsg != 0)
               {
                  postMsg->Post();
               }
            }
         }
      }
   }
   m_vecLastRenderedViews = m_vecActiveViews;
   m_vecActiveViews.clear();
}


void HMIRenderViewVisitor::updateActiveViewsContainer(Courier::ViewScene* view)
{
   if ((view != 0) && (view->IsRenderingEnabled()))
   {
      bool _update = false;
      tViewInfoVector::iterator it = m_vecLastRenderedViews.begin();

      for (; it != m_vecLastRenderedViews.end(); ++it)
      {
         if (strcmp(view->GetId().CStr(), it->viewName.c_str()) == 0)
         {
            break;
         }
      }

      if (m_vecLastRenderedViews.end() == it)
      {
         isNewView = true;
         _update = true;
         static bool firstScreenDrawn[] = { false, false };
         unsigned char idx = view->Is2D() ? 0 : 1;
         if (!firstScreenDrawn[idx])
         {
            firstScreenDrawn[idx] = true;
            HMI_STARTUP_INVESTIGATION(view->Is2D() ? hmibase::trace::StartupInvestigation::MP_FIRST_SCREEN_DRAWN_2D : hmibase::trace::StartupInvestigation::MP_FIRST_SCREEN_DRAWN_3D);
         }
      }

      //Courier::RTPtrVector RTVectorLoc;
      //view->GetRenderTargetPtrVector(RTVectorLoc, true);

      //Courier::RTPtrVector::Iterator iter = RTVectorLoc.Begin();
      ViewInfo vI;
      vI.viewName = view->GetId().CStr();
      vI.update = _update;
      vI.rtUnitType = Candera::DevicePackageDescriptor::Mixed2D3DDisplayTarget;
      vI.viewState = hmibase::enNormal;

      unsigned int surfaceId = 0;

      if (view->Is3D())
      {
         Courier::ViewScene3D* view3D = static_cast<Courier::ViewScene3D*>(view);
         if (view3D != 0)
         {
            Courier::Vector< Candera::Camera* > cameras3D = view3D->GetCameraPtrVector();

            for (FeatStd::SizeType index = 0; index < cameras3D.Size(); ++index)
            {
               if ((cameras3D[index] != NULL) && cameras3D[index]->IsRenderingEnabled())
               {
                  Candera::RenderTarget* rt = cameras3D[index]->GetRenderTarget();
                  if (NULL != rt)
                  {
                     vI.rtUnitType = getRTUnitCategoryType(rt);
                     if ((NULL != view->GetViewHandler()) && (NULL != view->GetViewHandler()->GetRenderer()))
                     {
                        ::Courier::Gdu* gdu = view->GetViewHandler()->GetRenderer()->GetGdu(rt);
                        if ((NULL != gdu) && (gdu->IsOffscreen()))
                        {
                           //hack to update blurred state when Rendering a viewscene
                           //it happens to be GDU name is NULL for those offscreen RT created by Blurwidget
                           if (rt->GetGraphicDeviceUnit()->GetName() == NULL)
                           {
                              vI.viewState = hmibase::enBlurred;
                           }
                           Candera::Internal::GraphicDeviceUnitOwnerAccess* owner = (&gdu->GetOwnerAccess());
                           if ((NULL != owner) && (NULL != owner->GetGraphicDeviceUnitOwner()))
                           {
                              rt = owner->GetGraphicDeviceUnitOwner()->ToRenderTarget3D();
                           }
                        }
                     }
                     if (getSurfaceId(rt, surfaceId))
                     {
                        std::vector<unsigned int>::const_iterator cit = std::find(vI.surfaceIds.begin(), vI.surfaceIds.end(), surfaceId);
                        if (cit == vI.surfaceIds.end())
                        {
                           vI.surfaceIds.push_back(surfaceId);
                        }
                     }
                  }
               }
            }
         }
      }
      else if (view->Is2D())
      {
         Courier::ViewScene2D* view2D = static_cast<Courier::ViewScene2D*>(view);
         Courier::ViewScene2D::CameraPtrVector cameras = view2D->GetCameraPtrVector();

         for (FeatStd::SizeType index = 0; index < cameras.Size(); ++index)
         {
            if ((cameras[index] != NULL) && cameras[index]->IsRenderingEnabled())
            {
               Candera::RenderTarget* rt = cameras[index]->GetRenderTarget();
               if (NULL != rt)
               {
                  vI.rtUnitType = getRTUnitCategoryType(rt);
                  if ((NULL != view->GetViewHandler()) && (NULL != view->GetViewHandler()->GetRenderer()))
                  {
                     ::Courier::Gdu* gdu = view->GetViewHandler()->GetRenderer()->GetGdu(rt);
                     if ((NULL != gdu) && (gdu->IsOffscreen()))
                     {
                        //hack to update blurred state when Rendering a viewscene
                        //it happens to be GDU name is NULL for those offscreen RT created by Blurwidget
                        if (rt->GetGraphicDeviceUnit()->GetName() == NULL)
                        {
                           vI.viewState = hmibase::enBlurred;
                        }
                        Candera::Internal::GraphicDeviceUnitOwnerAccess* owner = (&gdu->GetOwnerAccess());
                        if ((NULL != owner) && (NULL != owner->GetGraphicDeviceUnitOwner()))
                        {
                           rt = owner->GetGraphicDeviceUnitOwner()->ToRenderTarget2D();
                        }
                     }
                  }
                  if (getSurfaceId(rt, surfaceId))
                  {
                     std::vector<unsigned int>::const_iterator cit = std::find(vI.surfaceIds.begin(), vI.surfaceIds.end(), surfaceId);
                     if (cit == vI.surfaceIds.end())
                     {
                        vI.surfaceIds.push_back(surfaceId);
                     }
                  }
               }
            }
         }
      }
      if (vI.surfaceIds.size() > 0)
      {
         m_vecActiveViews.push_back(vI);
      }
   }
}


/**************************************************************************************
*FUNCTION:     IsEqualSurfaceId
*DESCRIPTION:  Checks if RT surface id is same as surface id in param
*PARAMETER:    1. Render target pointer
*               2. Surface id
*
*RETURNVALUE:   bool
**************************************************************************************/
bool HMIRenderViewVisitor::getSurfaceId(const Candera::Surface* surface, unsigned int& surfaceId)
{
   if (0 == surface)
   {
      return false;
   }
   Candera::GraphicDeviceUnit* gdu = surface->GetGraphicDeviceUnit();
   if (0 == gdu)
   {
      return false;
   }
   const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(gdu->GetUnitType());
   if (0 == metaInfo)
   {
      return false;
   }
   Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
   if (0 == propMetaInfo)
   {
      return false;
   }

   Candera::Char buffer[32];
   buffer[0] = '\0';
   if (!propMetaInfo->Get(gdu, buffer, 32))
   {
      return false;
   }
   ::sscanf(buffer, "%u", &surfaceId);
   return true;
}


Candera::DevicePackageDescriptor::UnitCategory HMIRenderViewVisitor::getRTUnitCategoryType(Candera::RenderTarget* rt)
{
   Candera::DevicePackageDescriptor::UnitCategory unit = Candera::DevicePackageDescriptor::GetUnitCategory(rt->GetGraphicDeviceUnit()->GetUnitType());
   return  unit;
}


}
}
