/* ***************************************************************************************
* FILE:          ColorManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ColorManager is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "gui_std_if.h"
#include "ColorManager.h"

#include <CanderaAssetLoader/AssetLoaderBase/AssetIdMacros.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetDataHandle.h>

#if ((CANDERA_VERSION_MAJOR == 3) && (CANDERA_VERSION_MINOR <= 4))
#include <CanderaAssetLoader/AssetLoaderBase/CFFAssetReader.h>
#else
#include <CanderaAssetLoader/AssetLoaderBase/CffReader/CffAssetReaderBase.h>
#endif
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_COLOR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/ColorManager.cpp.trc.h>
#endif

namespace hmibase {
namespace util {
namespace color {


/************************************************************************/
static const char* getAppName()
{
   static std::string _appName(hmibase::trace::TraceUtils::getInstance().getAppName());
   return _appName.c_str();
}


/************************************************************************/
const Color& ColorTable::getDefaultColor() const
{
   return _defaultColor;
}


/************************************************************************/
void ColorTable::setDefaultColor(const Color& defaultColor)
{
   _defaultColor = defaultColor;
}


/************************************************************************/
const Color& ColorTable::getColor(ColorId id) const
{
   ColorMapType::const_iterator it = _colorMap.find(id);
   if (it == _colorMap.end())
   {
      return _defaultColor;
   }
   return it->second;
}


/************************************************************************/
void ColorTable::setColor(ColorId id, const Color& color)
{
   _colorMap[id] = color;
}


/************************************************************************/
ColorManager::ColorManager() : _updateSequenceNr(0), _currentTable(NULL)
{
   updateSequenceNumber();
}


/************************************************************************/
ColorManager::~ColorManager()
{
   _currentTable = NULL;
}


/************************************************************************/
ColorManager& ColorManager::getInstance()
{
   static ColorManager _instance;
   return _instance;
}


/************************************************************************/
const ColorTable& ColorManager::getDefaultTable() const
{
   return _defaultTable;
}


/************************************************************************/
void ColorManager::updateSequenceNumber()
{
   ++_updateSequenceNr;
}


/************************************************************************/
const ColorTable* ColorManager::getTable(const TableId& id) const
{
   TableMapType::const_iterator it = _tableMap.find(id);
   if (it == _tableMap.end())
   {
      return NULL;
   }
   return it->second;
}


/************************************************************************/
void ColorManager::setCurrentTable(const ColorTable* table)
{
   if (_currentTable != table)
   {
      _currentTable = table;
      updateSequenceNumber();
   }
}


/************************************************************************/
class RawResourceReader
{
   public:
      RawResourceReader(Candera::ResourceDataHandle& resourceHandle)
         : _resourceHandle(resourceHandle), _assetDataHandle(resourceHandle), _position(0)
      {
      }

      /************************************************************************/
      bool isValid() const
      {
         return _assetDataHandle.IsValid();
      }

      /************************************************************************/
      template <typename T>
      bool read(T& val)
      {
         const FeatStd::UInt size = getSize(val);
         if (_position + size <= _resourceHandle.m_size)
         {
            val = Candera::Internal::CFFReader::CffGetValSafe<T>(_assetDataHandle, _position);
            _position += size;
            return true;
         }
         return false;
      }

   private:
      /************************************************************************/
      RawResourceReader(const RawResourceReader&);
      RawResourceReader& operator=(const RawResourceReader&);

      /************************************************************************/
      template <typename T>
      static FeatStd::UInt getSize(T&);

      /************************************************************************/
      Candera::ResourceDataHandle& _resourceHandle;
      Candera::Internal::AssetDataHandle _assetDataHandle;
      FeatStd::UInt _position;
};


/************************************************************************/
template<> FeatStd::UInt RawResourceReader::getSize<FeatStd::Float>(FeatStd::Float&)
{
   return Candera::Internal::CFF_FLOAT_SIZE;
}


/************************************************************************/
template<> FeatStd::UInt RawResourceReader::getSize<FeatStd::Int>(FeatStd::Int&)
{
   return Candera::Internal::CFF_INT_SIZE;
}


/************************************************************************/
template<> FeatStd::UInt RawResourceReader::getSize<FeatStd::UInt>(FeatStd::UInt&)
{
   return Candera::Internal::CFF_UINT_SIZE;
}


/************************************************************************/
ColorManagerInitializer::ColorManagerInitializer(ColorManager& colorManager) : _colorManager(colorManager)
{
}


/************************************************************************/
ColorTable& ColorManagerInitializer::defaultTable()
{
   return _colorManager._defaultTable;
}


/************************************************************************/
ColorManager::TableMapType& ColorManagerInitializer::tableMap()
{
   return _colorManager._tableMap;
}


/************************************************************************/
DefaultColorManagerInitializer::DefaultColorManagerInitializer(ColorManager& colorManager, const std::string& colorTableRawResourceName)
   : Base(colorManager), _colorTableRawResourceName(colorTableRawResourceName)
{
}


/************************************************************************/
bool DefaultColorManagerInitializer::initialize()
{
   const char* resName = _colorTableRawResourceName.size() > 0 ? _colorTableRawResourceName.c_str() : "Global#Config#Colors#Colors.dat";
   Candera::Id resId = Candera::Internal::AssetProviderFunctions::GetIdByName(&Candera::DefaultAssetProvider::GetInstance(), Candera::RawResourceLib, resName);
   Candera::ResourceDataHandle resHandle = Candera::DefaultAssetProvider::GetInstance().GetRawResourceById(resId);

   ETG_TRACE_USR1_THR(("ColorManager resourceHandle valid=%u, size=%u, app=%s",
                       resHandle.m_accessor.IsValid(), resHandle.m_size, getAppName()));

   if (!resHandle.m_accessor.IsValid())
   {
      ETG_TRACE_ERR_THR(("ColorManager invalid resource handle for data app=%s", getAppName()));
      return false;
   }

   RawResourceReader reader(resHandle);
   FeatStd::UInt version = 0;
   if (reader.read(version))
   {
      ETG_TRACE_USR1_THR(("ColorManager version=%u, app=%s", version, getAppName()));

      if (version != VERSION)
      {
         ETG_TRACE_ERR_THR(("ColorManager unsupported data version (found=%u, expected=%u), app=%s", version, VERSION, getAppName()));
         return false;
      }

      FeatStd::UInt tableCount = 0;
      if (reader.read(tableCount))
      {
         ETG_TRACE_USR1_THR(("ColorManager tableCount=%u, app=%s", tableCount, getAppName()));

         for (FeatStd::UInt tableIndex = 0; tableIndex < tableCount; ++tableIndex)
         {
            ColorTable* table = NULL;
            FeatStd::UInt tableId = 0;
            if (reader.read(tableId))
            {
               ETG_TRACE_USR1_THR(("ColorManager tableId=%u, app=%s", tableId, getAppName()));

               ColorManager::TableMapType::iterator tableIt = tableMap().find(tableId);
               if ((tableIt == tableMap().end()) || (tableIt->second == NULL))
               {
                  if (tableId == 0)
                  {
                     table = &(defaultTable());
                  }
                  else
                  {
                     table = FEATSTD_NEW(ColorTable);
                  }
                  tableMap()[tableIndex] = table;
               }
               else
               {
                  table = tableIt->second;
               }
            }

            FeatStd::UInt colorCount = 0;
            if (reader.read(colorCount))
            {
               ETG_TRACE_USR1_THR(("ColorManager colorCount=%u, app=%s", colorCount, getAppName()));

               FeatStd::UInt colorId = 0;
               FeatStd::Float red = 0.0f;
               FeatStd::Float green = 0.0f;
               FeatStd::Float blue = 0.0f;
               FeatStd::Float alpha = 0.0f;
               for (FeatStd::UInt colorIndex = 0; colorIndex < colorCount; ++colorIndex)
               {
                  if (reader.read(colorId) && reader.read(red) && reader.read(green) && reader.read(blue) && reader.read(alpha) && (table != NULL))
                  {
                     ETG_TRACE_USR1_THR(("ColorManager addColor tableId=%u, colorId=%u, red=%f, green=%f, blue=%f, alpha=%f, app=%s",
                                         tableId, colorId, red, green, blue, alpha, getAppName()));

                     table->setColor(colorId, Candera::Color(red, green, blue, alpha));
                  }
               }
            }
         }
      }
   }
   return true;
}


}
}


}
