/* ***************************************************************************************
* FILE:          ColorAccessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ColorAccessor is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "sys_std_if.h"

/************************************************************************/

// these declarations needs to be in the begining of the file to prevent compilation issues with gcc!

// this info is from sesa
// Due to the fact that TypeConvert function is further referenced in the below header it must be defined first as above to avoid compile warnings in gcc
// ToDo: split StdTypeConverter in two files: one with the above definitions for TypeConvert and one with the RegisterStdTypeConverters below

namespace Candera {
class Color;
}


namespace hmibase {
namespace util {
namespace color {
class ColorAccessor;
}


}
}


namespace Courier {

static bool TypeConvert(::Candera::Color& col, const ::hmibase::util::color::ColorAccessor& colAccessor);
static bool TypeConvert(::hmibase::util::color::ColorAccessor& colAccessor, const ::Candera::Color& col);
}


/************************************************************************/


#include "ColorAccessor.h"
#include "ColorManager.h"

#include <Courier/DataBinding/TypeConverter.h>
//#include <Candera/EngineBase/DataTypes/ColorDataType.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_COLOR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/ColorAccessor.cpp.trc.h>
#endif

namespace hmibase {
namespace util {
namespace color {


/************************************************************************/
static bool isColorDataEqual(const Color::Data& x, const Color::Data& y)
{
   return x.red == y.red
          && x.green == y.green
          && x.blue == y.blue
          && x.alpha == y.alpha;
}


/************************************************************************/
ColorAccessor::ColorAccessor() : _isSysColor(false)
{
   _data._colorData = Color::Data();
}


/************************************************************************/
ColorAccessor::ColorAccessor(const Color& c) : _isSysColor(false)
{
   _data._colorData = c.GetData();
}


/************************************************************************/
ColorAccessor::ColorAccessor(const Color::Data& colorData) : _isSysColor(false)
{
   _data._colorData = colorData;
}


/************************************************************************/
ColorAccessor::ColorAccessor(ColorId id) : _isSysColor(true)
{
   _data._sysColor._colorId = id;
   _data._sysColor._updateSeqNr = 0;
   _data._sysColor._color = NULL;
}


/************************************************************************/
ColorAccessor::~ColorAccessor()
{
   _data._sysColor._color = NULL;
}


/************************************************************************/
ColorAccessor& ColorAccessor::operator = (const Color& color)
{
   _isSysColor = false;
   _data._colorData = color.GetData();
   return *this;
}


/************************************************************************/
ColorAccessor& ColorAccessor::operator = (const Color::Data& colorData)
{
   _isSysColor = false;
   _data._colorData = colorData;
   return *this;
}


/************************************************************************/
ColorAccessor& ColorAccessor::operator = (ColorId id)
{
   _isSysColor = true;
   _data._sysColor._colorId = id;
   _data._sysColor._updateSeqNr = 0;
   _data._sysColor._color = NULL;
   return *this;
}


/************************************************************************/
bool ColorAccessor::operator == (const ColorAccessor& other) const
{
   if (&other == this)
   {
      return true;
   }
   if (_isSysColor != other._isSysColor)
   {
      return false;
   }
   if (_isSysColor)
   {
      return _data._sysColor._colorId == other._data._sysColor._colorId;
   }
   else
   {
      return isColorDataEqual(_data._colorData, other._data._colorData);
   }
}


/************************************************************************/
bool ColorAccessor::operator != (const ColorAccessor& other) const
{
   return !(*this == other);
}


/************************************************************************/
bool ColorAccessor::operator == (const Color& otherColor) const
{
   return isColorDataEqual(getColorData(), otherColor.GetData());
}


/************************************************************************/
bool ColorAccessor::operator != (const Color& otherColor) const
{
   return !(*this == otherColor);
}


/************************************************************************/
bool ColorAccessor::operator == (const Color::Data& otherData) const
{
   return isColorDataEqual(getColorData(), otherData);
}


/************************************************************************/
bool ColorAccessor::operator != (const Color::Data& otherData) const
{
   return !(*this == otherData);
}


/************************************************************************/
const Color::Data& ColorAccessor::getColorData() const
{
   if (_isSysColor)
   {
      ColorManager& colorManager = ColorManager::getInstance();
      if ((_data._sysColor._updateSeqNr != colorManager.getUpdateSequenceNr()) || (_data._sysColor._color == NULL))
      {
         _data._sysColor._color = &(colorManager.getCurrentTable().getColor(_data._sysColor._colorId));
         _data._sysColor._updateSeqNr = colorManager.getUpdateSequenceNr();
      }
      return _data._sysColor._color->GetData();
   }
   else
   {
      return _data._colorData;
   }
}


}
}


}


namespace Candera {
namespace MetaInfo {

using namespace Candera;

/************************************************************************/
enum ColorDataType
{
   StaticColor = 0,    ///0;Red,Green,Blue,Alpha
   DynamicColorId = 1  ///1;ColorId
};


/************************************************************************/
bool DataType< ::hmibase::util::color::ColorAccessor >::ConvertToString(const ::hmibase::util::color::ColorAccessor& val, Char* buf, UInt bufLen)
{
   bool result = false;

   //Red, Green, Blue, Alpha
   //$ColorId
   if ((buf != NULL) && (bufLen > 2))
   {
      if (val.isSystemColor())
      {
         StringPlatform::Copy(buf, "$");
         ++buf;
         --bufLen;

         if (SerializationPlatform::FromBinary(buf, bufLen, static_cast<UInt>(val.getSystemColorId())) == 1)
         {
            result = true;
         }
      }
      else
      {
         const Color::Data& colorData = val.getColorData();
         Float tab[4] = { colorData.red, colorData.green, colorData.blue, colorData.alpha };
         if (SerializationPlatform::FromBinary(buf, bufLen, ",", tab, tab + 4) == 4)
         {
            result = true;
         }
      }
   }
   return result;
}


/************************************************************************/
bool DataType< ::hmibase::util::color::ColorAccessor >::ConvertFromString(::hmibase::util::color::ColorAccessor& val, const Char* buf)
{
   bool result = false;

   //0;Red,Green,Blue,Alpha
   //1;ColorId
   Int count;
   UInt type;
   if ((buf != NULL) && (SerializationPlatform::ToBinary(buf, type, &count) == 1))
   {
      buf += count;
      ++buf;//skip ';'

      switch (type)
      {
         case 0:
         {
            Float tab[4] = { 0.0F, 0.0F, 0.0F, 0.0F };
            if (SerializationPlatform::ToBinary(buf, ",", tab, tab + 4, &count) == 4)
            {
               val = Color(tab[0], tab[1], tab[2], tab[3]);
               result = true;
            }
         }
         break;

         case 1:
         {
            UInt colorId;
            if (SerializationPlatform::ToBinary(buf, colorId, &count) == 1)
            {
               val = colorId;
               result = true;
            }
         }
         break;

         default:
            break;
      }
   }

   return result;
}


}
}


namespace FeatStd {

template<> UInt32 StringBufferAppender< ::hmibase::util::color::ColorAccessor >::Append(StringBuffer& stringBuffer, const ::hmibase::util::color::ColorAccessor& object)
{
   UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("::ColorAccessor {");
   tcharCount += stringBuffer.Append("isSystemColor = ");
   tcharCount += stringBuffer.AppendObject(object.isSystemColor());
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("systemColorId = ");
   tcharCount += stringBuffer.AppendObject(object.getSystemColorId());
   tcharCount += stringBuffer.Append(", ");
   tcharCount += stringBuffer.Append("color = ");
   tcharCount += stringBuffer.AppendObject(Candera::Color(object.getColorData()));
   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


}


namespace Courier {

static bool TypeConvert(::Candera::Color& col, const ::hmibase::util::color::ColorAccessor& colAccessor)
{
   col = colAccessor.getColorData();
   return true;
}


static bool TypeConvert(::hmibase::util::color::ColorAccessor& colAccessor, const ::Candera::Color& col)
{
   colAccessor = col.GetData();
   return true;
}


static bool RegisterColorConverter()
{
   static ::Courier::TypeConverter< ::Candera::Color, ::hmibase::util::color::ColorAccessor > _colorConverter;
   return true;
}


static bool _registerColorConverter = RegisterColorConverter();
}
