/* ***************************************************************************************
* FILE:          CloseOnTouchSession.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CloseOnTouchSession.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "CloseOnTouchSession.h"
#include "Courier/Visualization/FrameworkWidget.h"
#include "View/IMessageSystem.h"
#include <algorithm>

#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_INPUT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CloseOnTouchSession.cpp.trc.h"
#endif

namespace hmibase {
namespace input {

unsigned int CloseOnTouchSession::mCloseOnExternalTouchCounter = 0;

bool CloseOnTouchSession::ProcessMessage(const Courier::Message& msg)
{
   // check if we receive a registration message
   const RegisterForCloseOnTouchSessionReqMsg* registrationMsgPtr = Courier::message_cast<const RegisterForCloseOnTouchSessionReqMsg*>(&msg);
   const SetClosePopupExternalTouchMsg* setClosePopupMsgPtr = Courier::message_cast<const SetClosePopupExternalTouchMsg*>(&msg);
   if (registrationMsgPtr == 0 && setClosePopupMsgPtr == 0)
   {
      // return Base::ProcessMessage(msg);
      return Base::ProcessMessage(msg) || (msg.GetId() == Courier::TouchMsg::ID); // modified by SESA 18.08.2015
   }

   if (setClosePopupMsgPtr != 0)
   {
      updateCloseOnExternalTouch(setClosePopupMsgPtr->GetStatus());
      return true;
   }
   if (registrationMsgPtr != 0)
   {
      ETG_TRACE_USR4_THR(("*** CloseOnTouchSession : RegisterForCloseOnTouchSessionReqMsg received %d", registrationMsgPtr->GetRegister()));
      std::vector<Courier::ViewId>::iterator it = std::find(mRegisteredScenes.begin(), mRegisteredScenes.end(), registrationMsgPtr->GetViewId());
      if (registrationMsgPtr->GetRegister() == true)
      {
         if (it == mRegisteredScenes.end())
         {
            ETG_TRACE_USR2_THR(("*** CloseOnTouchSession : add registration for %s", registrationMsgPtr->GetViewId().CStr()));
            mRegisteredScenes.push_back(registrationMsgPtr->GetViewId());
         }
      }
      else
      {
         if (it != mRegisteredScenes.end())
         {
            ETG_TRACE_USR2_THR(("*** CloseOnTouchSession : release registration of %s", (*it).CStr()));
            mRegisteredScenes.erase(it);
         }
      }
   }
   return true;
}


bool CloseOnTouchSession::OnMessage(const Courier::TouchMsg* touchMsgPtr)
{
   COURIER_DEBUG_ASSERT(touchMsgPtr != 0);
   if (!touchMsgPtr)
   {
      return false;
   }

   bool msgConsumed = Base::OnMessage(touchMsgPtr);
   bool closeonInternalTouch = false;

   if ((touchMsgPtr->GetState() == Courier::TouchMsgState::Up) || (touchMsgPtr->GetState() == Courier::TouchMsgState::Down))
   {
      if (msgConsumed && IsTouchOnPopupSurface(touchMsgPtr->GetSourceId()))
      {
         //since message consumed by widget , just return and let popup in application handle it
         return msgConsumed;
      }
      //logic for close on popup registration, keeping this logic intent along with close on external touch
      for (std::vector<Courier::ViewId>::iterator it = mRegisteredScenes.begin();
            !msgConsumed && it != mRegisteredScenes.end(); ++it)
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(CloseOnTouchReqMsg)(*it, touchMsgPtr->GetState());
         if (msg != 0)
         {
            ETG_TRACE_USR2_THR(("*** CloseOnTouchSession : Fire CloseOnTouchReqMsg to %s", (*it).CStr()));
            msgConsumed = msg->Post();
            closeonInternalTouch = true;
         }
      }
      //logic for Close on external touch processing
      if ((!closeonInternalTouch) && (mCloseOnExternalTouchCounter > 0))
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(ClosePopupOnExternalTouchReqMsg)(touchMsgPtr->GetState(), touchMsgPtr->GetSourceId());
         if (msg != 0)
         {
            ETG_TRACE_USR2_THR(("*** CloseOnTouchSession : Fire ClosePopupOnExternalTouchReqMsg "));
            msgConsumed = msg->Post();
         }
      }
   }
   return msgConsumed;
}


bool CloseOnTouchSession::IsTouchOnPopupSurface(Courier::UInt32 const surfaceId) const
{
   return (((static_cast<unsigned int>(hmibase::SURFACEID_POPUPBASE)      <= surfaceId) &&
            (static_cast<unsigned int>(hmibase::SURFACEID_POPUPBASE_MAX)  >= surfaceId)));
}


void CloseOnTouchSession::updateCloseOnExternalTouch(bool const status)
{
   if (status)
   {
      mCloseOnExternalTouchCounter++;
   }
   else
   {
      if (mCloseOnExternalTouchCounter > 0) //to avoid negative values
      {
         mCloseOnExternalTouchCounter--;
      }
   }
   ETG_TRACE_USR4_THR(("*** CloseOnTouchSession : mCloseOnExternalTouchCounter [%d]", mCloseOnExternalTouchCounter));
}


}
}
