/* ***************************************************************************************
* FILE:          CGIAppViewFactoryBase.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CGIAppViewFactoryBase.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "CGIAppViewFactoryBase.h"
#include "ViewProfilehandler.h"
#include "ViewScene2D.h"
#include "ViewScene3D.h"

#include "SceneMapping.h"

#include "hmi_trace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_VIEW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/CGIAppViewFactoryBase.cpp.trc.h"
#endif

namespace hmibase {
   namespace view {
      /*-----------------------------------------------------------------
          SceneMappingForHash : Get the corresponding Hash id from the SceneMap
          class
          -----------------------------------------------------------------*/
      const SceneMapping* SceneMappingForHash(FeatStd::UInt32 hash, const FeatStd::UInt16 viewCount, const SceneMapping* sceneMap)
      {
         for (FeatStd::UInt32 i = 0; i < viewCount; i++)
         {
            if (sceneMap[i]._nameID.GetHashId() == hash)
            {
               return &(sceneMap[i]);
            }
         }
         if (SceneMapping::base)
         {
            for (FeatStd::UInt32 i = 0; i < SceneMapping::baseCnt; i++)
            {
               if (SceneMapping::base[i]._nameID.GetHashId() == hash)
               {
                  return &(SceneMapping::base[i]);
               }
            }
         }
         return 0;
      }


      /************************************************************************/
      const SceneMapping* SceneMappingForBaseName(const Courier::ViewId& id, unsigned int profileid, const FeatStd::UInt16 viewCount, const SceneMapping* sceneMap)
      {
         for (FeatStd::UInt32 i = 0; i < viewCount; i++)
         {
            if ((sceneMap[i]._profileID == profileid) && (sceneMap[i]._baseviewnameID.GetHashId() == id.GetHashId()))
            {
               return &(sceneMap[i]);
            }
         }
         if (SceneMapping::base)
         {
            for (FeatStd::UInt32 i = 0; i < SceneMapping::baseCnt; i++)
            {
               if ((SceneMapping::base[i]._profileID == profileid) && (SceneMapping::base[i]._baseviewnameID.GetHashId() == id.GetHashId()))
               {
                  return &(SceneMapping::base[i]);
               }
            }
         }
         return 0;
      }


      /************************************************************************/
      const SceneMapping* SceneMappingForName(const Courier::ViewId& id, const FeatStd::UInt16 viewCount, const SceneMapping* sceneMap)
      {
         for (FeatStd::UInt32 i = 0; i < viewCount; i++)
         {
            if ((sceneMap[i]._nameID.GetHashId() == id.GetHashId()))
            {
               return &(sceneMap[i]);
            }
         }
         if (SceneMapping::base)
         {
            for (FeatStd::UInt32 i = 0; i < SceneMapping::baseCnt; i++)
            {
               if ((SceneMapping::base[i]._nameID.GetHashId() == id.GetHashId()))
               {
                  return &(SceneMapping::base[i]);
               }
            }
         }
         return 0;
      }


      /************************************************************************/
      CGIAppViewFactoryBase::CreateDefaultViewSceneFunc CGIAppViewFactoryBase::_createDefaultViewSceneFunc = NULL;


      /************************************************************************/
      CGIAppViewFactoryBase::CGIAppViewFactoryBase() : _viewCnt(0), _scenes(0)
      {
         ViewProfilehandler::SetViewMap(this);
      }


      /************************************************************************/
      CGIAppViewFactoryBase::CGIAppViewFactoryBase(unsigned short cnt, SceneMapping* sm) : _viewCnt(cnt), _scenes(sm)
      {
         ViewProfilehandler::SetViewMap(this);
      }


      /************************************************************************/
      CGIAppViewFactoryBase::~CGIAppViewFactoryBase()
      {
         ViewProfilehandler::SetViewMap(0);
         _scenes = 0;
      }


      /************************************************************************/
      Courier::ViewScene* CGIAppViewFactoryBase::CreateViewScene(bool is2D)
      {
         return CreateDefaultViewScene(is2D, true);
      }


      /************************************************************************/
      void CGIAppViewFactoryBase::SetCreateDefaultViewSceneFunc(CreateDefaultViewSceneFunc func)
      {
         _createDefaultViewSceneFunc = func;
      }


      /************************************************************************/
      Courier::ViewScene* CGIAppViewFactoryBase::CreateDefaultViewScene(bool is2D, bool isManagedByViewHandler)
      {
         Courier::ViewScene* view = NULL;

         // use the custom method if available
         if (_createDefaultViewSceneFunc != NULL)
         {
            ETG_TRACE_USR4_THR(("CGIAppViewFactoryBase::CreateDefaultViewScene type=%uD (custom create method), managedByViewHandler=%u", is2D ? 2 : 3, isManagedByViewHandler));
            view = _createDefaultViewSceneFunc(is2D, isManagedByViewHandler);
         }
         else
         {
            if (is2D)
            {
               ETG_TRACE_USR4_THR(("CGIAppViewFactoryBase::CreateDefaultViewScene type=2D (hmibase::view::ViewScene2D), managedByViewHandler=%u", isManagedByViewHandler));
               view = FEATSTD_NEW(hmibase::view::ViewScene2D)(isManagedByViewHandler);
            }
            else
            {
               ETG_TRACE_USR4_THR(("CGIAppViewFactoryBase::CreateDefaultViewScene type=3D (hmibase::view::ViewScene3D), managedByViewHandler=%u", isManagedByViewHandler));
               view = FEATSTD_NEW(hmibase::view::ViewScene3D)(isManagedByViewHandler);
            }
         }

         return view;
      }


      /************************************************************************/
      Courier::ViewScene* CGIAppViewFactoryBase::CreateDefaultViewScene(const FeatStd::Char* viewName, bool isManagedByViewHandler)
      {
         // initialize the view scene type map
         if (_viewSceneTypeMap.empty())
         {
            InitViewSceneTypeMap();
         }

         Courier::ViewScene* view = NULL;
         enViewSceneType viewSceneType = GetViewSceneType(viewName);
         switch (viewSceneType)
         {
            case hmibase::view::CGIAppViewFactoryBase::en2D:
               ETG_TRACE_USR1_THR(("CGIAppViewFactoryBase::CreateDefaultViewScene type=2D, managedByViewHandler=%u, name=(%p)%s", isManagedByViewHandler, viewName, viewName));
               view = CreateDefaultViewScene(true, isManagedByViewHandler);
               break;

            case hmibase::view::CGIAppViewFactoryBase::en3D:
               ETG_TRACE_USR1_THR(("CGIAppViewFactoryBase::CreateDefaultViewScene type=3D, managedByViewHandler=%u, name=(%p)%s", isManagedByViewHandler, viewName, viewName));
               view = CreateDefaultViewScene(false, isManagedByViewHandler);
               break;

            default:
               ETG_TRACE_USR1_THR(("CGIAppViewFactoryBase::CreateDefaultViewScene type=Unknown, name=(%p)%s", viewName, viewName));
               break;
         }

         return view;
      }


      /************************************************************************/
      void CGIAppViewFactoryBase::InitViewSceneTypeMap()
      {
         ETG_TRACE_USR1_THR(("CGIAppViewFactoryBase::InitViewSceneTypeMap"));
         _viewSceneTypeMap.clear();

         Candera::DefaultAssetProvider& assetProvider = Candera::DefaultAssetProvider::GetInstance();
         const Candera::AssetDescriptor& assetDescriptor = assetProvider.GetAssetDescriptor();

         //iterate all 2D scenes from the asset
         for (Candera::AssetDescriptor::AssetIdIterator scene2DIt = assetDescriptor.GetAssetIdIterator(Candera::Scene2DLib); scene2DIt.IsValid(); ++scene2DIt)
         {
            const FeatStd::Char* viewName = assetProvider.GetNameById(Candera::Scene2DLib, *scene2DIt, 0);
            _viewSceneTypeMap[Courier::HashId(viewName)] = en2D;

            ETG_TRACE_USR4_THR(("CGIAppViewFactoryBase::InitViewSceneTypeMap found view scene type=2D, name=(%p)%s", viewName, viewName));
         }

         //iterate all 3D scenes from the asset
         for (Candera::AssetDescriptor::AssetIdIterator scene3DIt = assetDescriptor.GetAssetIdIterator(Candera::SceneLib); scene3DIt.IsValid(); ++scene3DIt)
         {
            const FeatStd::Char* viewName = assetProvider.GetNameById(Candera::SceneLib, *scene3DIt, 0);
            _viewSceneTypeMap[Courier::HashId(viewName)] = en3D;

            ETG_TRACE_USR4_THR(("CGIAppViewFactoryBase::InitViewSceneTypeMap found view scene type=3D, name=(%p)%s", viewName, viewName));
         }
      }


      /************************************************************************/
      CGIAppViewFactoryBase::enViewSceneType CGIAppViewFactoryBase::GetViewSceneType(const FeatStd::Char* viewName)
      {
         //search the view scene type
         Courier::HashId hashId(viewName);
         ViewSceneTypeMapType::const_iterator it = _viewSceneTypeMap.find(hashId);
         if (it != _viewSceneTypeMap.end())
         {
            return it->second;
         }

         //no info about the specified view scene
         ETG_TRACE_USR1_THR(("CGIAppViewFactoryBase::GetViewSceneType unknown type for view scene name='(%p)%s'", viewName, viewName));
         _viewSceneTypeMap[hashId] = enUndef;
         return enUndef;
      }


      /************************************************************************/
      const Courier::ViewId& CGIAppViewFactoryBase::ViewIdForProfileID(const Courier::ViewId& vid, unsigned int profileid) const
      {
         const SceneMapping* sceneMapping = SceneMappingForBaseName(vid, profileid, _viewCnt, _scenes);
         if (sceneMapping != 0)
         {
            const Courier::ViewId& id = sceneMapping->_nameID;
            return id;
         }
         else
         {
            sceneMapping = SceneMappingForName(vid, _viewCnt, _scenes);
            if (sceneMapping != 0)
            {
               const Courier::ViewId& id = sceneMapping->_baseviewnameID;
               return id;
            }
            else
            {
               return vid;
            }
         }
      }


      /* -----------------------------------------------------------------
          ViewIdForHash : Mapping the viewId to the HashId
          -----------------------------------------------------------------*/
      const Courier::ViewId& CGIAppViewFactoryBase::ViewIdForHash(FeatStd::UInt32 hash) const
      {
         static Courier::ViewId cInvalidId("InvalidView");
         const SceneMapping* sceneMapping = SceneMappingForHash(hash, _viewCnt, _scenes);
         if (sceneMapping != 0)
         {
            const Courier::ViewId& id = sceneMapping->_nameID;
            return id;
         }
         else
         {
            return cInvalidId;
         }
      }


      /* -----------------------------------------------------------------
          ViewForId :
          -----------------------------------------------------------------*/
      Courier::View* CGIAppViewFactoryBase::ViewForId(const Courier::ViewId& id) const
      {
         const SceneMapping* sceneMapping = SceneMappingForHash(id.GetHashId().HashValue(), _viewCnt, _scenes);
         if (sceneMapping == 0)
         {
            return 0;
         }
         return sceneMapping->_scene;
      }
   }
}