/* ***************************************************************************************
* FILE:          AppViewHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppViewHandler.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "sys_std_if.h"
#include "Courier/Version.h"
#include "CanderaWidget/String/String.h"
#include "AppViewHandler.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "Courier/Foundation/ComponentType.h"
#include "PopupConfig.h"
#include "hmibase/util/TraceUtils.h"
#include "hmibase/util/Ticker.h"
#include "Focus/FManager.h"
#include "CanderaWidget/Widget2D/Widget2D.h"
#include "Courier/Visualization/ViewVisitor.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "Candera/System/GlobalizationBase/CultureManager.h"
#include "View/CGI/CgiExtensions/HMIRenderViewVisitor.h"
#include "View/CGI/CgiExtensions/Renderer.h"
#include "View/CGI/InputHandling/SurfaceInputRegionManager.h"

#include "Gadget/SyncBlockConsumer.h"

#include "AppBase/AppConfigRegistry.h"

#if defined SESA_ARABIC_LAYOUT_FIX
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"
#else
#include "Candera/Engine2D/Layout/ArabicLayouterPatch.h"
#endif
#endif


#include "hmi_trace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AppViewHandler.cpp.trc.h"
#endif


static hmibase::view::RenderJobStrategy sRenderJobStrategy;

#if defined VARIANT_S_FTR_ENABLE_NICELEVEL_ADJUSTMENT

#include "RootDaemonClient/RootDaemon.h"
namespace hmibase {
namespace view {
static void SetNiceLevelCallback(const pid_t tid, const int nice)
{
   if (nice != 5) // due to security reasons only nice level of 5 is supported by root daemon client
   {
      ETG_TRACE_USR4_THR(("[%25s] request new thread nice level: tid %d, restriceted to nicel level of 5", hmibase::trace::getAppName().c_str(), tid));
   }
   else
   {
      ETG_TRACE_USR4_THR(("[%25s] request new thread nice level: tid %d, nice %d", hmibase::trace::getAppName().c_str(), tid, nice));
   }

   char buf[20];

   memset(buf, 0, sizeof(buf));
   snprintf(buf, sizeof(buf), "%d", tid);

   std::string cmdData = buf;
   HMIBASE_ROOTDAEMON_CALLER_PerformRootOp(E_HMIBASE_ROOTDAEMON_CMD_CODE_THREAD_NICENESS_5, cmdData);
}


}
}


#endif

AppViewHandler::AppViewHandler() :
   _isDirectApplicationActivationEnabled(true),
   _viewprofilehandler(0),
   _touchSession(0)
{
}


AppViewHandler::~AppViewHandler()
{
   _touchSession = 0;
   if (_viewprofilehandler)
   {
      delete _viewprofilehandler;
      _viewprofilehandler = 0;
   }
}


bool AppViewHandler::Init(Courier::ViewFactory* viewFactory, Candera::AssetConfig* assetConfiguration, Courier::Renderer* renderer, const Courier::Char* rootPath)
{
   if (viewFactory == 0)
   {
      ETG_TRACE_FATAL_THR(("ERROR AppViewHandler::Init 001 %s", rootPath));
      return false;
   }
   if (assetConfiguration == 0)
   {
      ETG_TRACE_FATAL_THR(("ERROR AppViewHandler::Init 002 %s", rootPath));
      return false;
   }
   if (renderer == 0)
   {
      ETG_TRACE_FATAL_THR(("ERROR AppViewHandler::Init 003 %s", rootPath));
      return false;
   }

   if (rootPath == 0)
   {
      ETG_TRACE_FATAL_THR(("ERROR AppViewHandler::Init 004 %s", rootPath));
      return false;
   }

   bool ret = Courier::ViewHandler::Init(viewFactory, assetConfiguration, renderer, rootPath);
   SetFinalizeRenderVisitor(this);
   hmibase::view::Renderer* hmibase_renderer = dynamic_cast<hmibase::view::Renderer*>(renderer);

   if (hmibase_renderer)
   {
      hmibase_renderer->SetRenderObserver(this);
   }

   renderer->SetRenderJobStrategy(&sRenderJobStrategy);

   _viewprofilehandler = new ::hmibase::view::ViewProfilehandler(this, &_HMIRenderViewVisitor);

   return ret;
}


void AppViewHandler::SetTouchSession(::hmibase::input::TouchSession* touchSession)
{
   _touchSession = touchSession;
   SetViewHandlerSession(touchSession);
}


void AppViewHandler::setBufferSwappingEnabled(bool bufferSwapEnabled)
{
   ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_FW, "setBufferSwappingEnabled[%25s] enabled %u", hmibase::trace::getAppName().c_str(), bufferSwapEnabled));

   bool earlySwap = !bufferSwapEnabled;
   if (earlySwap != sRenderJobStrategy.EarlySwap())
   {
      sRenderJobStrategy.SetEarlySwap(earlySwap);
   }
}


bool AppViewHandler::isBufferSwappingEnabled() const
{
   return sRenderJobStrategy.EarlySwap();
}


bool AppViewHandler::ActivateLayer(Courier::Int layerId, bool activate)
{
   ETG_TRACE_USR2_THR(("ActivateLayer[%25s], layer id %u, status %u", hmibase::trace::getAppName().c_str(), layerId, activate));
   return Base::ActivateLayer(layerId, activate);
}


bool AppViewHandler::onCourierMessage(const Courier::StartupMsg& msg)
{
   PARAM_UNUSED(msg);

   //  ActivationResMsg from Controller to View / ViewHandler redirected !!
   Courier::ActivationResMsg::Subscribe(static_cast<Courier::ComponentId>(Courier::ComponentType::View));

   // subscribe for surface state change message before controller component to give feedback
   //SurfaceStateChangedUpdMsg::Subscribe(Courier::ComponentType::View, Courier::ComponentType::Controller);

   // find the main surface and disable rendering at startup
   hmibase::view::Renderer* myRenderer = dynamic_cast<hmibase::view::Renderer*>(Base::GetRenderer());
   if (NULL != myRenderer)
   {
      const hmibase::view::Renderer::tLayerSurfaceMap& m = myRenderer->GetLayerSurfaceMap();

      for (hmibase::view::Renderer::tLayerSurfaceMap::const_iterator it = m.begin(); it != m.end(); ++it)
      {
         for (std::vector<unsigned int>::const_iterator it2 = it->second.begin(); it2 != it->second.end(); ++it2)
         {
            if (ScreenBrokerClient::GetInstance().IsMainSurface(*it2))
            {
               ActivateLayer(it->first, false);
            }
         }
      }
   }
   //registered message handlers also need it
   dispatchToMessageHandlers(msg);

   return false;//allow other handlers to also get this message
}


bool AppViewHandler::onCourierMessage(const PopupReqMsg& msg)
{
   if ((msg.GetCommand() == hmibase::popups::Show) || (msg.GetCommand() == hmibase::popups::ForceShow))
   {
      if (ExecuteViewAction(Courier::ViewAction::Create, msg.GetViewId(), true, true))
      {
         ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(msg.GetViewId().CStr()), 0, true, (msg.GetCommand() == hmibase::popups::ForceShow), msg.GetPopupDisplayID());
      }
   }
   else if (msg.GetCommand() == hmibase::popups::Hide)
   {
      ScreenBrokerClient::GetInstance().Hide(::hmibase::view::ViewIdentifier(msg.GetViewId().CStr()), 0, true, msg.GetPopupDisplayID());
   }
   else if (msg.GetCommand() == hmibase::popups::RestartTimer)
   {
      // Force Popup properties is not required for reste timer functionality
      ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(msg.GetViewId().CStr()), 0, true, false, msg.GetPopupDisplayID());
   }
   return true;
}


bool AppViewHandler::onCourierMessage(const ApplicationStateUpdMsg& msg)
{
   /************************************************************************/
   /* when application should come into foreground and this should be done */
   /* 'direct' so surface is already instanciated, rendering has to be     */
   /* enabled and content has to be rendered at least once                 */
   /************************************************************************/
   if ((msg.GetState() == hmibase::IN_FOREGROUND) || (msg.GetState() == hmibase::FORCE_IN_FOREGROUND))
   {
      // enable/disable rendering for visible/invisible layers/surfaces
      hmibase::view::Renderer* myRenderer = dynamic_cast<hmibase::view::Renderer*>(Base::GetRenderer());
      if (NULL != myRenderer)
      {
         const hmibase::view::Renderer::tLayerSurfaceMap& m = myRenderer->GetLayerSurfaceMap();

         for (hmibase::view::Renderer::tLayerSurfaceMap::const_iterator it = m.begin(); it != m.end(); ++it)
         {
            for (std::vector<unsigned int>::const_iterator it2 = it->second.begin(); it2 != it->second.end(); ++it2)
            {
               if (ScreenBrokerClient::GetInstance().IsMainSurface(*it2))
               {
                  if (ScreenBrokerClient::GetInstance().CheckSurfaceIdDisplayIdMapping(*it2, msg.GetDisplayID()))
                  {
                     // enable rendering on this layer
                     //POST_MSG(COURIER_MESSAGE_NEW(Courier::LayerReqMsg)(it->first, true))
                     SYSTEMD_LOG("GUISTARTUP(n): hmibase::IN_FOREGROUND/BACHGROUND called\n");
                     ActivateLayer(it->first, true);
                     //invalidate to trigger rende, if any view is available at this time to wait for postrender callback
                     InvalidateAll(false);
                     ScreenBrokerSHowRequestState RequestState;
                     // onPostRender has to be call three times from now on (triple buffering),
                     // before we can guarantee, that the view is available in framebuffer
                     RequestState = EN_ON_RENDER_CALLBACK;
                     AddtoAppStateDisplayList(msg.GetDisplayID(), *it2, RequestState);
                  }
                  else
                  {
                     ETG_TRACE_ERR_CLS_THR((TR_CLASS_HMI_FW, "ApplicationStateUpdMsg - Requested DisplayID [%u] for surfaceID [%u] and its registered DisplayID[%u] Failed !! ", msg.GetDisplayID(), *it2, ScreenBrokerClient::GetInstance().GetDisplayID(*it2)));
                  }
               }
            }
         }
      }
   }

   adjustNiceLevelOnApplicationStatus(msg.GetState());

   return true;
}


void AppViewHandler::adjustNiceLevelOnApplicationStatus(hmibase::hmiappstates applicationState)
{
#if defined VARIANT_S_FTR_ENABLE_NICELEVEL_ADJUSTMENT
   tenHmiBaseRootDaemonCmdCode cmd;
   switch (applicationState)
   {
      case hmibase::IN_FOREGROUND:
      case hmibase::FORCE_IN_FOREGROUND:
      {
         cmd = E_HMIBASE_ROOTDAEMON_CMD_CODE_NICE_LVL_FG;
      }
      break;
      case hmibase::IN_BACKGROUND:
      {
         cmd = E_HMIBASE_ROOTDAEMON_CMD_CODE_NICE_LVL_BG;
      }
      break;
      default:
         return;
   }

   std::stringstream tmp;
   tmp << getpid();
   std::string cmdData = tmp.str();

   HMIBASE_ROOTDAEMON_CALLER_PerformRootOp(cmd, cmdData);

   // modify AsyncLoaderWorker niceness
   FeatStd::AsyncRequestDispatcherWorkerThread* asyncRequestDispatcherWorkerThread = Candera::DefaultAssetProvider::GetInstance().AsyncProxy().GetAssetProviderDispatcherWorkerThread();

   if (asyncRequestDispatcherWorkerThread)
   {
      // set nice level for AsyncAssetLoaderWorker to 5 in any case
      asyncRequestDispatcherWorkerThread->SetNiceLevel(5, hmibase::view::SetNiceLevelCallback);
   }
#else
   PARAM_UNUSED(applicationState);
#endif
}


void AppViewHandler::AddtoAppStateDisplayList(const Courier::UInt32 displayid, const Courier::UInt32 surfaceId, const ScreenBrokerSHowRequestState RequestState)
{
   if (_appstateReqlist.empty())
   {
      // register myself for OnPostSwapBuffer callback
      hmibase::view::RenderJobStrategy::RegisterListener(this);
   }

   //push to the back
   AppstateDisplayReq appstatedisplaydata = { displayid, surfaceId, RequestState };
   _appstateReqlist.push_back(appstatedisplaydata);
   ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_FW, "AddtoAppStateDisplayList for DisplayID [%u]", displayid));
}


void AppViewHandler::RemoveFromAppStateDisplayList()
{
   if (_appstateReqlist.size() > 0)
   {
      //remove the first entry
      _appstateReqlist.erase(_appstateReqlist.begin());
   }

   if (_appstateReqlist.empty())
   {
      // no more application state changes expected, so deregister listener
      hmibase::view::RenderJobStrategy::DeregisterListener(this);
   }
}


/************************************************************************/
/* check if a surface holds already an active scene                     */
/************************************************************************/
bool AppViewHandler::isViewOnSurface(unsigned int surfaceId) const
{
   const HMIRenderViewVisitor::tViewInfoVector& viewInfo = _HMIRenderViewVisitor.getActiveViewInfo();

   for (HMIRenderViewVisitor::tViewInfoVector::const_iterator cit = viewInfo.begin(); cit != viewInfo.end(); ++cit)
   {
      std::vector<unsigned int>::const_iterator cit2 = std::find(cit->surfaceIds.begin(), cit->surfaceIds.end(), surfaceId);
      if (cit2 != cit->surfaceIds.end())
      {
         return true;
      }
   }

   return false;
}


bool AppViewHandler::onCourierMessage(const SurfaceStateChangedUpdMsg& msg)
{
   // check if main surface has changed to unmapped and send  information about application in background
   if ((msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_INVISIBLE))
         && (ScreenBrokerClient::GetInstance().IsMainSurface(msg.GetSurfaceId())))
   {
      COURIER_MESSAGE_NEW(ApplicationStateUpdMsg)(hmibase::IN_BACKGROUND, msg.GetDisplayId())->Post();
   }

   //registered message handlers also need it
   dispatchToMessageHandlers(msg);

   // disable rendering for invisible layers/surfaces
   hmibase::view::Renderer* myRenderer = dynamic_cast<hmibase::view::Renderer*>(Base::GetRenderer());
   if (NULL != myRenderer)
   {
      const hmibase::view::Renderer::tLayerSurfaceMap& m = myRenderer->GetLayerSurfaceMap();

      for (hmibase::view::Renderer::tLayerSurfaceMap::const_iterator it = m.begin(); it != m.end(); ++it)
      {
         for (std::vector<unsigned int>::const_iterator it2 = it->second.begin(); it2 != it->second.end(); ++it2)
         {
            if ((*it2 == msg.GetSurfaceId()) && (ScreenBrokerClient::GetInstance().IsMainSurface(msg.GetSurfaceId())))
            {
               if ((msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_VISIBLE))
                     || (msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_REGISTERED))
                     || (msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_ANIM_STARTED))
                     || (msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_ANIM_STOPPED)))
               {
                  // do nothing
                  continue;
               }
               //POST_MSG(COURIER_MESSAGE_NEW(Courier::LayerReqMsg)(it->first, false))
               ActivateLayer(it->first, false);
            }
         }
      }
   }

   if ((msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_INVISIBLE))
         || (msg.GetState() == static_cast<int>(hmibase::SURFACESTATE_QUEUED)))
   {
      hmibase::input::TouchAbort touchAbort(msg.GetSurfaceId());
      Courier::ViewHandler::OnMessage(touchAbort);
   }

   if (!_appstateReqlist.empty())
   {
      /*
       * There can be 2 possible pending requests, one waiting for Surface availability and the other waiting after completion of Swap buffer
       * */
      if (((_appstateReqlist.front().SBShowRequestState == EN_ON_SURFACE_AVAILABILITY) && (_appstateReqlist.front().surfaceID == msg.GetSurfaceId()))\
            || (_appstateReqlist.front().SBShowRequestState == EN_ON_UNBLOCKED))
      {
         if (ScreenBrokerClient::GetInstance().IsMainSurface(msg.GetSurfaceId()))
         {
            ProcessPendingSBShowRequests();
         }
      }
   }

   // do not consume message by intend so that other components can do their stuff also
   return false;
}


bool AppViewHandler::onCourierMessage(const Courier::ActivationResMsg& msg)
{
   //registered message handlers also need it
   dispatchToMessageHandlers(msg);

   COURIER_MESSAGE_NEW(Courier::RenderReqMsg)()->Post();
   return false;//allow other handlers to also get this message
}


bool AppViewHandler::onCourierMessage(const Courier::ViewRenderingResMsg& msg)
{
   //registered message handlers also need it
   dispatchToMessageHandlers(msg);
   COURIER_MESSAGE_NEW(Courier::RenderReqMsg)()->Post();
   return false;//allow other handlers to also get this message
}


bool AppViewHandler::onCourierMessage(const SbAnimationStateChangedUpdMsg& msg)
{
   ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_FW_SBCLIENT, "AppViewHandler::onCourierMessage received SbAnimationStateChangedUpdMsg in app %20s.",
                           hmibase::trace::getAppName().c_str()));

   if (msg.GetAnimStatus())
   {
      Courier::ActivationReqMsg* activationMsg = COURIER_MESSAGE_NEW(Courier::ActivationReqMsg)(msg.GetViewId(), false, true);

      if (activationMsg)
      {
         activationMsg->Post();

         ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_FW_SBCLIENT, "AppViewHandler::onCourierMessage Send ViewActivationReq with activate=false for view: '%s'.",
                                 msg.GetViewId().CStr()));
      }
      else
      {
         ETG_TRACE_ERR_CLS_THR((TR_CLASS_HMI_FW_SBCLIENT, "AppViewHandler::onCourierMessage Send ViewActivationReq with activate=false for view '%s' failed!",
                                msg.GetViewId().CStr()));
      }
   }
   else
   {
      Courier::ActivationReqMsg* activationMsg = COURIER_MESSAGE_NEW(Courier::ActivationReqMsg)(msg.GetViewId(), true, true);

      if (activationMsg)
      {
         activationMsg->Post();

         ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_FW_SBCLIENT, "AppViewHandler::onCourierMessage Send ViewActivationReq with activate=true for view: '%s'.",
                                 msg.GetViewId().CStr()));
      }
      else
      {
         ETG_TRACE_ERR_CLS_THR((TR_CLASS_HMI_FW_SBCLIENT, "AppViewHandler::onCourierMessage Send ViewActivationReq with activate=true for view '%s' failed!",
                                msg.GetViewId().CStr()));
      }
   }

   return true;
}


/**
 * Windows only to simulate timer controled popup closing
 * on target ScreenBroker will do the work
 */
bool AppViewHandler::onCourierMessage(const SBPopupTimerExpiryUpdMsg& msg)
{
#ifdef WIN32
   ScreenBrokerClient::GetInstance().Hide(::hmibase::view::ViewIdentifier(msg.GetViewId().CStr()), msg.GetSurfaceId(), true);
#else
   PARAM_UNUSED(msg);
#endif
   return true;
}


void AppViewHandler::registerMessageHandler(::hmibase::view::MessageHandler& handler)
{
   _messageHandlers.push_back(&handler);
}


void AppViewHandler::unregisterMessageHandler(::hmibase::view::MessageHandler& handler)
{
   MessageHandlersType::iterator it = std::find(_messageHandlers.begin(), _messageHandlers.end(), &handler);
   if (it != _messageHandlers.end())
   {
      _messageHandlers.erase(it);
   }
}


void AppViewHandler::RegisterViewVisitor(Courier::ViewVisitor& visitor)
{
   _viewVisitors.insert(&visitor);
}


void AppViewHandler::DeregisterViewVisitor(Courier::ViewVisitor& visitor)
{
   ViewVisitorsType::iterator it = std::find(_viewVisitors.begin(), _viewVisitors.end(), &visitor);
   if (it != _viewVisitors.end())
   {
      _viewVisitors.erase(it);
   }
}


bool AppViewHandler::dispatchToMessageHandlers(const Courier::Message& msg)
{
   for (MessageHandlersType::iterator it = _messageHandlers.begin(); it != _messageHandlers.end(); ++it)
   {
      ::hmibase::view::MessageHandler* handler = *it;
      if ((handler != NULL) && handler->onMessage(msg))
      {
         return true;
      }
   }
   return false;
}


class WidgetFinderCallback : public WidgetCheckCallback
{
   public:
      explicit WidgetFinderCallback(const Courier::Identifier& widgetId) : _widgetId(widgetId), _widget(NULL) { }

      ~WidgetFinderCallback()
      {
         _widget = NULL;
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         if ((widget != NULL) && (_widgetId == Courier::Identifier(widget->GetLegacyName())))
         {
            _widget = widget;
            return true;
         }
         return false;
      }

      Candera::WidgetBase* GetWidget() const
      {
         return _widget;
      }

   private:
      Courier::Identifier _widgetId;
      Candera::WidgetBase* _widget;
};


Courier::FrameworkWidget* AppViewHandler::FindWidget(const Courier::ViewId& viewId, const Courier::CompositePath& compositePath, const Courier::ItemId& widgetId)
{
   Courier::FrameworkWidget* widget = Base::FindWidget(viewId, compositePath, widgetId);

   if (widget == NULL)
   {
      Courier::View* view = FindView(viewId);
      if (view != NULL)
      {
         WidgetFinderCallback callback(Courier::Identifier(widgetId.CStr()));
         if (view->OnMessage(WidgetCheckReqMsg(&callback)))
         {
            widget = callback.GetWidget();
         }
      }
   }
   return widget;
}


Candera::WidgetBase* AppViewHandler::FindWidget(const Courier::ViewId& viewId, const Courier::Identifier& widgetId)
{
   Candera::WidgetBase* widget = NULL;

   Courier::View* view = FindView(viewId);
   if (view != NULL)
   {
      WidgetFinderCallback callback(widgetId);
      if (view->OnMessage(WidgetCheckReqMsg(&callback)))
      {
         widget = callback.GetWidget();
      }
   }
   return widget;
}


bool AppViewHandler::OnMessageViewTreeOnly(const Courier::Message& msg)
{
   return Base::OnMessageViewTreeOnly(msg);
}


bool AppViewHandler::LoseFocus(Courier::FrameworkWidget* widget)
{
   if ((widget != NULL) && (widget->GetParentView() != NULL) && (_currentFocusView == widget->GetParentView()->GetId().CStr()))
   {
      hmibase::widget::Widget* widgetBase = static_cast<hmibase::widget::Widget*>(widget);
      const FeatStd::Char* name = widgetBase->GetLegacyName();
      if (_currentFocusWidget == name)
      {
         widgetBase->OnLostFocus();
         _currentFocusView = "";
         _currentFocusWidget = "";
         return true;
      }
   }
   return false;
}


bool AppViewHandler::SetFocus(const Courier::ViewId& viewId,


                              const Courier::CompositePath& compositePath,

                              const Courier::ItemId& widgetId)
{
   bool rc = true;

   if ((_currentFocusView != viewId.CStr()) || (_currentFocusWidget != widgetId.CStr()))
   {
      if (!_currentFocusView.IsEmpty() && !_currentFocusWidget.IsEmpty())
      {
         Courier::FrameworkWidget* widget = FindWidget(Courier::ViewId(_currentFocusView.GetCString()),

                                            compositePath,

                                            Courier::ItemId(_currentFocusWidget.GetCString()));
         if (widget != NULL)
         {
            //const Courier::LostFocusEvent lostFocusEvent;
            //widget->DispatchEvent(lostFocusEvent);
            widget->OnLostFocus();

            Courier::Message* postMsg = (COURIER_MESSAGE_NEW(Courier::LostFocusIndMsg)(Courier::ViewId(_currentFocusView.GetCString()),

                                         Courier::CompositePath(),

                                         Courier::ItemId(_currentFocusWidget.GetCString())));
            rc = (postMsg != 0) && postMsg->Post();
         }
      }

      _currentFocusView = viewId.CStr();
      _currentFocusWidget = widgetId.CStr();
      if (!_currentFocusView.IsEmpty() && !_currentFocusWidget.IsEmpty())
      {
         Candera::WidgetBase* widget = static_cast<Candera::WidgetBase*>(FindWidget(viewId,

                                       Courier::CompositePath(),

                                       widgetId));
         if (widget != NULL)
         {
            //const Courier::FocusEvent focusEvent;
            //widget->DispatchEvent(focusEvent);
            widget->OnFocus();
         }
      }
   }

   return rc;
}


// Localization stuff

class InvalidateLayoutViewVisitor : public Courier::ViewVisitor
{
   public:
      InvalidateLayoutViewVisitor() {}
      virtual ~InvalidateLayoutViewVisitor() {}
      virtual bool OnBegin()
      {
         return true;
      }
      virtual bool Visit(Courier::ViewContainer*)
      {
         return true;
      }
      virtual void OnEnd() {}
      virtual void Visit(Courier::ViewScene3D*)
      {
         // layouting not supported in 3D
      }
      virtual void Visit(Courier::ViewScene2D* view)
      {
         if (0 != view && 0 != view->GetScene2DContext() && 0 != view->GetScene2DContext()->GetScene())
         {
            view->GetScene2DContext()->GetScene()->SetLayoutInvalid();
         }
      }
};


bool AppViewHandler::onCourierMessage(const Courier::SetCultureResMsg& oMsg)
{
   Candera::Globalization::Culture::SharedPointer cultureCurrent = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();

   // InvalidateLayout, touch all scenes
   InvalidateLayoutViewVisitor invalidateLayoutViewVisitor;
   Accept(&invalidateLayoutViewVisitor);
   if (!cultureCurrent.PointsToNull())
   {
      if (oMsg.GetSuccess())
      {
         ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_FW, "Language / Culture change was successful: '%s'.",
                                 cultureCurrent->GetLocale()));
      }
      else
      {
         ETG_TRACE_ERR_CLS_THR((TR_CLASS_HMI_FW, "Language / Culture  change failed '%s'",
                                cultureCurrent->GetLocale()));
      }
      Courier::ViewHandler::SetCurrentCulture(cultureCurrent->GetLocale());
   }

   // repaint all
   POST_MSG((COURIER_MESSAGE_NEW(Courier::RenderReqMsg)(true)));
   return true;
}


bool AppViewHandler::onCourierMessage(const LocalizationReqMsg& msg)
{
   // Courier::SetCultureResMsg::Subscribe(Courier::ComponentType::Model);
   Courier::SetCultureResMsg::Subscribe(static_cast<Courier::ComponentId>(Courier::ComponentType::View));

   static char sCulture[10];

   memset(sCulture, 0, sizeof(sCulture));
   strncpy(sCulture, msg.GetLocale().GetCString(), sizeof(sCulture) - 1); /* ar_SA, en_GB, de_DE, ... */

   int cultureCnt = Candera::Globalization::CultureManager::GetInstance().GetCultureCount();
   if (cultureCnt > 0)
   {
      Candera::Globalization::Culture::SharedPointer cultureCurrent = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture();
      Candera::Globalization::Culture::SharedPointer culture = Candera::Globalization::CultureManager::GetInstance().GetCulture(sCulture);
      if (!culture.PointsToNull())
      {
         if (std::string(sCulture) != cultureCurrent->GetLocale())
         {
            Courier::Message* pMsg = (COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(sCulture)));
            if (pMsg)
            {
               pMsg->Post();
            }
            //bool check = culture->GetTextDirection() == Candera::Globalization::RightToLeft;
            //Candera::Globalization::CultureManager::GetInstance().SetCurrentCulture(culture->GetLocale());
         }
         else
         {
            ETG_TRACE_USR1_THR(("OnLocalizationReqMsg[%25s]: Culture already set %s", hmibase::trace::getAppName().c_str(), sCulture));
         }
      }
      else
      {
         ETG_TRACE_ERRMEM_THR(("OnLocalizationReqMsg[%25s]: Culture not found in asset: %s", hmibase::trace::getAppName().c_str(), sCulture));
      }
   }
   else
   {
      ETG_TRACE_USR1_THR(("OnLocalizationReqMsg[%25s]: No cultures available", hmibase::trace::getAppName().c_str()));
   }

   return true;
}


bool AppViewHandler::SendSBShowRequest(AppstateDisplayReq& appStateReq)
{
   bool retVal = false;
   uint32_t surfaceId = appStateReq.surfaceID;

   if (ScreenBrokerClient::GetInstance().IsMainSurface(surfaceId) &&
         (ScreenBrokerClient::GetInstance().CheckSurfaceIdDisplayIdMapping(surfaceId, appStateReq.displayID)))
   {
      if (ScreenBrokerClient::GetInstance().GetMainSurfaceState(surfaceId) != ScreenBroker::SurfaceState::Deregistered)
      {
         ScreenBrokerClient::GetInstance().Show(::hmibase::view::ViewIdentifier(), surfaceId);
         RemoveFromAppStateDisplayList();
         retVal = true;

         static bool traceDisplayEventTimestamp = true;
         if (traceDisplayEventTimestamp && retVal)
         {
            // trace timestamp of the first time a display event is received
            ETG_TRACE_USR1_CLS_THR((TR_CLASS_HMI_PERFORMANCE_MP, "[%20s] first time on display: %u ms", hmibase::trace::getAppName().c_str(), hmibase::util::Ticker::getTickCountMsec()));
            traceDisplayEventTimestamp = false;
         }
      }
      else
      {
         ETG_TRACE_USR1_THR(("SBShowRequest not executed for surface %d as main surface is not yet registered", surfaceId));
         appStateReq.SBShowRequestState = EN_ON_SURFACE_AVAILABILITY;
      }
   }

   return retVal;
}


void AppViewHandler::OnPostSwapBuffer(Courier::Gdu& gdu)
{
   if ((!_appstateReqlist.empty()) && (_appstateReqlist.front().SBShowRequestState == EN_ON_RENDER_CALLBACK))
   {
      uint32_t surfaceId = 0;
      if (GetSurface(gdu, surfaceId))
      {
         if (_appstateReqlist.front().surfaceID == surfaceId)
         {
            SendSBShowRequest(_appstateReqlist.front());
         }
         else
         {
            for (AppstatDisplayReqType::iterator it = _appstateReqlist.begin(); it != _appstateReqlist.end(); ++it)
            {
               if ((*it).surfaceID == surfaceId)
               {
                  // set state to ON_UNBLOCKED and execute later when becoming first in the list
                  (*it).SBShowRequestState = EN_ON_UNBLOCKED;
                  break;
               }
            }
         }
         //_appstateReqlist.front().SBShowRequestState = EN_ON_RENDER_CALLBACK;
      }
      else
      {
         //do nothing
      }
   }
}


bool AppViewHandler::onCourierMessage(const EglWaitNativeReqMsg& msg)
{
   sRenderJobStrategy.WaitForNativeAfterSwap(msg.GetSurfaceId(), msg.GetCount());
   return true;
}


bool AppViewHandler::onCourierMessage(const hmibase::gadget::PositionInfoReqMsg& msg)
{
   const unsigned int* id = hmibase::app::base::AppConfigValRegistry::getItem(hmibase::app::base::ConfigItems::SYNCBLOCK_ID);

   if (id == NULL)
   {
      // not a consumer process, so there is no gadget position info available at all
      return true;
   }

   bool consumed = OnMessageViewTreeOnly(msg);
   if (!consumed)
   {
      if (msg.GetIPCForwarding() == false)
      {
         Courier::Message* msg1 = COURIER_MESSAGE_NEW(hmibase::gadget::PositionInfoResMsg)(msg.GetInstanceId(), true, Candera::Vector2(-FLT_MAX, -FLT_MAX));
         if (msg1)
         {
            msg1->Post();
            consumed = true;
         }
      }
   }

   return consumed;
}


// deprecated
bool AppViewHandler::onCourierMessage(const ResetGadgetCacheReqMsg& /*msg*/)
{
   return true;
}


bool AppViewHandler::onCourierMessage(const hmibase::gadget::ConsumerMsg& msg)
{
   bool consumed = OnMessageViewTreeOnly(msg);

   if (!consumed)
   {
      // update on syncblock not handled by widgets, so exchange buffer from here to avoid frame drops
      for (size_t i = 0; i < msg.GetInstIds().size(); ++i)
      {
         int instanceId = msg.GetInstIds()[i];
         int frameBuffer = hmibase::gadget::SyncBlockConsumer::getInstance().exchangeFramebuffer(instanceId);
         ETG_TRACE_USR4_THR(("exchange not consumed sync block update for id %d, new fb %d", instanceId, frameBuffer));
      }
   }
   return true;
}


bool AppViewHandler::onCourierMessage(const CreateNewDestroyOld& msg)
{
   if (msg.GetDestinationView() == msg.GetSourceView())
   {
      ETG_TRACE_USR1_THR(("CreateNewDestroyOld carries the same view to create and destroy, '%s'", msg.GetSourceView().CStr()));
      return true;
   }

   Courier::ViewFacade viewFacade;

   viewFacade.Init(this);

   Courier::ViewReqMsg viewReqMsg(Courier::ViewAction::Create, msg.GetDestinationView(), true, true);
   viewFacade.OnViewComponentMessage(viewReqMsg);
   Courier::ActivationReqMsg activationReqMsg(msg.GetDestinationView(), true, true);
   viewFacade.OnViewComponentMessage(activationReqMsg);
   Courier::ViewReqMsg viewReqMsg2(Courier::ViewAction::Destroy, msg.GetSourceView(), false, false);
   viewFacade.OnViewComponentMessage(viewReqMsg2);
   return true;
}


void AppViewHandler::setArabicLayoutPatch(bool flag, bool invalidateViews)
{
#if defined SESA_ARABIC_LAYOUT_FIX
   Candera::ArabicLayouterPatch::SetDefaultEnabled(flag);
#endif

   if (invalidateViews)
   {
      InvalidateLayoutViewVisitor invalidateLayoutViewVisitor;
      Accept(&invalidateLayoutViewVisitor);

      InvalidateAll(true);
   }
}


void AppViewHandler::onPreRender()
{
   hmibase::input::SurfaceInputRegionManager::getInstance().sendInputRegions();
}


void AppViewHandler::onPostRender(bool continousExecutionEnabled)
{
   PARAM_UNUSED(continousExecutionEnabled);

   ETG_TRACE_USR4_CLS_THR((TR_CLASS_HMI_PERFORMANCE_MP, "[%20s] Performance FPS: %d ", hmibase::trace::getAppName().c_str(), GetFramesPerSecond2D())); // GetFramesPerSecond() same as 2D
}


bool AppViewHandler::OnBegin()
{
   _HMIRenderViewVisitor.OnBegin();

   ViewVisitorsType::iterator it = _viewVisitors.begin();
   while (it != _viewVisitors.end())
   {
      (*it)->OnBegin();
      ++it;
   }
   return true;
}


bool AppViewHandler::Visit(Courier::ViewContainer* viewContainer)
{
   _HMIRenderViewVisitor.Visit(viewContainer);

   ViewVisitorsType::iterator it = _viewVisitors.begin();
   while (it != _viewVisitors.end())
   {
      (*it)->Visit(viewContainer);
      ++it;
   }
   return true;
}


void AppViewHandler::Visit(Courier::ViewScene2D* view)
{
   _HMIRenderViewVisitor.Visit(view);

   ViewVisitorsType::iterator it = _viewVisitors.begin();
   while (it != _viewVisitors.end())
   {
      (*it)->Visit(view);
      ++it;
   }
}


void AppViewHandler::Visit(Courier::ViewScene3D* view)
{
   _HMIRenderViewVisitor.Visit(view);

   ViewVisitorsType::iterator it = _viewVisitors.begin();
   while (it != _viewVisitors.end())
   {
      (*it)->Visit(view);
      ++it;
   }
}


void AppViewHandler::OnEnd()
{
   _HMIRenderViewVisitor.OnEnd();

   ViewVisitorsType::iterator it = _viewVisitors.begin();
   while (it != _viewVisitors.end())
   {
      (*it)->OnEnd();
      ++it;
   }
}


bool AppViewHandler::GetSurface(Courier::Gdu& gdu, uint32_t& surfaceId)
{
   // get surface id from gdu
   Candera::GraphicDeviceUnit* canderaGdu = gdu.GetGdu();
   if (0 == canderaGdu)
   {
      return false;
   }
   const Candera::MetaInfo::GraphicDeviceUnitMetaInfo* metaInfo = Candera::DevicePackageDescriptor::GetMetaInformation(canderaGdu->GetUnitType());
   if (0 == metaInfo)
   {
      return false;
   }
   Candera::MetaInfo::GraphicDeviceUnitPropertyMetaInfo* propMetaInfo = metaInfo->LookupItem("SurfaceId");
   if (0 == propMetaInfo)
   {
      return false;
   }

   char buffer[32];
   buffer[0] = '\0';
   if (!propMetaInfo->Get(canderaGdu, buffer, 32))
   {
      return false;
   }
   ::sscanf(buffer, "%u", &surfaceId);

   return true;
}


void AppViewHandler::ProcessPendingSBShowRequests()
{
   bool success = true;
   ETG_TRACE_USR1_THR(("ProcessPendingSBShowRequests start: %d pending requests", _appstateReqlist.size()));
   while ((_appstateReqlist.size() > 0) && success)
   {
      // try to send first request in the queue
      success = SendSBShowRequest(_appstateReqlist.front());

      if (success)
      {
         // check if the first request was blocked before
         if (!_appstateReqlist.empty() && (_appstateReqlist.front().SBShowRequestState == EN_ON_UNBLOCKED))
         {
            // reset state and go on
            _appstateReqlist.front().SBShowRequestState = EN_ON_RENDER_CALLBACK;
         }
      }
   }
   ETG_TRACE_USR1_THR(("ProcessPendingSBShowRequests end: %d pending requests", _appstateReqlist.size()));
}
