/* ***************************************************************************************
* FILE:          AnimationWrapper.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationWrapper.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef __ANIMATION_WRAPPER_H__
#define __ANIMATION_WRAPPER_H__

class AppViewHandler;

namespace hmibase {
namespace view {


class AnimationPropertySetterUtils
{
   public:
      ///////////////////////////////////////////////////////////////////////////////
      template <typename TPropertySetterType>
      static Candera::Animation::AnimationPropertySetter::SharedPointer createNodePropertySetter(Candera::Node2D* node)
      {
         typename TPropertySetterType::SharedPointer propertySetter = TPropertySetterType::Create();
         if (!propertySetter.PointsToNull())
         {
            configureNodePropertySetter(*propertySetter, node);
         }
         return propertySetter;
      }

      ///////////////////////////////////////////////////////////////////////////////
      static void configureNodePropertySetter(Candera::Animation::BaseTransformable2DPropertySetter& propertySetter, Candera::Node2D* node);
      static void configureNodePropertySetter(Candera::Animation::BaseNode2DPropertySetter& propertySetter, Candera::Node2D* node);
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
      static void configureNodePropertySetter(Candera::Animation::AbstractNodePropertySetter& propertySetter, Candera::Node2D* node);
#endif
};


class AnimatedPropertyWrapper
{
   public:
      ///////////////////////////////////////////////////////////////////////////////
      AnimatedPropertyWrapper();
      virtual ~AnimatedPropertyWrapper();

      ///////////////////////////////////////////////////////////////////////////////
      bool init(FeatStd::SizeType keyFrameCount, FeatStd::SizeType componentCount,
                const Candera::Animation::InterpolationStrategy::SharedPointer& interpolationStrategy);
      void release();

      ///////////////////////////////////////////////////////////////////////////////
      bool addToPlayer(Candera::Animation::AnimationPlayer::SharedPointer player);
      bool removeFromPlayer(Candera::Animation::AnimationPlayer::SharedPointer player);

      ///////////////////////////////////////////////////////////////////////////////
      void setPropertySetter(const Candera::Animation::AnimationPropertySetter::SharedPointer& propertySetter);
      Candera::Animation::AnimationPropertySetter::SharedPointer getPropertySetter() const;

      ///////////////////////////////////////////////////////////////////////////////
      void setSequenceTime(FeatStd::SizeType kfIndex, Candera::Animation::SequenceTimeType seqTime);
      Candera::Animation::SequenceTimeType getSequenceTime(FeatStd::SizeType kfIndex) const;
      Candera::Animation::SequenceTimeType getDuration() const;

      ///////////////////////////////////////////////////////////////////////////////
      void setKeyframeValue(FeatStd::SizeType kfIndex, FeatStd::SizeType componentIndex, FeatStd::Float value);
      Candera::Float getKeyframeValue(FeatStd::SizeType kfIndex, FeatStd::SizeType componentIndex) const;

      ///////////////////////////////////////////////////////////////////////////////
      FeatStd::SizeType getKeyframeCount() const
      {
         return _keyFrameCount;
      }

      ///////////////////////////////////////////////////////////////////////////////
      FeatStd::SizeType getComponentCount() const
      {
         return _componentCount;
      }

      ///////////////////////////////////////////////////////////////////////////////
      Candera::Animation::AnimationBlendedProperty::SharedPointer getProperty() const
      {
         return _property;
      }

   private:
      AnimatedPropertyWrapper(const AnimatedPropertyWrapper&);
      AnimatedPropertyWrapper& operator=(const AnimatedPropertyWrapper&);

      FeatStd::SizeType _keyFrameCount;
      FeatStd::SizeType _componentCount;
      Candera::Animation::SequenceTimeType* _sequenceTimes;
      FeatStd::Float* _keyframeValues;
      Candera::Animation::AnimationKeyframeSequence::SharedPointer _keyframeSequence;
      Candera::Animation::AnimationBlendedProperty::SharedPointer _property;
};


class AnimationWrapper
{
   public:
      ///////////////////////////////////////////////////////////////////////////////
      AnimationWrapper(AppViewHandler& viewHandler);
      virtual ~AnimationWrapper();

      ///////////////////////////////////////////////////////////////////////////////
      bool init();
      void release();

      ///////////////////////////////////////////////////////////////////////////////
      void applyProperties(const Courier::AnimationProperties& properties);
      bool start(const Courier::AnimationProperties& properties = Courier::AnimationProperties(), bool resetBeforeApplyingProperties = true);
      void stop();
      void reset();
      bool isEnabled() const;

      ///////////////////////////////////////////////////////////////////////////////
      bool addListener(Candera::Animation::AnimationPlayerListener* listener);
      bool removeListener(Candera::Animation::AnimationPlayerListener* listener);

      ///////////////////////////////////////////////////////////////////////////////
      Candera::Animation::SequenceTimeType getMaximumDuration() const;

      ///////////////////////////////////////////////////////////////////////////////
      void setPlayer(Candera::Animation::AnimationPlayer::SharedPointer player);
      Candera::Animation::AnimationPlayer::SharedPointer getPlayer() const
      {
         return _player;
      }

   private:
      AnimationWrapper(const AnimationWrapper&);
      AnimationWrapper& operator=(const AnimationWrapper&);

      Candera::Animation::AnimationPlayer::SharedPointer _player;
      AppViewHandler& _viewHandler;
};


}
}


#endif
