/* ***************************************************************************************
* FILE:          AnimationWrapper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationWrapper.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"

#include "View/CGI/CgiExtensions/AppViewHandler.h"
#include "CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactory.h"

#include "AnimationWrapper.h"

#include "hmi_trace_if.h"
#include "hmibase/util/TraceUtils.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationWrapper.cpp.trc.h"
#endif

namespace hmibase {
namespace view {
///////////////////////////////////////////////////////////////////////////////
void AnimationPropertySetterUtils::configureNodePropertySetter(Candera::Animation::BaseTransformable2DPropertySetter& propertySetter, Candera::Node2D* node)
{
   propertySetter.SetTransformable(node);
}


///////////////////////////////////////////////////////////////////////////////
void AnimationPropertySetterUtils::configureNodePropertySetter(Candera::Animation::BaseNode2DPropertySetter& propertySetter, Candera::Node2D* node)
{
   propertySetter.SetNode(node);
}


#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
void AnimationPropertySetterUtils::configureNodePropertySetter(Candera::Animation::AbstractNodePropertySetter& propertySetter, Candera::Node2D* node)
{
   propertySetter.SetNode(node);
}


#endif

///////////////////////////////////////////////////////////////////////////////
AnimatedPropertyWrapper::AnimatedPropertyWrapper() : _keyFrameCount(0), _componentCount(0), _sequenceTimes(NULL), _keyframeValues(NULL)
{
}


///////////////////////////////////////////////////////////////////////////////
AnimatedPropertyWrapper::~AnimatedPropertyWrapper()
{
   _sequenceTimes = NULL;
   _keyframeValues = NULL;
}


///////////////////////////////////////////////////////////////////////////////
void AnimatedPropertyWrapper::release()
{
   _keyFrameCount = 0;
   _componentCount = 0;
   _sequenceTimes = NULL;
   _keyframeValues = NULL;
   _keyframeSequence.Release();
   _property.Release();
}


///////////////////////////////////////////////////////////////////////////////
bool AnimatedPropertyWrapper::init(FeatStd::SizeType keyFrameCount, FeatStd::SizeType componentCount,
                                   const Candera::Animation::InterpolationStrategy::SharedPointer& interpolationStrategy)
{
   bool result = false;

   _keyFrameCount = keyFrameCount;
   _componentCount = componentCount;
   _sequenceTimes = CANDERA_NEW_ARRAY(Candera::Animation::SequenceTimeType, keyFrameCount);
   _keyframeValues = CANDERA_NEW_ARRAY(Candera::Float, componentCount * keyFrameCount);
   _keyframeSequence = Candera::Animation::AnimationKeyframeSequence::Create();
   _property = Candera::Animation::AnimationBlendedProperty::Create();

   if ((_sequenceTimes != NULL) && (_keyframeValues != NULL) && !_keyframeSequence.PointsToNull() && !_property.PointsToNull())
   {
      memset(_sequenceTimes, 0, sizeof(Candera::Animation::SequenceTimeType) * keyFrameCount);
      memset(_keyframeValues, 0, sizeof(Candera::Float) * componentCount * keyFrameCount);

      _keyframeSequence->SetKeyframes(static_cast<FeatStd::Int>(componentCount), static_cast<FeatStd::Int>(keyFrameCount),
                                      _sequenceTimes, Candera::Animation::KeyframeSequence::TimeTypeDisposer::Dispose,
                                      _keyframeValues, Candera::Animation::KeyframeSequence::ValuesDisposer::Dispose);
      _keyframeSequence->SetInterpolationStrategy(interpolationStrategy);

      result = _property->AddKeyframeSequence(_keyframeSequence);
   }

   if (!result)
   {
      //if kf seq is created it will delete the seqTimes and kfValues but if not we need to delete them
      if (_keyframeSequence.PointsToNull())
      {
         if (_sequenceTimes != NULL)
         {
            CANDERA_DELETE_ARRAY(_sequenceTimes);
            _sequenceTimes = NULL;
         }
         if (_keyframeValues != NULL)
         {
            CANDERA_DELETE_ARRAY(_keyframeValues);
            _keyframeValues = NULL;
         }
      }

      release();
   }

   return result;
}


///////////////////////////////////////////////////////////////////////////////
bool AnimatedPropertyWrapper::addToPlayer(Candera::Animation::AnimationPlayer::SharedPointer player)
{
   bool result = false;
   if (!_property.PointsToNull() && !player.PointsToNull() && !player->GetController().PointsToNull())
   {
      Candera::Animation::AnimationController::SharedPointer controller = player->GetController();
      result = controller->AddProperty(_property);
   }
   return result;
}


///////////////////////////////////////////////////////////////////////////////
bool AnimatedPropertyWrapper::removeFromPlayer(Candera::Animation::AnimationPlayer::SharedPointer player)
{
   bool result = false;
   if (!_property.PointsToNull() && !player.PointsToNull() && !player->GetController().PointsToNull())
   {
      Candera::Animation::AnimationController::SharedPointer controller = player->GetController();
      for (int i = static_cast<int>(controller->GetNumberOfProperties()) - 1; i >= 0; --i)
      {
         Candera::Animation::AnimationBlendedProperty::SharedPointer temp = controller->GetProperty(static_cast<FeatStd::SizeType>(i));
         if (!temp.PointsToNull() && temp.GetPointerToSharedInstance() == _property.GetPointerToSharedInstance())
         {
            result = controller->RemoveProperty(static_cast<FeatStd::SizeType>(i)) || result;
         }
      }
   }
   return result;
}


///////////////////////////////////////////////////////////////////////////////
void AnimatedPropertyWrapper::setPropertySetter(const Candera::Animation::AnimationPropertySetter::SharedPointer& propertySetter)
{
   if (!_property.PointsToNull())
   {
      _property->SetAnimationPropertySetter(propertySetter);
   }
}


///////////////////////////////////////////////////////////////////////////////
Candera::Animation::AnimationPropertySetter::SharedPointer AnimatedPropertyWrapper::getPropertySetter() const
{
   return !_property.PointsToNull() ? _property->GetAnimationPropertySetter() : Candera::Animation::AnimationPropertySetter::SharedPointer();
}


///////////////////////////////////////////////////////////////////////////////
void AnimatedPropertyWrapper::setSequenceTime(FeatStd::SizeType kfIndex, Candera::Animation::SequenceTimeType seqTime)
{
   if ((_sequenceTimes != NULL) && (kfIndex < _keyFrameCount))
   {
      _sequenceTimes[kfIndex] = seqTime;
   }
}


///////////////////////////////////////////////////////////////////////////////
Candera::Animation::SequenceTimeType AnimatedPropertyWrapper::getSequenceTime(FeatStd::SizeType kfIndex) const
{
   if ((_sequenceTimes != NULL) && (kfIndex < _keyFrameCount))
   {
      return _sequenceTimes[kfIndex];
   }
   return Candera::Animation::SequenceTimeType();
}


///////////////////////////////////////////////////////////////////////////////
void AnimatedPropertyWrapper::setKeyframeValue(FeatStd::SizeType kfIndex, FeatStd::SizeType componentIndex, FeatStd::Float value)
{
   if ((_keyframeValues != NULL) && (kfIndex < _keyFrameCount) && (componentIndex < _componentCount))
   {
      _keyframeValues[kfIndex * _componentCount + componentIndex] = value;
   }
}


///////////////////////////////////////////////////////////////////////////////
Candera::Float AnimatedPropertyWrapper::getKeyframeValue(FeatStd::SizeType kfIndex, FeatStd::SizeType componentIndex) const
{
   if ((_keyframeValues != NULL) && (kfIndex < _keyFrameCount) && (componentIndex < _componentCount))
   {
      return _keyframeValues[kfIndex * _componentCount + componentIndex];
   }
   return Candera::Float();
}


///////////////////////////////////////////////////////////////////////////////
Candera::Animation::SequenceTimeType AnimatedPropertyWrapper::getDuration() const
{
   return ((_keyFrameCount > 0) && (_sequenceTimes != NULL)) ? _sequenceTimes[_keyFrameCount - 1] : 0;
}


///////////////////////////////////////////////////////////////////////////////
AnimationWrapper::AnimationWrapper(AppViewHandler& viewHandler) : _viewHandler(viewHandler)
{
}


///////////////////////////////////////////////////////////////////////////////
AnimationWrapper::~AnimationWrapper()
{
}


///////////////////////////////////////////////////////////////////////////////
void AnimationWrapper::setPlayer(Candera::Animation::AnimationPlayer::SharedPointer player)
{
   _player = player;
}


///////////////////////////////////////////////////////////////////////////////
bool AnimationWrapper::init()
{
   bool result = false;
   _player = Candera::Animation::AnimationPlayer::Create();
   if (!_player.PointsToNull())
   {
      Candera::Animation::AnimationController::SharedPointer controller = Candera::Animation::AnimationController::Create();
      if (!controller.PointsToNull())
      {
         _player->SetController(controller);
         result = true;
      }
   }

   if (!result)
   {
      _player.Release();
   }

   return result;
}


///////////////////////////////////////////////////////////////////////////////
void AnimationWrapper::release()
{
   _player.Release();
}


///////////////////////////////////////////////////////////////////////////////
void AnimationWrapper::applyProperties(const Courier::AnimationProperties& properties)
{
   if (!_player.PointsToNull())
   {
      //RepeatCount is not copied by the other apply method
      properties.ApplyProperties(static_cast<Candera::Animation::AnimationPlayerBase*>(_player.GetPointerToSharedInstance()));

      properties.ApplyProperties(_player.GetPointerToSharedInstance());
   }
}


///////////////////////////////////////////////////////////////////////////////
bool AnimationWrapper::start(const Courier::AnimationProperties& properties, bool resetBeforeApplyingProperties)
{
   bool result = false;
   if (!_player.PointsToNull() && !_player->GetController().PointsToNull())
   {
      if (resetBeforeApplyingProperties)
      {
         reset();
      }
      applyProperties(properties);

      _viewHandler.RemoveAnimationPlayer(_player);
      if (_viewHandler.AddAnimationPlayer(_player))
      {
         //_viewHandler.DefineWorldTime();
         Courier::Renderer::WakeUpAllRenderComponents();
         result = _player->Start();
      }
   }
   if (!result)
   {
      stop();
   }
   return result;
}


///////////////////////////////////////////////////////////////////////////////
void AnimationWrapper::stop()
{
   if (!_player.PointsToNull())
   {
      if (_player->IsEnabled())
      {
         _player->Stop();
      }
      _viewHandler.RemoveAnimationPlayer(_player);
   }
}


///////////////////////////////////////////////////////////////////////////////
void AnimationWrapper::reset()
{
   stop();
   if (!_player.PointsToNull() && !_player->GetController().PointsToNull())
   {
      Courier::AnimationProperties props;
      props.SetDirection(Candera::Animation::AnimationPlayer::Forward);
      props.SetRepeatMode(Candera::Animation::AnimationPlayer::Replay);
      props.SetRepeatCount(1);
      props.SetSpeedFactor(1.0f);
      props.SetSequenceStartTimeMs(0);
      props.SetSequenceDurationMs(getMaximumDuration());

      props.ApplyProperties(_player.GetPointerToSharedInstance());
   }
}


///////////////////////////////////////////////////////////////////////////////
bool AnimationWrapper::isEnabled() const
{
   return !_player.PointsToNull() && _player->IsEnabled();
}


///////////////////////////////////////////////////////////////////////////////
bool AnimationWrapper::addListener(Candera::Animation::AnimationPlayerListener* listener)
{
   if ((listener != NULL) && !_player.PointsToNull())
   {
      _player->RemoveAnimationPlayerListener(listener);
      return _player->AddAnimationPlayerListener(listener);
   }
   return false;
}


///////////////////////////////////////////////////////////////////////////////
bool AnimationWrapper::removeListener(Candera::Animation::AnimationPlayerListener* listener)
{
   return (listener != NULL) && !_player.PointsToNull() && _player->RemoveAnimationPlayerListener(listener);
}


///////////////////////////////////////////////////////////////////////////////
Candera::Animation::SequenceTimeType AnimationWrapper::getMaximumDuration() const
{
   Candera::Animation::SequenceTimeType duration = 0;
   if (!_player.PointsToNull())
   {
      Candera::Animation::AnimationController::SharedPointer controller = _player->GetController();
      if (!controller.PointsToNull())
      {
         for (FeatStd::SizeType i = 0; i < controller->GetNumberOfProperties(); ++i)
         {
            Candera::Animation::AnimationBlendedProperty::SharedPointer prop = controller->GetProperty(i);
            if (!prop.PointsToNull())
            {
               for (FeatStd::SizeType j = 0; j < prop->GetNumberOfKeyframeSequences(); ++j)
               {
                  Candera::Animation::AnimationKeyframeSequence::SharedPointer keyframeSequence = prop->GetKeyframeSequence(j);
                  if (!keyframeSequence.PointsToNull() && (keyframeSequence->GetKeyframeSequenceTimes() != NULL) && (keyframeSequence->GetKeyframeCount() > 0))
                  {
                     Candera::Animation::SequenceTimeType temp = keyframeSequence->GetKeyframeSequenceTimes()[keyframeSequence->GetKeyframeCount() - 1];
                     if (temp > duration)
                     {
                        duration = temp;
                     }
                  }
               }
            }
         }
      }
   }
   return duration;
}


}
}
