/* ***************************************************************************************
* FILE:          ToString.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ToString.h is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#ifndef _HMIBASE_TO_STRING_H_
#define _HMIBASE_TO_STRING_H_

#include "IOSExtensions.h"
#include "IOSExtensionsTpl.h"
#include <sstream>


namespace hmibase {
namespace trace {
namespace util {

typedef unsigned int HintType;

//to string hint
namespace tsh {
static const HintType _bit_Object = 0;
static const HintType _bit_Pointer = 1;
static const HintType _bit_Path = 2;
static const HintType _bit_Container = 3;

static const HintType Object = 1u << _bit_Object;
static const HintType Pointer = 1u << _bit_Pointer;
static const HintType Path = 1u << _bit_Path;
static const HintType Container = 1u << _bit_Container;

static const HintType Default = Object;
static const HintType PathObject = Path | Object;
static const HintType ContainerObject = Container | Object;
static const HintType ContainerPath = Container | Path;
static const HintType ContainerPathObject = Container | Path | Object;
static const HintType PointerObject = Pointer | Object;
static const HintType PointerPathObject = Pointer | Path | Object;
static const HintType PointerContainerObject = Pointer | Container | Object;
static const HintType PointerContainerPath = Pointer | Container | Path;
static const HintType PointerContainerPathObject = Pointer | Container | Path | Object;
}


/************************************************************************/
template <typename TVal>
class Manipulator
{
   public:
      Manipulator(const TVal* value, HintType hint) : _value(value), _hint(hint)
      {
      }

      Manipulator(const TVal& value, HintType hint) : _value(&value), _hint(hint)
      {
      }

      ~Manipulator()
      {
         _value = NULL;
      }

      void insertInto(::std::ostream& os) const;

   private:
      const TVal* _value;
      HintType _hint;
};


/************************************************************************/
template <typename TVal>
::std::ostream& operator<<(::std::ostream& os, const Manipulator<TVal>& manipulator)
{
   manipulator.insertInto(os);
   return os;
}


//#define _HMIBASE_EXTRACT_STRING(_sstream, _string) _sstream >> _string;
#define _HMIBASE_EXTRACT_STRING(_sstream, _string) _string = _sstream.str();


/************************************************************************/
class ToString
{
   public:
      /************************************************************************/
      template <typename TVal>
      const char* c_str(const TVal& value)
      {
         ::std::stringstream ss;
         ss << '[' << value << ']';
         _HMIBASE_EXTRACT_STRING(ss, _string)
         return _string.c_str();
      }

      /************************************************************************/
      template <typename TVal>
      const char* c_str(const TVal* value)
      {
         ::std::stringstream ss;
         ss << '[' << static_cast<const void*>(value);
         if (value != NULL)
         {
            ss << '.' << *value;
         }
         ss << ']';
         _HMIBASE_EXTRACT_STRING(ss, _string)
         return _string.c_str();
      }

      /************************************************************************/
      template <typename TVal>
      const char* c_str(TVal* value)
      {
         ::std::stringstream ss;
         ss << '[' << static_cast<const void*>(value);
         if (value != NULL)
         {
            ss << '.' << *value;
         }
         ss << ']';
         _HMIBASE_EXTRACT_STRING(ss, _string)
         return _string.c_str();
      }

      /************************************************************************/
      template <typename TVal>
      const char* c_str(const TVal& value, HintType hint)
      {
         ::std::stringstream ss;
         ss << '[' << Manipulator<TVal>(value, hint) << ']';
         _HMIBASE_EXTRACT_STRING(ss, _string)
         return _string.c_str();
      }

      /************************************************************************/
      template <typename TVal>
      const char* c_str(const TVal* value, HintType hint)
      {
         ::std::stringstream ss;
         ss << '[' << Manipulator<TVal>(value, hint) << ']';
         _HMIBASE_EXTRACT_STRING(ss, _string)
         return _string.c_str();
      }

      /************************************************************************/
#define _HMIBASE_TO_STRING_DECL(TVal)\
   const char* c_str(const TVal& value);\
   const char* c_str(const TVal* value);\
   const char* c_str(TVal* value);

#define _HMIBASE_TO_STRING_DECL_HINT(TVal)\
   const char* c_str(const TVal& value, HintType hint);\
   const char* c_str(const TVal* value, HintType hint);

      /* these non template methods generate less code */
      _HMIBASE_TO_STRING_DECL(::Candera::Bitmap)
      _HMIBASE_TO_STRING_DECL_HINT(::Candera::CanderaObject)
      _HMIBASE_TO_STRING_DECL(::Candera::Color)
      _HMIBASE_TO_STRING_DECL_HINT(::Candera::GraphicDeviceUnit)
      _HMIBASE_TO_STRING_DECL(::Candera::Image2D)
      _HMIBASE_TO_STRING_DECL(::Candera::Margin)
      _HMIBASE_TO_STRING_DECL(::Candera::Matrix3)
      _HMIBASE_TO_STRING_DECL(::Candera::Matrix3x2)
      _HMIBASE_TO_STRING_DECL(::Candera::Matrix4)
      _HMIBASE_TO_STRING_DECL_HINT(::Candera::Node)
      _HMIBASE_TO_STRING_DECL_HINT(::Candera::Node2D)
      _HMIBASE_TO_STRING_DECL(::Candera::Vector2)
      _HMIBASE_TO_STRING_DECL(::Candera::Vector3)
      _HMIBASE_TO_STRING_DECL(::Candera::Vector4)
      _HMIBASE_TO_STRING_DECL(::Candera::Rectangle)

      _HMIBASE_TO_STRING_DECL(::Courier::Identifier);
      _HMIBASE_TO_STRING_DECL(::Courier::ItemId);
      _HMIBASE_TO_STRING_DECL(::Courier::Message);
      _HMIBASE_TO_STRING_DECL(::Courier::TouchInfo);
      _HMIBASE_TO_STRING_DECL_HINT(::Courier::View)
      _HMIBASE_TO_STRING_DECL(::Courier::ViewId);

      _HMIBASE_TO_STRING_DECL(::FeatStd::String);

      _HMIBASE_TO_STRING_DECL(::hmibase::util::geometry::Point)
      _HMIBASE_TO_STRING_DECL(::hmibase::util::geometry::Rect)
      _HMIBASE_TO_STRING_DECL(::hmibase::util::geometry::Size)

      _HMIBASE_TO_STRING_DECL(::hmibase::input::gesture::GestureEvent)
      _HMIBASE_TO_STRING_DECL_HINT(::hmibase::widget::Widget)

   private:
      ::std::string _string;
};


}
}


}


#define _HMIBASE_TO_STRING(value) ::hmibase::trace::util::ToString().c_str(value)
#define _HMIBASE_TO_STRING_MANIPULATOR(objType, value, hint) ::hmibase::trace::util::ToString().c_str(static_cast<const objType*>(value), hint)
#define _HMIBASE_TO_STRING_MANIPULATOR_TPL(objType, value, hint) ::hmibase::trace::util::ToString().c_str<objType>(value, hint)


//general
#define HMIBASE_TO_STRING(value) _HMIBASE_TO_STRING(value)
#define HMIBASE_TO_STRING_EXT(value, hint) _HMIBASE_TO_STRING_MANIPULATOR(value, hint)

//candera object
#define HMIBASE_TO_STRING_CO(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::CanderaObject, value, ::hmibase::trace::util::tsh::PointerObject)

//gdu
#define HMIBASE_TO_STRING_GDU(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::GraphicDeviceUnit, value, ::hmibase::trace::util::tsh::PointerObject)

//node
#define HMIBASE_TO_STRING_N2D(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::Node2D, value, ::hmibase::trace::util::tsh::PointerObject)

//path, node
#define HMIBASE_TO_STRING_PN2D(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::Node2D, value, ::hmibase::trace::util::tsh::PointerPathObject)

//scene, path, node
#define HMIBASE_TO_STRING_SPN2D(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::Node2D, value, ::hmibase::trace::util::tsh::PointerContainerPathObject)

//node
#define HMIBASE_TO_STRING_N3D(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::Node, value, ::hmibase::trace::util::tsh::PointerObject)

//path, node
#define HMIBASE_TO_STRING_PN3D(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::Node, value, ::hmibase::trace::util::tsh::PointerPathObject)

//scene, path, node
#define HMIBASE_TO_STRING_SPN3D(value) _HMIBASE_TO_STRING_MANIPULATOR(::Candera::Node, value, ::hmibase::trace::util::tsh::PointerContainerPathObject)

//view
#define HMIBASE_TO_STRING_V(value) _HMIBASE_TO_STRING_MANIPULATOR(::Courier::View, value, ::hmibase::trace::util::tsh::Object)

//widget
#define HMIBASE_TO_STRING_W(value) _HMIBASE_TO_STRING_MANIPULATOR(::hmibase::widget::Widget, value, ::hmibase::trace::util::tsh::PointerObject)

//view, widget
#define HMIBASE_TO_STRING_VW(value) _HMIBASE_TO_STRING_MANIPULATOR(::hmibase::widget::Widget, value, ::hmibase::trace::util::tsh::PointerContainerObject)


#endif //  _HMIBASE_TO_STRING_H_
